import torch
import torch.nn as nn

class RevIN(nn.Module):
    def __init__(self, num_features: int, eps=1e-5, subtract_last=False):
        """
        :param num_features: the number of features or channels
        :param eps: a value added for numerical stability
        :param affine: if True, RevIN has learnable affine parameters
        """
        super(RevIN, self).__init__()
        self.num_features = num_features
        self.eps = eps

        self.subtract_last = subtract_last

    def forward(self, x, mode:str, idx = None):
        if mode == 'norm':
            self._get_statistics(x)
            x = self._normalize(x)
        elif mode == 'denorm':
            x = self._denormalize(x, idx)
        else: raise NotImplementedError
        return x

    def _get_statistics(self, x):
        dim2reduce = tuple(range(1, x.ndim-1))
        if self.subtract_last:
            self.last = x[:,-1,:].unsqueeze(1)
        else:
            self.mean = torch.mean(x, dim=dim2reduce, keepdim=True).detach()
        self.stdev = torch.sqrt(torch.var(x, dim=dim2reduce, keepdim=True, unbiased=False) + self.eps).detach()

    def _normalize(self, x):
        if self.subtract_last:
            x = x - self.last
        else:
            x = x - self.mean
        x = x / self.stdev
        
        return x

    def _denormalize(self, x, idx):
        
        
        if idx is None:
            x = x * self.stdev
            x = x + (self.last if self.subtract_last else self.mean)
        else:
            stdev = self.stdev.squeeze(1)[idx[0], idx[1]].unsqueeze(1)
            sub = self.last.squeeze(1)[idx[0], idx[1]].unsqueeze(1) if self.subtract_last else self.mean.squeeze(1)[idx[0], idx[1]].unsqueeze(1)
            x = x * stdev
            x = x + sub
        return x