use std::collections::{HashMap, HashSet};
use std::error::Error;

use regex_automata::dfa::{dense, StartKind};
use regex_automata::nfa::thompson;
use regex_automata::util::syntax;
use serde_json::json;

use crate::compiler::ir::ast::SpatialFormula;
use crate::compiler::ir::ops::QuantifierOperatorKind;
use crate::compiler::symbolizer::ast::{Symbol, SymbolicAbstractSyntaxTree as AST};
use crate::datastream::frame::Frame;
use crate::matcher::automata::{AutomatonType, State};
use crate::matcher::simulation::Simulation;
use crate::matcher::simulation::{self as sim, Transition};

use super::DeterministicFiniteAutomaton;

/// A reverse matching DFA.
///
/// This DFA is configured for anchored searches. Therefore, it should be ran
/// only to find the start position of a search.
pub struct DeterministicFiniteAutomata<'a> {
    pub automata: AutomatonType,

    pub fmap: HashMap<&'a Symbol, &'a SpatialFormula>,
    pub qmap: HashMap<&'a Symbol, &'a QuantifierOperatorKind>,
}

impl<'a> DeterministicFiniteAutomaton<'a> for DeterministicFiniteAutomata<'a> {
    /// Simulate the DFA.
    ///
    /// This simulates the DFA on a slice of [`Frame`]. The default behavior is
    /// to find the longest leftmost match. It is assumed that all matches are
    /// anchored (i.e., a match always begins at the first frame provided).
    ///
    /// As a result of this behavior, it is recommended to call run incrementally
    /// to collect all possible matches over the complete haystack.
    fn run(&self, haystack: &'a [Frame]) -> Result<Simulation, Box<dyn Error>> {
        let mut simulation = Simulation::new();

        // Find and add the initial state to the [`Simulation`].
        //
        // This first retrieves the initial state ([`State`]) from the DFA and
        // then adds the first [`sim::State`] to the simulation.
        let initial = self.initial(&self.automata)?;
        let idx = simulation.graph.add_node(sim::State::new(initial));

        // Keep a fringe.
        //
        // The fringe is a cache for speeding up the execution time of the DFA.
        // It relieves the need to traverse the [`Simulation::graph`] to collect
        // the set of deepest [`sim::State`], accordingly.
        //
        // In this case, it stores the index that is mapped in the
        // [`Simulation::graph`] data structure.
        let mut fringe = HashSet::new();
        fringe.insert(idx);

        let mut frames = haystack.iter().rev();
        let mut frame = frames.next();

        while let Some(f) = frame {
            let mut isquantified = false;
            let mut nexts = HashSet::new();

            // Compute the next [`State`] from the current [`Frame`].
            //
            // This will return a set of [`State`] (targets) to transition to
            // from the current state (source).
            for i in fringe.iter() {
                // Build graph.
                //
                // This sets the children of [`state`] above to all the resulting
                // [`sim::State`] created from quantified transitions,
                // accordingly.
                for next in self.qtransition(&self.automata, &self.qmap, &simulation.graph[*i], f) {
                    isquantified = true;

                    // Add node to [`Simulation::graph`].
                    //
                    // This returns us the index for us to then create an edge
                    // between the parent node and this one as well as insert
                    // into the temporary fringe (i.e., [`nexts`]).
                    let idx = simulation.graph.add_node(next);

                    simulation
                        .graph
                        .add_edge(*i, idx, Transition::Quantified(json!(1)));
                    nexts.insert(idx);
                }

                // Build graph.
                //
                // This sets the children of [`state`] above to all the resulting
                // [`sim::State`] created from formula-based transitions,
                // accordingly.
                for (next, statement) in
                    self.ftransition(&self.automata, &self.fmap, &simulation.graph[*i], f)
                {
                    isquantified = false;
                    // Add node to [`Simulation::graph`].
                    //
                    // This returns us the index for us to then create an edge
                    // between the parent node and this one as well as insert
                    // into the temporary fringe (i.e., [`nexts`]).
                    let idx = simulation.graph.add_node(next);

                    simulation
                        .graph
                        .add_edge(*i, idx, Transition::Normal(statement));
                    nexts.insert(idx);
                }

                // Build graph.
                //
                // This sets the children of [`state`] above to all the resulting
                // [`sim::State`] created from formula-based transitions,
                // accordingly.
                if nexts.is_empty() {
                    if let Some(next) = self.etransition(&self.automata, &simulation.graph[*i]) {
                        if !matches!(next.state, State::Dead(..)) {
                            let idx = simulation.graph.add_node(next);

                            simulation.graph.add_edge(*i, idx, Transition::EndOfInput);
                            nexts.insert(idx);
                        }
                    }
                }
            }

            // At this point, we have collected all of the next states from all
            // possible paths. Therefore, reset [`fringe`].
            //
            // This allows the next iteration of frames to transition only on the
            // deepest set of states, accordingly.
            fringe = nexts;

            // Return [`Simulation`] if all paths have died.
            //
            // This is checked by the condition that all [`sim::State`] of the
            // [`fringe`] are of the [`State::Dead(..)`] kind.
            if fringe
                .iter()
                .all(|i| matches!(simulation.graph[*i].state, State::Dead(..)))
            {
                // There does not exist a non-dead state.
                //
                // In this case, all paths of the [`Simulation::graph`] are of
                // kind [`State::Dead(..)`]. Therefore, return early.
                //
                // This avoids wasted computation time.
                return Ok(simulation);
            }

            if !isquantified {
                frame = frames.next();
            }
        }

        // Perform EOI (End-of-Input) transitions.
        //
        // This is a required step for the `regex-automata` crate. This ensures
        // that all possible matches are accounted for.
        //
        // For more information, see:
        // https://docs.rs/regex-automata/latest/regex_automata/#build-a-full-dfa-and-walk-it-manually
        for i in fringe.iter() {
            if let Some(next) = self.etransition(&self.automata, &simulation.graph[*i]) {
                let idx = simulation.graph.add_node(next);
                simulation.graph.add_edge(*i, idx, Transition::EndOfInput);
            }
        }

        Ok(simulation)
    }
}

impl<'a> DeterministicFiniteAutomata<'a> {
    /// Create a new reverse-matching DFA.
    ///
    /// This function is exposed if a different configuration is requierd.
    /// Otherwise, for all other cases, use the [`self::build`] interface to
    /// construct this DFA.
    pub fn new(
        automata: AutomatonType,
        fmap: HashMap<&'a Symbol, &'a SpatialFormula>,
        qmap: HashMap<&'a Symbol, &'a QuantifierOperatorKind>,
    ) -> Self {
        DeterministicFiniteAutomata {
            automata,
            fmap,
            qmap,
        }
    }
}

/// Build a reverse searching DFA.
///
/// The `regex-automata` library is used primarily here to construct the
/// underlying state machine that performs matching. We then wrap this result
/// into a [`DeterministicFiniteAutomata`] for simple interfacing.
pub fn build(ast: &AST) -> Result<DeterministicFiniteAutomata, Box<dyn Error>> {
    let automata = dense::Builder::new()
        .configure(
            dense::Config::new()
                .minimize(true)
                .accelerate(false)
                .start_kind(StartKind::Anchored)
                .specialize_start_states(true),
        )
        .syntax(syntax::Config::new().unicode(false).utf8(true))
        .thompson(thompson::Config::new().reverse(true).utf8(true))
        .build(&super::super::super::regexify(ast))?;

    Ok(DeterministicFiniteAutomata::new(
        automata,
        ast.fmap(),
        ast.qmap(),
    ))
}
