import logging
import psutil
import tracemalloc

from dataclasses import dataclass
from typing import Callable
from typing import List

from bencher.matcher.matcher import Match
from bencher.bencher import Bench
from bencher.bencher import BenchMeasurement


logger = logging.getLogger(__name__)


class MemoryUtilization(Bench):
    """A class for measuring the memory utilization.
    """

    def __init__(self):
        """Initialize the `MemoryUtilization`.
        """

        self.process = psutil.Process()

    def benchmark(self, callback: Callable[[], List[Match]]) -> BenchMeasurement:
        """Benchmark the memory utilization.
        """
        tracemalloc.start()

        process = psutil.Process()
        
        start, speak = tracemalloc.get_traced_memory()
        data = callback()
        end, epeak = tracemalloc.get_traced_memory()

        tracemalloc.stop()

        m = BenchMeasurement(
            measurement=MemoryUtilizationResult(
                utilization=(end - start)
            ),
        )

        # Add logging.
        #
        # This logging outputs the resulting benchmark measurement computed after
        # running it.
        logger.debug(f"callback={callback.__module__} measurement={m.measurement}")

        return m


@dataclass
class MemoryUtilizationResult:
    """Represents a memory utilization result.
    """

    utilization: int
