import logging
import pandas as pd
import time

from enum import StrEnum
from openai import OpenAI as OpenAi
from pathlib import Path
from typing import List

from bencher.matcher.llm.matcher import Matcher as BaseMatcher
from bencher.matcher.matcher import Match
from bencher.matcher.matcher import DataStream
from bencher.matcher.llm.matcher import StructuredResponse
from perceptor.models.detector.base import Detection


logger = logging.getLogger(__name__)


class ModelKind(StrEnum):
    """Represents the GPT models kinds.
    """

    GptFourOmni = "gpt-4o"
    GptFourOmniMini = "gpt-4o-mini"
    GptFourTurbo = "gpt-4-turbo"
    GptThreeFiveTurbo = "gpt-3.5-turbo"


class Matcher(BaseMatcher):
    """A matcher based on the GPT tool.
    """

    def __init__(
            self,
            query: str,
            datastream: DataStream,
            key: str,
            kind: ModelKind = ModelKind.GptFourOmni,
            temperature: float = 0.2
    ) -> None:
        """Initialize the Generative Pre-Trained Transformer tool.
        """

        self.query = query
        self.datastream = super().format(datastream)
        self.kind = kind
        self.temperature = temperature

        path = (
            Path(__file__).parent.parent
            .joinpath("data")
            .joinpath("prompts")
            .joinpath("roleA.v6.txt")
        )

        with open(path, "r") as infile:
            self.role = infile.read()

        # Load the model.
        #
        # This establishes a connection with the OpenAI servers to prepare for
        # sending/receiving queries.
        self.model = OpenAi(api_key=key)

    def run(self) -> List[Match]:
        """Find the matches.
        """

        response = self.model.beta.chat.completions.parse(
            model=self.kind,
            messages=[
                {
                    "role": "system",
                    "content": [
                        {"type": "text", "text": self.role}
                    ]
                },
                {
                    "role": "user",
                    "content": [
                        {"type": "text", "text": f"<query>{self.query}</query>"},
                        {"type": "text", "text": f"<data>{self.datastream}</data>"},
                    ]
                }
            ],
            temperature=self.temperature,
            response_format=StructuredResponse,
        )

        matches = []
        for m in response.choices[0].message.parsed.matches:
            matches.append(Match(
                start=m.start,
                end=m.end,
            ))

        return matches
