//! This module provides the necessary monitors to evaluate spatial formulas.
//!
//! Currently, the implemented monitors include evaluation of S4/S4u topological
//! formulas interpreted over frames.

use serde_json::Value;

use crate::compiler::ir::ast::SpatialFormula;
use crate::datastream::frame::sample::Sample;
use crate::datastream::frame::Frame;
use crate::resolver::Environment;

pub mod s4;
pub mod s4m;
pub mod s4u;

/// The main monitor.
///
/// This is a entrypoint for monitoring spatial formulas found within SpREs. This
/// interface is also responsible for managing evaluating these formulas against
/// different sample types.
///
/// For example, point clouds, object detections, etc.
#[derive(Default)]
pub struct Monitor<'a> {
    pub environment: Option<Environment<'a>>,
}

impl<'a> Monitor<'a> {
    /// Create a new [`Monitor`].
    pub fn new() -> Self {
        Self { environment: None }
    }

    /// The main interface to evaluating a frame sample against a spatial formula.
    ///
    /// This considers all possible sample types.
    pub fn evaluate(&self, frame: &Frame, formula: &SpatialFormula) -> Option<Value> {
        for sample in frame.samples.iter() {
            match sample {
                Sample::ObjectDetection(record) => {
                    let monitor = s4u::Monitor::new(self.environment.clone());

                    // Evaluate formula.
                    //
                    // If the result is [`true`], return early as we are only
                    // looking for a satisfaction, not all satisfactions,
                    // accordingly.
                    if let Some(value) = monitor.evaluate(&record.annotations, formula) {
                        return Some(value);
                    }
                }
            };
        }

        None
    }

    /// Set the [`Environment`] of the [`Monitor`].
    pub fn environment(mut self, environment: Environment<'a>) -> Self {
        self.environment = Some(environment);
        self
    }
}
