use std::collections::HashMap;

use serde_json::{json, Value};

use crate::compiler::ir::ast::{OperandKind, SpatialFormula};
use crate::compiler::ir::ops::{Operator, S4OperatorKind, SpatialOperatorKind};
use crate::compiler::ir::Node;
use crate::datastream::frame::sample::detections::Annotation;
use crate::resolver::Environment;

/// A monitor for evaluating S4 formulas.
#[derive(Default)]
pub struct Monitor<'a> {
    environment: Option<Environment<'a>>,
}

impl<'a> Monitor<'a> {
    pub fn new(environment: Option<Environment<'a>>) -> Self {
        Self { environment }
    }

    /// Evaluate formula satisfaction against set of annotations.
    ///
    /// This returns is a boolean result. If true, the formula is satisifed;
    /// else, if false, then it is not satisfied.
    pub fn evaluate(
        &self,
        detections: &'a HashMap<String, Vec<Annotation>>,
        formula: &SpatialFormula,
    ) -> Option<Vec<(Vec<&'a Annotation>, Value)>> {
        match formula {
            Node::Operand(op) => match op {
                OperandKind::Symbol(label) => {
                    // Retrieve an annotation with the same class category as
                    // specified by the label.
                    if let Some(annotations) = detections.get(label) {
                        if let Some(annotation) = annotations.first() {
                            let statement = json!({
                                "statement": format!("the bounding box of a {}", annotation.label.clone()),
                            });

                            return Some(vec![(annotations.iter().collect(), statement)]);
                        }
                    }

                    None
                }
                OperandKind::Variable(name) => {
                    if let Some(environment) = &self.environment {
                        if let Some((binding, _)) = environment.scope().get(name.as_str()) {
                            if let Some(annotations) = detections.get(&binding.label) {
                                for annotation in annotations.iter() {
                                    // This compares the [`Option`].
                                    //
                                    // For the result to be [`true`], the
                                    // following cases should be considered:
                                    //
                                    // 1. Some(a) == Some(b)
                                    // 2. None == None
                                    //
                                    // For the second case, this will evaluate to
                                    // [`true`] if the label (i.e., class) is the
                                    // same but they do not have identifiers.
                                    if binding.identifier == annotation.identifier {
                                        let statement = json!({
                                            "statement": format!("the bounding box of a {}", annotation.label.clone())
                                        });

                                        return Some(vec![(vec![annotation], statement)]);
                                    }
                                }
                            }
                        }
                    }

                    None
                }
                _ => panic!("monitor: s4: operand: unsupported `{:?}`", op),
            },
            Node::UnaryExpr { op, .. } => match op {
                Operator::SpatialOperator(SpatialOperatorKind::S4Operator(
                    S4OperatorKind::Complement,
                )) => {
                    todo!()
                }
                _ => panic!("monitor: s4: unrecognized unary operator"),
            },
            Node::BinaryExpr { op, lhs, rhs } => {
                let lhs = self.evaluate(detections, lhs);
                let rhs = self.evaluate(detections, rhs);

                match op {
                    Operator::SpatialOperator(op) => match op {
                        SpatialOperatorKind::S4Operator(op) => match op {
                            S4OperatorKind::Intersection => {
                                let mut result = Vec::new();

                                if let Some(lhs) = lhs {
                                    if let Some(rhs) = &rhs {
                                        for lannotations in &lhs {
                                            for rannotations in rhs {
                                                for l in lannotations.0.iter() {
                                                    for r in rannotations.0.iter() {
                                                        if l != r
                                                            && l.bbox.intersects(&r.bbox).is_some()
                                                        {
                                                            result.push((vec![*l, *r], json!({
                                                                "statement": "intersection",
                                                                "children": [
                                                                    {"statement": l.label.clone()},
                                                                    {"statement": r.label.clone()},
                                                                ]
                                                            })))
                                                        }
                                                    }
                                                }
                                            }
                                        }

                                        // Return [`Some`] if intersections exist.
                                        //
                                        // If there were no successful
                                        // intersections, then we return [`None`],
                                        // accordingly
                                        if !result.is_empty() {
                                            return Some(result);
                                        }
                                    }
                                }

                                None
                            }
                            S4OperatorKind::Union => {
                                let mut result = Vec::new();

                                if let Some(lhs) = lhs {
                                    for l in lhs {
                                        result.push(l);
                                    }
                                }

                                if let Some(rhs) = rhs {
                                    for r in rhs {
                                        result.push(r);
                                    }
                                }

                                if !result.is_empty() {
                                    return Some(result);
                                }

                                None
                            }
                            _ => panic!("monitor: s4: unknown binary operator"),
                        },
                        _ => panic!("monitor: unknown binary operator {:#?}", op),
                    },
                    _ => panic!("monitor: unknown binary operator {:#?}", op),
                }
            }
        }
    }
}
