use std::collections::HashMap;

use serde_json::{json, Value};

use crate::{
    compiler::ir::{
        ast::{OperandKind, SpatialFormula},
        ops::{Operator, S4mOperatorKind, SpatialOperatorKind},
        Node,
    },
    datastream::frame::sample::detections::{bbox::BoundingBox, Annotation},
    resolver::Environment,
};

use super::s4;

/// A monitor for evaluating S4m expressions.
///
/// This monitor evaluates against a series of object detections obtained from the
/// perception stream.
#[derive(Default)]
pub struct Monitor<'a> {
    environment: Option<Environment<'a>>,
}

impl<'a> Monitor<'a> {
    /// Create a new [`Monitor`]
    pub fn new(environment: Option<Environment<'a>>) -> Self {
        Monitor { environment }
    }

    /// Evaluate the formula against the set of annotations.
    ///
    /// This returns a set of possible real numbers obtained from evaluating the
    /// expression, accordingly.
    pub fn evaluate(
        &self,
        detections: &HashMap<String, Vec<Annotation>>,
        formula: &SpatialFormula,
    ) -> Option<Vec<(Vec<f64>, Value)>> {
        match formula {
            Node::Operand(op) => match op {
                OperandKind::Number(num) => Some(vec![(
                    vec![*num],
                    json!({"statement": format!("{}", *num)}),
                )]),
                _ => panic!("monitor: s4m: operand: unsupported `{:?}`", op),
            },
            Node::UnaryExpr { op, child } => match op {
                Operator::SpatialOperator(op) => match op {
                    SpatialOperatorKind::S4mOperator(op) => match op {
                        S4mOperatorKind::Inverse => panic!("not implemented"),
                        S4mOperatorKind::Function(name) => match &name[..] {
                            // Retrieve the x-coordinate value.
                            //
                            // The direction that the x-axis represents is
                            // entirely dependent on the format/representation
                            // selected by the user.
                            "x" => {
                                let items = s4::Monitor::new(self.environment.clone())
                                    .evaluate(detections, child);

                                if let Some(items) = items {
                                    let mut result = Vec::new();

                                    for (annotations, statements) in items.iter() {
                                        // Retrieve the center.
                                        //
                                        // From this, get the x-position and create
                                        // a new entry for the return value.
                                        for annotation in annotations.iter() {
                                            let center = match &annotation.bbox {
                                                BoundingBox::AxisAligned(region) => region.center(),
                                                BoundingBox::Oriented(region) => region.center(),
                                            };

                                            result.push((
                                                vec![center.x],
                                                json!({
                                                    "statement": "x position of",
                                                    "children": [statements]
                                                }),
                                            ));
                                        }
                                    }

                                    return Some(result);
                                }

                                None
                            }

                            // Retrieve the y-coordinate value.
                            //
                            // The direction that the y-axis represents is
                            // entirely dependent on the format/representation
                            // selected by the user.
                            "y" => {
                                let items = s4::Monitor::new(self.environment.clone())
                                    .evaluate(detections, child);

                                if let Some(items) = items {
                                    let mut result = Vec::new();

                                    for (annotations, statements) in items.iter() {
                                        // Retrieve the center.
                                        //
                                        // From this, get the x-position and create
                                        // a new entry for the return value.
                                        for annotation in annotations.iter() {
                                            let center = match &annotation.bbox {
                                                BoundingBox::AxisAligned(region) => region.center(),
                                                BoundingBox::Oriented(region) => region.center(),
                                            };

                                            result.push((
                                                vec![center.y],
                                                json!({
                                                    "statement": "y position of",
                                                    "children": [statements]
                                                }),
                                            ));
                                        }
                                    }

                                    return Some(result);
                                }

                                None
                            }

                            // Compute the distance from an annotation to origin.
                            //
                            // This is equivalent to computing the Euclidean
                            // distance between a bounding box and the origin
                            // point of the space.
                            "dist" => {
                                let items = s4::Monitor::new(self.environment.clone())
                                    .evaluate(detections, child);

                                if let Some(items) = items {
                                    let mut result = Vec::new();

                                    for (annotations, statements) in items.iter() {
                                        // Retrieve the center.
                                        //
                                        // From this, get the x-position and create
                                        // a new entry for the return value.
                                        for annotation in annotations.iter() {
                                            let center = match &annotation.bbox {
                                                BoundingBox::AxisAligned(region) => region.center(),
                                                BoundingBox::Oriented(region) => region.center(),
                                            };

                                            result.push((
                                                vec![f64::sqrt(
                                                    (center.x).powi(2) + (center.y).powi(2),
                                                )],
                                                json!({
                                                    "statement": "distance of",
                                                    "children": [statements]
                                                }),
                                            ));
                                        }
                                    }

                                    return Some(result);
                                }

                                None
                            }

                            // Compute the area of the annotation.
                            //
                            // This works only on 2D-based bounding boxes such as
                            // Axis-Aligned or Oriented.
                            "area" => {
                                let items = s4::Monitor::new(self.environment.clone())
                                    .evaluate(detections, child);

                                if let Some(items) = items {
                                    let mut result = Vec::new();

                                    for (annotations, statements) in items.iter() {
                                        // Retrieve the center.
                                        //
                                        // From this, get the x-position and create
                                        // a new entry for the return value.
                                        for annotation in annotations.iter() {
                                            let area = match &annotation.bbox {
                                                BoundingBox::AxisAligned(region) => {
                                                    region.width() * region.height()
                                                }
                                                BoundingBox::Oriented(region) => {
                                                    region.width() * region.height()
                                                }
                                            };

                                            result.push((
                                                vec![area],
                                                json!({
                                                    "statement": "area of",
                                                    "children": [statements]
                                                }),
                                            ));
                                        }
                                    }

                                    return Some(result);
                                }

                                None
                            }
                            _ => panic!(
                                "monitor: s4m: unary: operator: function not supported: `{}`",
                                name
                            ),
                        },
                        _ => panic!("monitor: s4m: unary: operator: unsupported `{:?}`", op),
                    },
                    _ => panic!("monitor: s4m: unary: operator: unsupported `{:?}`", op),
                },
                _ => panic!("monitor: s4m: unary: operator: unsupported `{:?}`", op),
            },
            Node::BinaryExpr { op, lhs, rhs } => match op {
                Operator::SpatialOperator(op) => match op {
                    SpatialOperatorKind::S4mOperator(op) => match op {
                        S4mOperatorKind::Addition => panic!("not implemented"),
                        S4mOperatorKind::Subtraction => panic!("not implemented"),
                        S4mOperatorKind::Multiplication => panic!("not implemented"),
                        S4mOperatorKind::Division => panic!("not implemented"),
                        S4mOperatorKind::Function(name) => match &name[..] {
                            // Compute the distance from an annotation to another
                            // annotation.
                            //
                            // This is equivalent to computing the Euclidean
                            // distance between a bounding box and another
                            // bounding box in space.
                            "dist" => {
                                let lhs = s4::Monitor::new(self.environment.clone())
                                    .evaluate(detections, lhs);
                                let rhs = s4::Monitor::new(self.environment.clone())
                                    .evaluate(detections, rhs);

                                if let Some(lhs) = &lhs {
                                    if let Some(rhs) = &rhs {
                                        let mut result = Vec::new();

                                        for (lannotations, lstatements) in lhs {
                                            for (rannotations, rstatements) in rhs {
                                                for l in lannotations {
                                                    for r in rannotations {
                                                        if let Some(distance) =
                                                            self::euclidean(&l.bbox, &r.bbox)
                                                        {
                                                            result.push((vec![distance], json!({
                                                                "statement": "distance between",
                                                                "children": [lstatements, rstatements]
                                                            })))
                                                        }
                                                    }
                                                }
                                            }
                                        }

                                        return Some(result);
                                    }
                                }

                                None
                            }
                            _ => panic!(
                                "monitor: s4m: binary: operator: function not supported: `{}`",
                                name
                            ),
                        },
                        _ => panic!("monitor: s4m: binary: operator: unsupported `{:?}`", op),
                    },
                    _ => panic!("monitor: s4m: binary: operator: unsupported `{:?}`", op),
                },
                _ => panic!("monitor: s4m: binary: operator: unsupported `{:?}`", op),
            },
        }
    }
}

/// Compute the Euclidean distance between [`BoundingBox`].
///
/// This performs a distance computation based on the center point of the
/// relevant bounding boxes, accordingly.
fn euclidean(a: &BoundingBox, b: &BoundingBox) -> Option<f64> {
    if let BoundingBox::AxisAligned(a) = a {
        if let BoundingBox::AxisAligned(b) = b {
            let a = a.center();
            let b = b.center();

            return Some(f64::sqrt((b.x - a.x).powi(2) + (b.y - a.y).powi(2)));
        }
    }

    if let BoundingBox::AxisAligned(a) = a {
        if let BoundingBox::Oriented(b) = b {
            let a = a.center();
            let b = b.center();

            return Some(f64::sqrt((b.x - a.x).powi(2) + (b.y - a.y).powi(2)));
        }
    }

    if let BoundingBox::Oriented(a) = a {
        if let BoundingBox::AxisAligned(b) = b {
            let a = a.center();
            let b = b.center();

            return Some(f64::sqrt((b.x - a.x).powi(2) + (b.y - a.y).powi(2)));
        }
    }

    if let BoundingBox::Oriented(a) = a {
        if let BoundingBox::Oriented(b) = b {
            let a = a.center();
            let b = b.center();

            return Some(f64::sqrt((b.x - a.x).powi(2) + (b.y - a.y).powi(2)));
        }
    }

    None
}
