use std::collections::HashMap;

use serde_json::{json, Value};

use crate::compiler::ir::ast::{OperandKind, SpatialFormula};
use crate::compiler::ir::ops::{FolOperatorKind, Operator, S4uOperatorKind, SpatialOperatorKind};
use crate::compiler::ir::Node;
use crate::datastream::frame::sample::detections::Annotation;
use crate::resolver::Environment;

use super::{s4, s4m};

/// A monitor for evaluating S4u formulas.
///
/// This monitor evaluates against a series of object detection obtained from the
/// perception stream.
#[derive(Default)]
pub struct Monitor<'a> {
    environment: Option<Environment<'a>>,
}

impl<'a> Monitor<'a> {
    pub fn new(environment: Option<Environment<'a>>) -> Self {
        Self { environment }
    }

    /// Evaluate formula satisfaction against set of annotations.
    ///
    /// This returns is a boolean result. If true, the formula is satisifed;
    /// else, if false, then it is not satisfied.
    pub fn evaluate(
        &self,
        detections: &HashMap<String, Vec<Annotation>>,
        formula: &SpatialFormula,
    ) -> Option<Value> {
        match formula {
            Node::Operand(op) => match op {
                OperandKind::Symbol(label) => {
                    if let Some(annotations) = detections.get(label) {
                        if let Some(annotation) = annotations.first() {
                            // Return annotation.
                            //
                            // We select the first annotation from the list of
                            // annotations that matched the [`label`] and return
                            // a JSON-valued entity.
                            return Some(json!({
                                "statement": annotation.label.clone(),
                            }));
                        }
                    }

                    None
                }
                OperandKind::Variable(name) => {
                    if let Some(environment) = &self.environment {
                        if let Some((binding, _)) = environment.scope().get(name.as_str()) {
                            if let Some(annotations) = detections.get(&binding.label) {
                                for annotation in annotations.iter() {
                                    // This compares the [`Option`].
                                    //
                                    // For the result to be [`true`], the
                                    // following cases should be considered:
                                    //
                                    // 1. Some(a) == Some(b)
                                    // 2. None == None
                                    //
                                    // For the second case, this will evaluate to
                                    // [`true`] if the label (i.e., class) is the
                                    // same but they do not have identifiers.
                                    if binding.identifier == annotation.identifier {
                                        return Some(json!({
                                            "statement": annotation.label.clone(),
                                        }));
                                    }
                                }
                            }
                        }
                    }

                    None
                }
                _ => panic!("monitor: s4u: operand: unsupported `{:?}`", op),
            },
            Node::UnaryExpr { op, child } => match op {
                Operator::SpatialOperator(op) => match op {
                    SpatialOperatorKind::S4uOperator(op) => match op {
                        S4uOperatorKind::NonEmpty => {
                            let monitor = s4::Monitor::new(self.environment.clone());

                            // Evaluate.
                            //
                            // This evaluates the sub-formula which is assumed
                            // to be a valid S4+ formula.
                            if let Some(annotations) = monitor.evaluate(detections, child) {
                                if let Some((_, statements)) = annotations.first() {
                                    return Some(json!({
                                        "statement": "non-empty",
                                        "children": [statements]
                                    }));
                                }
                            }

                            None
                        }
                    },
                    SpatialOperatorKind::FolOperator(op) => match op {
                        FolOperatorKind::Negation => panic!("monitor: s4u: negation not supported"),
                        _ => panic!("monitor: s4u: unrecognized unary FOL operator"),
                    },
                    _ => panic!("monitor: s4u: unrecognized unary operator"),
                },
                _ => panic!("monitor: s4u: unrecognized unary operator"),
            },
            Node::BinaryExpr { op, lhs, rhs } => match op {
                Operator::SpatialOperator(kind) => match kind {
                    SpatialOperatorKind::FolOperator(kind) => match kind {
                        FolOperatorKind::Conjunction => {
                            if let Some(lhs) = self.evaluate(detections, lhs) {
                                if let Some(rhs) = self.evaluate(detections, rhs) {
                                    // True.
                                    //
                                    // We can now join the two results together
                                    // since both have [`Some`].
                                    return Some(json!({
                                        "statement": "and",
                                        "children": [lhs, rhs]
                                    }));
                                }
                            }

                            None
                        }
                        FolOperatorKind::Disjunction => {
                            if let Some(lhs) = self.evaluate(detections, lhs) {
                                // True.
                                //
                                // We can return this since it evaluates to true
                                // and we only care about one expression
                                // valuating to true.
                                return Some(lhs);
                            }

                            if let Some(rhs) = self.evaluate(detections, rhs) {
                                // True.
                                //
                                // We can return this since it evaluates to true
                                // and we only care about one expression
                                // valuating to true.
                                return Some(rhs);
                            }

                            None
                        }
                        FolOperatorKind::LessThan => {
                            let lhs = s4m::Monitor::new(self.environment.clone())
                                .evaluate(detections, lhs);
                            let rhs = s4m::Monitor::new(self.environment.clone())
                                .evaluate(detections, rhs);

                            // Compute the comparison of all possible options.
                            //
                            // In order to accurately get a result, all
                            // combinations of comparisons are needed.
                            if let Some(lhs) = &lhs {
                                if let Some(rhs) = &rhs {
                                    for (lvalues, lstatements) in lhs {
                                        for (rvalues, rstatements) in rhs {
                                            for l in lvalues {
                                                for r in rvalues {
                                                    if l < r {
                                                        return Some(json!({
                                                            "statement": "less than",
                                                            "children": [lstatements, rstatements]
                                                        }));
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            }

                            None
                        }
                        FolOperatorKind::GreaterThan => {
                            let lhs = s4m::Monitor::new(self.environment.clone())
                                .evaluate(detections, lhs);
                            let rhs = s4m::Monitor::new(self.environment.clone())
                                .evaluate(detections, rhs);

                            // Compute the comparison of all possible options.
                            //
                            // In order to accurately get a result, all
                            // combinations of comparisons are needed.
                            if let Some(lhs) = &lhs {
                                if let Some(rhs) = &rhs {
                                    for (lvalues, lstatements) in lhs {
                                        for (rvalues, rstatements) in rhs {
                                            for l in lvalues {
                                                for r in rvalues {
                                                    if l > r {
                                                        return Some(json!({
                                                            "statement": "greater than",
                                                            "children": [lstatements, rstatements]
                                                        }));
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            }

                            None
                        }
                        FolOperatorKind::LessThanEqualTo => None,
                        FolOperatorKind::GreaterThanEqualTo => None,
                        _ => panic!("monitor: unkown FOL operator {:#?}", kind),
                    },
                    _ => panic!("monitor: unknown binary operator {:#?}", kind),
                },
                _ => panic!("monitor: unknown binary operator {:#?}", op),
            },
        }
    }
}
