import itertools
import logging

from dataclasses import dataclass
from typing import Callable
from typing import List
from typing import Sequence

from bencher.bencher import Bench
from bencher.bencher import BenchMeasurement
from bencher.matcher.matcher import Match
from bencher.metrics.accuracy import fscore
from bencher.metrics.accuracy import bascore

logger = logging.getLogger(__name__)

class Accuracy(Bench):
    """A class for measuring the accuracy.
    """

    def __init__(self, expected: Sequence[bool]):
        """Initialize the `Accuracy`.
        """

        self.expected = expected

    def benchmark(self, callback: Callable[[], List[Match]]) -> BenchMeasurement:
        """Benchmark the accuracy.
        """

        actual = list(itertools.repeat(0, len(self.expected)))
        data = callback()

        for m in data:
            for i in range(m.start, m.end):
                actual[i] = True

        m = BenchMeasurement(
            measurement=AccuracyResult(
                fscore=fscore(self.expected, actual, beta=1.0),
                bascore=bascore(self.expected, actual)
            ),
            data=data,
        )

        # Add logging.
        #
        # This logging outputs the resulting benchmark measurement computed after
        # running it.
        logger.debug(f"callback={callback.__module__} measurement={m.measurement}")

        return m


@dataclass
class AccuracyResult:
    """The expected
    """

    fscore: float
    bascore: float
