import json
import logging
import re
import subprocess

from typing import List
from fests.matcher.matcher import Matcher as BaseMatcher
from fests.matcher.matcher import Match
from fests.matcher.matcher import DataStream


logger = logging.getLogger(__name__)


class Matcher(BaseMatcher):
    """A matcher based on the STREM tool.
    """

    def __init__(self, query: str, datastream: DataStream) -> None:
        """Initialize the `Matcher`.
        """

        self.query = query
        # logger.debug(f"query=\"{self.query}\"; frames={len(datastream.frames)}")

        # self.data = {
        #     "version": "0.1.1",
        #     "frames": []
        # }

        # for i, frame in enumerate(datastream.frames):
        #     self.data["frames"].append({
        #         "index": i,
        #         "samples": [{
        #             "type": "@stremf/sample/detection",
        #             "channel": "cam::front",
        #             "image": {
        #                 "path": "",
        #                 "dimensions": {
        #                     "width": 1920,
        #                     "height": 1080
        #                 }
        #             },
        #             "annotations": []
        #         }]
        #     })

        #     for annotation in frame.annotations:
        #         self.data["frames"][i]["samples"][0]["annotations"].append({
        #             "class": annotation.label,
        #             "score": annotation.score,
        #             "bbox": {
        #                 "type": "@stremf/bbox/aabb",
        #                 "region": {
        #                     "center": {
        #                         "x": (annotation.bbox.xmax - annotation.bbox.xmin) / 2.0,
        #                         "y": (annotation.bbox.ymax - annotation.bbox.ymin) / 2.0,
        #                     },
        #                     "dimensions": {
        #                         "w": annotation.bbox.xmax - annotation.bbox.xmin,
        #                         "h": annotation.bbox.ymax - annotation.bbox.ymin,
        #                     }
        #                 }
        #             }
        #         })
        self.data = datastream


    def run(self) -> List[Match]:
        """Find the matches.
        """

        # NOTE: It is important to mention here that the `subprocess.run`
        # function is optimized and thus creates faster results when running a
        # command when the `input` keyword argument is used opposed to
        # conventional shell-like operations of piping, i/o, etc.
        #
        # For now, we leave this as when we are evaluating our tool, we are also
        # not concerned with the shell overhead, so this is more accurate than
        # including the shell overhead in our execution time result.
        #
        # However, further investigation/discussion is needed to determine if
        # this choice is truly the best/most representative user experience of
        # the tool, itself; as most users will not be spawning the tool through
        # a Python subprocess, accordingly.
        #
        # For example, the process below, when ran in the shell (i.e., not
        # through Python `subprocess.run`), results in 0.17s running time. When
        # ran through the Python `subprocess.run`, the process results in
        # 0.00242s running time which is a 98.63% improvement. In short, this
        # large improvement should not be ignored.
        response = subprocess.run(
            [
                "strem",
                "--export",
                self.query
            ],
            input=json.dumps(self.data),
            capture_output=True,
            text=True,
        )

        # Print the stderr.
        #
        # Here, we add a special case that is commonly printed out by the
        # `hyperfine` command when we enable the `--ignore-failure` so that the
        # log does not get crowded.
        #
        # This is allowed as a return code of `1` in the case of `strem`
        # indicates that no matches were found.
        if response.stderr and response.stderr.strip() != "Warning: Ignoring non-zero exit code.":
            logger.error(f"stderr=\"{response.stderr.strip()}\"")

        # Process the match results.
        #
        # This uses the results of the `strem` command to retrieve the matches
        # of this particular run, accordingly.
        matches = []
        for line in response.stdout.splitlines():
            matches.append(json.loads(line))
        
        # Log the token usage.
        #
        # This will notify to us how many matches were found for quick debugging
        # purposes, accordingly.
        logger.debug(f"matches={len(matches)}")

        return matches
