from typing import Sequence


def fscore(a: Sequence[bool], b: Sequence[bool], beta: float = 1.0) -> float:
    """Compute the F-score of `b` against `a`.

    For more information, see:
    - https://en.wikipedia.org/wiki/F-score#Diagnostic_testing
    - https://scikit-learn.org/stable/modules/generated/sklearn.metrics.f1_score.html#f1-score
    """

    assert len(a) == len(b)

    p = precision(a, b)
    r = recall(a, b)

    # Compute the F-score.
    #
    # This calculation computes the F-score with a particular weight provided
    # by the `beta` parameter. To weigh recall higher than precision, use
    # beta=2.0; to weigh recall lower than precision, use beta=0.5; to weigh
    # recall equally to precision, use beta=1.0.
    #
    # In the case that precision and recall sum to zero, we return zero,
    # accordingly (i.e., if precision is zero and recall is zero, then the
    # accuracy of the result should be zero, not one).
    if p + r == 0:
        return 0

    return (1 + (beta ** 2)) * ((p * r) / ((beta ** 2 * p) + r))


def precision(a: Sequence[bool], b: Sequence[bool]) -> float:
    """Compute the precision of `b` against `a`.

    For more information, see:
    - https://scikit-learn.org/stable/modules/generated/sklearn.metrics.precision_score.html#precision-score
    """

    assert len(a) == len(b)

    tp = sum((p == True and q == True) for p, q in zip(a, b))
    fp = sum((p == False and q == True) for p, q in zip(a, b))

    # Compute the precision.
    #
    # This calculation computes the precision of `b` with respect to `a` to
    # determine the effective accuracy in the positive results.
    #
    # In the case that the true-positives and true-negatives sum to zero, we
    # return zero (i.e., we are not precise if no positives exist).
    if tp + fp == 0:
        return 0

    return tp / (tp + fp)


def recall(a: Sequence[bool], b: Sequence[bool]) -> float:
    """Compute the recall of `b` against `a`.

    For more information, see:
    - https://scikit-learn.org/stable/modules/generated/sklearn.metrics.recall_score.html#recall-score
    """

    assert len(a) == len(b)

    tp = sum((p == True and q == True) for p, q in zip(a, b))
    fn = sum((p == True and q == False) for p, q in zip(a, b))

    # Compute the recall.
    #
    # This calculation computes the recall of `b` with respect to `a` to
    # determine the effective positive matches (i.e., from the expected
    # positives, how many actual cases were found effectively).
    #
    # In the case that the true-positives and false-negatives sum to zero, we
    # return zero (i.e., no cases exist).
    if tp + fn == 0:
        return 1

    return tp / (tp + fn)


def bscore(a: Sequence[bool], b: Sequence[bool]) -> float:
    """Compute the balanced accuracy of `b` against `a`.

    For more information, see:
    - https://scikit-learn.org/stable/modules/generated/sklearn.metrics.balanced_accuracy_score.html#balanced-accuracy-score
    - https://neptune.ai/blog/balanced-accuracy
    """

    assert len(a) == len(b)

    tpr = recall(a, b)
    tnr = specificity(a, b)

    # Compute the balanced accuracy.
    #
    # This calculation computes the accuracy over imbalanced data (e.g., in
    # the case of more negatives then positives).
    return (tpr + tnr) / 2.0


def specificity(a: Sequence[bool], b: Sequence[bool]) -> float:
    """Compute the specificity of `b` against `a`.

    For more information, see:
    """

    assert len(a) == len(b)

    tn = sum((p == False and q == False) for p, q in zip(a, b))
    fp = sum((p == False and q == True) for p, q in zip(a, b))

    # Compute the specificity.
    #
    # In the case that the true-negatives and false-positives sum to zero, we
    # return zero (i.e., no cases exist).
    if tn + fp == 0:
        return 1

    return tn / (tn + fp)
