//! Intermediate module for symbolic structures.
//!
//! This module is necessary to effectively match against a Spatial Regular
//! Expression (SpRE).
//!
//! Generally, it provides the mechanisms and interfaces to map a each unique
//! spatial-based formula to be evaluate to a unique symbol.

use std::error::Error;
use std::fmt;

use ast::{SymbolicNode, SymbolicSpatialRegularExpression};

use crate::compiler::ir::ast::{AbstractSyntaxTree, SpatialRegularExpression};
use crate::compiler::ir::Node;

use self::ast::SymbolicAbstractSyntaxTree;

use super::ir::ops::{FolOperatorKind, Operator, QuantifierOperatorKind, SpatialOperatorKind};

pub mod ast;

#[derive(Default)]
pub struct Symbolizer<'a> {
    current: usize,
    alphabet: &'a [char],
}

impl<'a> Symbolizer<'a> {
    /// Create a new [`Symbolizer`] with provided alphabet.
    pub fn new(alphabet: &'a [char]) -> Self {
        let current = 0;
        Self { current, alphabet }
    }

    /// Construct a [`SymbolicAbstractSyntaxTree`] from an [`AbstractSyntaxTree`].
    ///
    /// This step is used for the matcher that requires symbols to execute its
    /// underlying matching mechanisms.
    pub fn symbolize(
        &mut self,
        ast: AbstractSyntaxTree,
    ) -> Result<SymbolicAbstractSyntaxTree, Box<dyn Error>> {
        if let Some(root) = ast.root {
            return Ok(SymbolicAbstractSyntaxTree::new(Some(
                self.symbolizeit(root)?,
            )));
        }

        Ok(SymbolicAbstractSyntaxTree::new(None))
    }

    /// Recursively build the Symbolic Abstract Syntax Tree.
    ///
    /// The main procedure done here is to take each root node of the spatial
    /// formulas and wrap the root node with a uniquely mapped symbol.
    fn symbolizeit(
        &mut self,
        node: SpatialRegularExpression,
    ) -> Result<SymbolicSpatialRegularExpression, Box<dyn Error>> {
        match node {
            Node::Operand(formula) => Ok(SymbolicNode::from(formula).symbol(self.advance()?)),
            Node::UnaryExpr { op, child } => match op {
                Operator::SpatialOperator(SpatialOperatorKind::FolOperator(
                    FolOperatorKind::Quantifier(QuantifierOperatorKind::Exists(..)),
                )) => {
                    let symbol = self.advance()?;

                    // Recurse after advancing the [`symbol`].
                    //
                    // We want to keep the order of the advancement based on the
                    // [`AbstractSyntaxTree`]. Therefore, we first advance the
                    // symbol, and then we call [`Symbolizer::symbolizeit`] on
                    // [`child`] after.
                    Ok(SymbolicNode::unary(op, self.symbolizeit(*child)?).symbol(symbol))
                }

                Operator::SpatialOperator(SpatialOperatorKind::FolOperator(
                    FolOperatorKind::Quantifier(QuantifierOperatorKind::Forall(..)),
                )) => {
                    let symbol = self.advance()?;

                    // Recurse after advancing the [`symbol`].
                    //
                    // We want to keep the order of the advancement based on the
                    // [`AbstractSyntaxTree`]. Therefore, we first advance the
                    // symbol, and then we call [`Symbolizer::symbolizeit`] on
                    // [`child`] after.
                    Ok(SymbolicNode::unary(op, self.symbolizeit(*child)?).symbol(symbol))
                }

                _ => Ok(SymbolicNode::unary(op, self.symbolizeit(*child)?)),
            },
            Node::BinaryExpr { op, lhs, rhs } => {
                let lhs = self.symbolizeit(*lhs)?;
                let rhs = self.symbolizeit(*rhs)?;

                Ok(SymbolicNode::binary(op, lhs, rhs))
            }
        }
    }

    /// Retrieve the next unique symbol in the alphabet.
    ///
    /// This procedure will raise an error if an insufficient number of symbols
    /// are present for the number of spatial formulas written.
    fn advance(&mut self) -> Result<char, Box<dyn Error>> {
        if let Some(symbol) = self.alphabet.get(self.current) {
            self.current += 1;
            return Ok(*symbol);
        }

        Err(Box::new(SymbolizerError::from(format!(
            "insufficient symbols ({}) for formulas ({})",
            self.alphabet.len(),
            self.current
        ))))
    }
}

#[derive(Debug, Clone)]
struct SymbolizerError {
    msg: String,
}

impl From<&str> for SymbolizerError {
    fn from(msg: &str) -> Self {
        SymbolizerError {
            msg: msg.to_string(),
        }
    }
}

impl From<String> for SymbolizerError {
    fn from(msg: String) -> Self {
        SymbolizerError { msg }
    }
}

impl fmt::Display for SymbolizerError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "symbolizer: {}", self.msg)
    }
}

impl Error for SymbolizerError {}
