//! Symbolic Abstract Syntax Tree (SAST) representation.
//!
//! This Intermediate Representation (IR) of the SpRE is a necessary step to
//! perform matching with the underlying library.

use std::collections::HashMap;

use crate::compiler::ir::ast::SpatialFormula;
use crate::compiler::ir::ops::{
    FolOperatorKind, Operator, QuantifierOperatorKind, SpatialOperatorKind,
};

pub type SymbolicSpatialRegularExpression = SymbolicNode<SpatialFormula>;
pub type Symbol = char;

/// Symbolic representation of an AST.
///
/// This AST is used as an Intermediate Representation (IR) of a tree-based graph
/// structure where each node may hold a [`Symbol`].
#[derive(Debug)]
pub enum SymbolicNode<T> {
    Operand {
        symbol: Option<Symbol>,
        data: T,
    },
    UnaryExpr {
        symbol: Option<Symbol>,

        op: Operator,
        child: Box<Self>,
    },
    BinaryExpr {
        symbol: Option<Symbol>,

        op: Operator,
        lhs: Box<Self>,
        rhs: Box<Self>,
    },
}

impl<T> From<T> for SymbolicNode<T> {
    fn from(data: T) -> Self {
        SymbolicNode::Operand { symbol: None, data }
    }
}

impl<T> SymbolicNode<T> {
    pub fn unary<C>(op: Operator, child: C) -> Self
    where
        C: Into<SymbolicNode<T>>,
    {
        SymbolicNode::UnaryExpr {
            symbol: None,
            op,
            child: Box::new(child.into()),
        }
    }

    pub fn binary<L, R>(op: Operator, lhs: L, rhs: R) -> Self
    where
        L: Into<SymbolicNode<T>>,
        R: Into<SymbolicNode<T>>,
    {
        SymbolicNode::BinaryExpr {
            symbol: None,
            op,
            lhs: Box::new(lhs.into()),
            rhs: Box::new(rhs.into()),
        }
    }

    pub fn symbol(mut self, s: Symbol) -> Self {
        match self {
            SymbolicNode::Operand { ref mut symbol, .. } => *symbol = Some(s),
            SymbolicNode::UnaryExpr { ref mut symbol, .. } => *symbol = Some(s),
            SymbolicNode::BinaryExpr { ref mut symbol, .. } => *symbol = Some(s),
        }

        self
    }
}

/// The symbolically-represented AST.
///
/// Within this AST, each internal node is a RE-based operation (e.g.,
/// alternation, concatenation, etc); and each operand is a [`SymbolicFormula`].
#[derive(Debug)]
pub struct SymbolicAbstractSyntaxTree {
    pub root: Option<SymbolicSpatialRegularExpression>,
}

impl SymbolicAbstractSyntaxTree {
    pub fn new(root: Option<SymbolicSpatialRegularExpression>) -> Self {
        Self { root }
    }

    /// From the [`SymbolicAbstractSyntaxTree`], return a map of
    /// [`SpatialFormula`] mapped from a [`Symbol`].
    pub fn fmap(&self) -> HashMap<&Symbol, &SpatialFormula> {
        if let Some(root) = &self.root {
            return SymbolicAbstractSyntaxTree::fmapit(root);
        }

        HashMap::new()
    }

    /// The recursive helper function to generate the list of spatial formulas.
    ///
    /// The result is a list of [`SymbolicFormula`] found on the symbolic-AST.
    /// Since the AST operands are [`SpatialFormula`], for each operand, the
    /// tree is returned.
    fn fmapit(node: &SymbolicSpatialRegularExpression) -> HashMap<&Symbol, &SpatialFormula> {
        match node {
            SymbolicNode::Operand { symbol, data } => {
                if let Some(symbol) = symbol {
                    return HashMap::from([(symbol, data)]);
                }

                HashMap::new()
            }
            SymbolicNode::UnaryExpr { child, .. } => SymbolicAbstractSyntaxTree::fmapit(child),
            SymbolicNode::BinaryExpr { lhs, rhs, .. } => {
                let mut fmap = SymbolicAbstractSyntaxTree::fmapit(lhs);
                fmap.extend(SymbolicAbstractSyntaxTree::fmapit(rhs));

                fmap
            }
        }
    }

    /// From the [`SymbolicAbstractSyntaxTree`], return the set of
    /// [`QuantifierOperatorKind`] mapped from a [`Symbol`].
    pub fn qmap(&self) -> HashMap<&Symbol, &QuantifierOperatorKind> {
        if let Some(root) = &self.root {
            return SymbolicAbstractSyntaxTree::qmapit(root);
        }

        HashMap::new()
    }

    /// The recursive helper function to generate the list of [`Bindings`].
    ///
    /// The result is a [`HashMap`] of [`QuantifierOperatorKind`] found on the
    /// symbolic-AST, accordingly.
    fn qmapit(
        node: &SymbolicSpatialRegularExpression,
    ) -> HashMap<&Symbol, &QuantifierOperatorKind> {
        match node {
            SymbolicNode::Operand { .. } => HashMap::new(),
            SymbolicNode::UnaryExpr { symbol, op, child } => match op {
                Operator::SpatialOperator(SpatialOperatorKind::FolOperator(
                    FolOperatorKind::Quantifier(op),
                )) => {
                    if let Some(symbol) = symbol {
                        let mut qmap = HashMap::from([(symbol, op)]);
                        qmap.extend(SymbolicAbstractSyntaxTree::qmapit(child));

                        return qmap;
                    }

                    HashMap::new()
                }
                _ => SymbolicAbstractSyntaxTree::qmapit(child),
            },
            SymbolicNode::BinaryExpr { lhs, rhs, .. } => {
                let mut qmap = SymbolicAbstractSyntaxTree::qmapit(lhs);
                qmap.extend(SymbolicAbstractSyntaxTree::qmapit(rhs));

                qmap
            }
        }
    }
}
