use std::collections::HashMap;
use std::error::Error;

use regex_automata::dfa::{Automaton, StartError};
use regex_automata::util::start::Config;
use regex_automata::Anchored;
use serde_json::{json, Value};

use crate::compiler::ir::ast::SpatialFormula;
use crate::compiler::ir::ops::QuantifierOperatorKind;
use crate::compiler::symbolizer::ast::Symbol;
use crate::datastream::frame::Frame;
use crate::matcher::simulation as sim;
use crate::matcher::simulation::Simulation;
use crate::monitor::Monitor;
use crate::resolver::Resolver;

use super::{AutomatonType, State};

pub mod forward;
pub mod reverse;

/// A trait for all DFA's.
///
/// This trait provides a common interface that all matchers may use.
pub trait DeterministicFiniteAutomaton<'a> {
    /// Run the DFA.
    ///
    /// The main interface for which all DFA's must implement is to simulate the
    /// corresponding DFA and return a set of valid [`HalfMatch`].
    fn run(&self, haystack: &'a [Frame]) -> Result<Simulation, Box<dyn Error>>;

    /// Take the quantification transition.
    ///
    /// For this implementation, whether to take a transition is determined by
    /// whether there is a transition on a symbol from
    /// [`DeterministicFiniteAutomata::bmap`] that does not end in the
    /// [`State::Dead`].
    fn qtransition(
        &self,
        automata: &AutomatonType,
        qmap: &HashMap<&'a Symbol, &'a QuantifierOperatorKind>,
        state: &sim::State<'a>,
        frame: &'a Frame,
    ) -> Vec<sim::State<'a>> {
        // If [`State::Dead`], return.
        //
        // This prevents unnecessary computations as once we are in the
        // [`State::Dead`], no outgoing transitions exist.
        if matches!(state.state, State::Dead(..)) {
            return Vec::new();
        }

        // If EOI-based, return.
        //
        // This prevents unnecessary computations as if we are to transition to
        // an EOI state, then
        let eoi = self.etransition(automata, state).unwrap();
        let mut nexts = Vec::new();

        for (symbol, quantification) in qmap.iter() {
            let sid = automata.next_state(*state.state.id(), **symbol as u8);
            let next = sim::State::new(State::new(sid, automata));

            if !matches!(next.state, State::Dead(..)) && next.state != eoi.state {
                // Set up the set of [`Environment`].
                //
                // We are able to take a quantification transition at this point;
                // therefore, to perform a quantification transition, three steps
                // are required: (1) propagate, (2) generate, (3) branch.
                let mut resolver = Resolver::new();

                // Propagate [`Environment`].
                //
                // Here, we propagate the [`Environment`] to the [`Resolver`] as
                // it uses it to set up the next round of [`Environment`](s).
                if let Some(environment) = &state.environment {
                    resolver = resolver.environment(environment.clone());
                }

                // Generate & Branch.
                //
                // Here, we generate the new set of [`Environment`] from the
                // [`bindings`] and create a copy of the [`next`] state with each
                // unique [`Environment`].
                for environment in resolver.resolve(frame, quantification) {
                    nexts.push(next.clone().environment(environment));
                }
            }
        }

        nexts
    }

    /// Take the next transition on the `Frame`.
    ///
    /// For this implementation, whether to take a transition is determined by
    /// whether the [`Monitor`] evaluates to true on the [`Frame`]. The cases are
    /// as follows:
    ///
    /// I. If true, transition on the corresponding symbol from the [`State`].
    /// II. If false, transition on a blank symbol from the [`State`].
    ///
    /// For (II), this is similar to transitioning on a byte that is not in teh
    /// pattern of a traditional RE.
    fn ftransition(
        &self,
        automata: &AutomatonType,
        fmap: &HashMap<&'a Symbol, &'a SpatialFormula>,
        state: &sim::State<'a>,
        frame: &Frame,
    ) -> Vec<(sim::State<'a>, Value)> {
        // If [`State::Dead`], return.
        //
        // This prevents unnecessary computations as once we are in the
        // [`State::Dead`], no outgoing transitions exist.
        if matches!(state.state, State::Dead(..)) {
            return Vec::new();
        }

        // If EOI-based, return.
        //
        // This prevents unnecessary computations as if we are to transition to
        // an EOI state, then
        let eoi = self.etransition(automata, state).unwrap();
        let mut nexts = Vec::new();

        for (symbol, formula) in fmap.iter() {
            let sid = automata.next_state(*state.state.id(), **symbol as u8);
            let mut next = sim::State::new(State::new(sid, automata));

            // In this case, before committing to any monitoring, check
            // the potential to transition.
            //
            // Here, we assume that the transition occurs, but we only add the
            // next state iff the evaluation of the associated sub-formula is
            // satisfied by the [`frame`], accordingly.
            //
            // NOTE: This may also improve performance as checking for a
            // transition is generally cheaper than evaluating a formula.
            if !matches!(next.state, State::Dead(..)) && next.state != eoi.state {
                let mut monitor = Monitor::new();

                // Propagate [`Environment`].
                //
                // If the previous state had an [`Environment`], be sure to copy
                // it to the next state(s).
                //
                // p.s., Set the [`Monitor`] environment as well if the
                // environment exists, accordingly.
                if let Some(environment) = &state.environment {
                    monitor = monitor.environment(environment.clone());
                    next = next.environment(environment.clone());
                }

                if let Some(tree) = monitor.evaluate(frame, formula) {
                    // Format the object.
                    //
                    // We should be the object within another object adding any
                    // auxiliary information that may be useful for
                    // post-processing purposes, accordingly.
                    let statement = json!({
                        "index": Value::Number(frame.index.into()),
                        "reasoning": tree
                    });

                    nexts.push((next, statement));
                }
            }
        }

        nexts
    }

    /// Take the extra byte transition.
    ///
    /// This function must be called as the last transition before checking if a
    /// the DFA is in a matching state.
    ///
    /// For more information, see
    /// [here](https://docs.rs/regex-automata/0.4.3/regex_automata/dfa/trait.Automaton.html#tymethod.next_eoi_state).
    fn etransition(
        &self,
        automata: &AutomatonType,
        state: &sim::State<'a>,
    ) -> Option<sim::State<'a>> {
        // If [`State::Dead`], return.
        //
        // This prevents unnecessary computations as once we are in the
        // [`State::Dead`], no outgoing transitions exist.
        if matches!(state.state, State::Dead(..)) {
            return None;
        }

        // If the [`nexts`] is empty, then this [`frame`] satisifes no
        // formulas, accordingly.
        //
        // As a result, if there are no outgoing transitions, the next state
        // should be a [`State::Dead`] (i.e., end of this path). However,
        // before simply adding the [`State::Dead`] to the set of [`nexts`],
        // a transition on the EOI should be performed to account for the
        // one-byte offset, accordingly.
        //
        // For more information, see:
        // https://docs.rs/regex-automata/latest/regex_automata/dfa/trait.Automaton.html#tymethod.next_eoi_state
        let mut next = sim::State::new(State::new(
            automata.next_eoi_state(*state.state.id()),
            automata,
        ));

        if let Some(environment) = &state.environment {
            next = next.environment(environment.clone());
        }

        Some(next)
    }

    /// Retrieve the initial [`State`] to start from an Automata.
    ///
    /// For further information, see `regex_automata::util::start`.
    fn initial(&self, automata: &AutomatonType) -> Result<State, StartError> {
        // Retrieve the start state.
        //
        // The start state is anchored as all inputs to this
        // [`DeterministicFiniteAutomata`] begin searching at index 0. Therefore,
        // matches are only found starting from the beginning (i.e., anchored).
        let sid = automata.start_state(&Config::new().anchored(Anchored::Yes))?;

        // The start state shall never be the match state.
        //
        // This is true as all matches in the DFA configurations are delayed by
        // a single transition (i.e., byte).
        //
        // For more information, see
        // [here](https://github.com/rust-lang/regex/blob/027eebd6fde307076603530c999afcfd271bb037/regex-automata/src/dfa/search.rs#L552).
        debug_assert!(!automata.is_match_state(sid));

        Ok(State::new(sid, automata))
    }
}

/// The default size to offset all matches by.
///
/// This is set as the end part of a match is exclusive (i.e., open), so the
/// actual end index should be offset, accordingly.
pub const OFFSET: usize = 1;
