use std::fmt;

use petgraph::{graph::NodeIndex, visit::Walker, Direction, Graph};
use regex_automata::{HalfMatch, PatternID};
use serde_json::{json, Value};

use crate::resolver::Environment;

use super::automata::{self, dfa::OFFSET};

/// A representation of a DFA simulation.
///
/// This captures the core execution path from which all states are derived from
/// alongside auxiliary data to perform post-processing of the result from
/// executing the [`crate::matcher::automata::dfa::DeterministicFiniteAutomaton::run`]
/// function, effectively.
#[derive(Debug, Default)]
pub struct Simulation<'a> {
    pub graph: Graph<State<'a>, Transition>,
}

impl<'a> Simulation<'a> {
    /// Create a new [`Simulation`].
    pub fn new() -> Self {
        Simulation {
            graph: Graph::new(),
        }
    }

    /// Check whether the [`Simulation`] contains a semantically valid match.
    pub fn ismatch(&self) -> Option<HalfMatch> {
        if let Some(root) = self.graph.node_indices().next() {
            // The root exists, proceed.
            //
            // With the root, we want to return the boolean valuation of the
            // resulting Depth First Search (DFS) visit.
            if let Some(offset) = self.ismatchit(root) {
                return Some(HalfMatch::new(PatternID::new(0).unwrap(), offset - OFFSET));
            }
        }

        None
    }

    fn ismatchit(&self, node: NodeIndex) -> Option<usize> {
        let mut res = None;
        if matches!(self.graph[node].state, automata::State::Accepting(..)) {
            res = Some(self.depth(node));
        };

        let neighbors: Vec<NodeIndex> = self
            .graph
            .neighbors(node)
            .filter(|a| {
                self.graph
                    .edges_connecting(node, *a)
                    .any(|b| matches!(b.weight(), Transition::Quantified(..)))
            })
            .collect();

        if let Some(neighbor) = neighbors.first() {
            res = if let Some(environment) = &self.graph[*neighbor].environment {
                match environment {
                    Environment::Exists(..) => match res {
                        Some(a) => self.ismatchit(*neighbor).map(|b| std::cmp::max(a, b)),
                        None => self.ismatchit(*neighbor),
                    },
                    Environment::Forall(..) => match res {
                        Some(a) => self.ismatchit(*neighbor).map(|b| std::cmp::min(a, b)),
                        None => self.ismatchit(*neighbor),
                    },
                }
            } else {
                None
            };
        }

        for neighbor in neighbors {
            if let Some(environment) = &self.graph[neighbor].environment {
                match environment {
                    Environment::Exists(..) => {
                        res = match res {
                            Some(a) => self.ismatchit(neighbor).map(|b| std::cmp::max(a, b)),
                            None => self.ismatchit(neighbor),
                        }
                    }
                    Environment::Forall(..) => {
                        res = match res {
                            Some(a) => self.ismatchit(neighbor).map(|b| std::cmp::min(a, b)),
                            None => None,
                        }
                    }
                }
            }
        }

        let neighbors: Vec<NodeIndex> = self
            .graph
            .neighbors(node)
            .filter(|a| {
                self.graph
                    .edges_connecting(node, *a)
                    .any(|b| !matches!(b.weight(), Transition::Quantified(..)))
            })
            .collect();

        for neighbor in neighbors {
            res = match res {
                Some(a) => self.ismatchit(neighbor).map(|b| std::cmp::max(a, b)),
                None => self.ismatchit(neighbor),
            }
        }

        res
    }

    /// Retrieve the set of [`HalfMatch`].
    ///
    /// This is a post-processing function that returns all possible match
    /// offsets from the [`Simulation`]. This is done by first collecting all the
    /// states that are [`super::automata::State::Accepting`] followed by
    /// traversing up the tree to the root node while tracking the number of
    /// edges crossed (i.e., the node's depth/offset), accordingly.
    pub fn matches(&self) -> Option<Vec<HalfMatch>> {
        // If the [`Simulation::graph`] has no nodes, return [`None`].
        //
        // We do not want to waste any effort on computations here as this is
        // performed many times (in most cases). Therefore, if the
        // [`Simulation::graph`] has no nodes, then return, accordingly.
        if let Some(accepting) = self.accepting() {
            let mut matches = Vec::new();

            // For each accepting, compute its depth.
            //
            // This depth computation is equivalent to the offset as each edge in
            // the graph represents a symbol consumed from the haystack.
            for accept in accepting.iter() {
                let offset = self.depth(*accept);

                if offset >= OFFSET {
                    matches.push(HalfMatch::new(PatternID::new(0).unwrap(), offset - OFFSET));
                }
            }

            return Some(matches);
        }

        None
    }

    // Retrieve indices of all the [`State:Accepting`] in the [`Graph`].
    //
    // This returns the indices as we cannot be certain, in advance, that the
    // data to be worked with is the node weight of the graph.
    pub fn accepting(&self) -> Option<Vec<NodeIndex>> {
        // If the [`Simulation::graph`] has no nodes, return [`None`].
        //
        // We do not want to waste any effort on computations here as this is
        // performed many times (in most cases). Therefore, if the
        // [`Simulation::graph`] has no nodes, then return, accordingly.
        if let Some(root) = self.graph.node_indices().next() {
            let accepts = petgraph::visit::Dfs::new(&self.graph, root)
                .iter(&self.graph)
                .filter(|i| matches!(self.graph[*i].state, automata::State::Accepting(..)))
                .collect();

            return Some(accepts);
        }

        None
    }

    pub fn explanation(&self, idx: NodeIndex) -> Value {
        let mut current = idx;
        let mut statements = Vec::new();

        while let Some(parent) = self
            .graph
            .neighbors_directed(current, Direction::Incoming)
            .next()
        {
            // Check if the edge is [`Transition::Normal`].
            //
            // In this case, we append the information with a concatenation as
            // subsequent edges are concatenations within the execution of
            // the automata, accordingly.
            if let Some(edge) = self.graph.edges_connecting(parent, current).next() {
                match edge.weight() {
                    Transition::Normal(statement) => statements.push(statement.clone()),
                    Transition::Quantified(statement) => statements.push(statement.clone()),
                    Transition::EndOfInput => {}
                }
            }

            current = parent;
        }

        // Reverse the statements.
        //
        // This is necessary as we traverse bottom-up adding the first element to
        // the list of statements last.
        statements.reverse();

        json!({
            "version": "0.1.0",
            "statements": statements,
        })
    }

    pub fn depth(&self, idx: NodeIndex) -> usize {
        let mut current = idx;
        let mut offset = 0;

        // Compute depth.
        //
        // This iterates until the current node has no incoming edges
        // (i.e., it is the root node).
        while let Some(parent) = self
            .graph
            .neighbors_directed(current, Direction::Incoming)
            .next()
        {
            // Check if edge is [`Transition::Normal`].
            //
            // We only count edges where the transition from the source
            // to target is a [`Transition::Normal`]. This is because if
            // it is not, transitions in the DFA occur that do not
            // progress the haystack, so this should not be counted when
            // computing the offset, accordingly.
            if let Some(edge) = self.graph.edges_connecting(parent, current).next() {
                if !matches!(edge.weight(), Transition::Quantified(..)) {
                    offset += 1;
                }
            }

            current = parent;
        }

        offset
    }
}

/// A snapshot of the [`Simulation`].
///
/// This includes any relevant data at that point in execution of the DFA that
/// is necessary for post-processing (e.g., [`automata::State`]).
///
/// TODO: In the future, data related to the valuation of a frame against a
/// formula may be important to keep here for debugging purposes.
#[derive(Clone, Debug)]
pub struct State<'a> {
    pub state: automata::State,
    pub environment: Option<Environment<'a>>,
}

impl<'a> State<'a> {
    /// Create a new [`State`].
    pub fn new(state: automata::State) -> Self {
        State {
            state,
            environment: None,
        }
    }

    /// Set the environment of the [`State`].
    pub fn environment(mut self, environment: Environment<'a>) -> Self {
        self.environment = Some(environment);
        self
    }
}

impl<'a> fmt::Display for State<'a> {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        let mut display = String::new();

        if let Some(environment) = &self.environment {
            match environment {
                Environment::Exists(..) => display += "E",
                Environment::Forall(..) => display += "A",
            }

            for (key, (value, _)) in environment.scope() {
                if let Some(identifier) = &value.identifier {
                    display += &format!("({}: {}) ", key, &identifier[..8])[..];
                } else {
                    display += &format!("({}, {{}}) ", key);
                }
            }
        }

        write!(f, "{:?}, {{{}}}", self.state, display)
    }
}

#[derive(Clone, Debug)]
pub enum Transition {
    EndOfInput,
    Normal(Value),
    Quantified(Value),
}

impl fmt::Display for Transition {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "{:?}", self)
    }
}
