use std::collections::HashMap;

use itertools::Itertools;
use serde_json::Value;

use crate::compiler::ir::ast::SpatialTerm;
use crate::compiler::ir::ops::QuantifierOperatorKind;
use crate::datastream::frame::sample::detections::Annotation;
use crate::datastream::frame::sample::Sample;
use crate::datastream::frame::Frame;
use crate::monitor::s4;

/// A scope.
///
/// A scope is generally a collection of mappings between a variable and an
/// [`Annotation`] to be used for name resolution, accordingly.
pub type Scope<'a> = HashMap<&'a str, (&'a Annotation, Value)>;

/// An environment.
///
/// This is also referred to generally as a lookup table, symbol table, bindings,
/// scope, etc.
#[derive(Clone, Debug)]
pub enum Environment<'a> {
    Exists(Scope<'a>),
    Forall(Scope<'a>),
}

impl<'a> Environment<'a> {
    #[inline]
    pub fn scope(&self) -> &HashMap<&'a str, (&'a Annotation, Value)> {
        match self {
            Environment::Exists(scope) => scope,
            Environment::Forall(scope) => scope,
        }
    }
}

/// A binding set.
///
/// A set of bindings which are sourced from the SpRE and map a variable to a
/// spatial term, accordingly.
pub type Bindings = HashMap<String, SpatialTerm>;

/// An interface to resolve quantification bindings.
///
/// The resolver is used to produce a series of environments and a direct result
/// of supporting quantification.
#[derive(Default)]
pub struct Resolver<'a> {
    pub environment: Option<Environment<'a>>,
}

impl<'a> Resolver<'a> {
    /// Create a new [`Resolver`].
    pub fn new() -> Self {
        Resolver { environment: None }
    }

    /// Set the [`Environment`] of the [`Resolver`].
    pub fn environment(mut self, environment: Environment<'a>) -> Self {
        self.environment = Some(environment);
        self
    }

    /// Resolve environments from a [`Frame`].
    ///
    /// This accepts a spatial term and returns a list of environments from
    /// evaluating said spatial term.
    pub fn resolve(
        &self,
        frame: &'a Frame,
        quantifier: &'a QuantifierOperatorKind,
    ) -> Vec<Environment<'a>> {
        let mut environments = Vec::new();

        for sample in frame.samples.iter() {
            match sample {
                Sample::ObjectDetection(record) => {
                    let monitor = s4::Monitor::new(self.environment.clone());
                    let mut binds = Vec::new();

                    let bindings = match quantifier {
                        QuantifierOperatorKind::Exists(bindings) => bindings,
                        QuantifierOperatorKind::Forall(bindings) => bindings,
                    };

                    // Resolve formula.
                    //
                    // This will (presumably) return a list of annotations that
                    // will then be mapped to a variable, accordingly.
                    for (variable, formula) in bindings.iter() {
                        let mut resolutions = Vec::new();

                        if let Some(items) = monitor.evaluate(&record.annotations, formula) {
                            for (annotations, statements) in items {
                                for annotation in annotations {
                                    resolutions.push((variable, annotation, statements.clone()));
                                }
                            }

                            binds.push(resolutions);
                        }
                    }

                    // For each binding, create an environment.
                    //
                    // In this case, we must create all possible combinations of
                    // variables and annotations.
                    for bind in binds.iter().multi_cartesian_product() {
                        // Create an [`Environment`].
                        //
                        // This table maps a variable to an annotation, so it may
                        // be used later during monitoring.
                        let mut scope = Scope::new();

                        // Extend the lookup table.
                        //
                        // The lookup table needs to check for parent lookup
                        // tables declared beforehand and include them,
                        // accordingly.
                        //
                        // p.s., To resolve name clashes, we use the the most
                        // recent name (i.e., the youngest lookup table).
                        if let Some(environment) = &self.environment {
                            for (variable, annotation) in environment.scope().iter() {
                                scope.insert(*variable, annotation.clone());
                            }
                        }

                        // Insert the most recent binds.
                        //
                        // This ensures the most recent definitions are used by
                        // placing them after inserting the prior definitions.
                        for (variable, annotation, statements) in bind.iter() {
                            scope.insert(variable, (annotation, statements.clone()));
                        }

                        match quantifier {
                            QuantifierOperatorKind::Exists(..) => {
                                environments.push(Environment::Exists(scope))
                            }
                            QuantifierOperatorKind::Forall(..) => {
                                environments.push(Environment::Forall(scope))
                            }
                        };
                    }
                }
            }
        }

        environments
    }
}
