import numpy as np
import time
from options.train_options import TrainOptions
from data import create_dataset
from models import create_model
from util.visualizer import Visualizer
import util
if __name__ == '__main__':
    opt = TrainOptions().parse()   # get training options
    dataset = create_dataset(opt,'train')  # create a dataset given opt.dataset_mode and other options
    dataset_size = len(dataset)
    validation = create_dataset(opt,'valid')
    validation_size = len(validation) # get the number of images in the dataset.
    print('The number of training images = %d' % dataset_size)
    print('The number of validation images = %d' % validation_size)
    model = create_model(opt)      # create a model given opt.model and other options
    model.setup(opt)               # regular setup: load and print networks; create schedulers
    visualizer = Visualizer(opt)   # create a visualizer that display/save images and plots
    total_iters = 0                # the total number of training iterations
    best_psnr = 0
    best_loss = 2
    for epoch in range(opt.epoch_count, opt.n_epochs + opt.n_epochs_decay + 1):    # outer loop for different epochs; we save the model by <epoch_count>, <epoch_count>+<save_latest_freq>
        epoch_start_time = time.time()  # timer for entire epoch
        iter_data_time = time.time()    # timer for data loading per iteration
        epoch_iter = 0                  # the number of training iterations in current epoch, reset to 0 every epoch
        visualizer.reset()              # reset the visualizer: make sure it saves the results to HTML at least once every epoch

        for i, data in enumerate(dataset):  # inner loop within one epoch
            iter_start_time = time.time()  # timer for computation per iteration
            if total_iters % opt.print_freq == 0:
                t_data = iter_start_time - iter_data_time

            total_iters += opt.batch_size
            epoch_iter += opt.batch_size
            model.set_param_s(epoch_iter)
            model.set_input(data)
            model.set_sigma(epoch_iter) # unpack data from dataset and apply preprocessing
            model.optimize_parameters()   # calculate loss functions, get gradients, update network weights

            if total_iters % opt.display_freq == 0:   # display images on visdom and save images to a HTML file
                save_result = total_iters % opt.update_html_freq == 0
                model.compute_visuals()
                visualizer.display_current_results(model.get_current_visuals(), epoch, save_result)

            if total_iters % opt.print_freq == 0:    # print training losses and save logging information to the disk
                losses = model.get_current_losses()
                t_comp = (time.time() - iter_start_time) / opt.batch_size
                visualizer.print_current_losses(epoch, epoch_iter, losses, t_comp, t_data)
                if opt.display_id > 0:
                    visualizer.plot_current_losses(epoch, float(epoch_iter) / dataset_size, losses)

            if total_iters % opt.save_latest_freq == 0:   # cache our latest model every <save_latest_freq> iterations
                print('saving the latest model (epoch %d, total_iters %d)' % (epoch, total_iters))
                save_suffix = 'iter_%d' % total_iters if opt.save_by_iter else 'latest'
                model.save_networks(save_suffix)

            iter_data_time = time.time()
        if epoch % opt.save_epoch_freq == 0:              # cache our model every <save_epoch_freq> epochs
            print('saving the model at the end of epoch %d, iters %d' % (epoch, total_iters))
            model.save_networks('latest')
            model.save_networks(epoch)
        
        valid_psnr = 0
        valid_loss = 0
        for i, data in enumerate(validation):
            model.set_input_val(data)
            psnr,loss = model.forward_psnr()
            valid_psnr += psnr
            valid_loss += loss
        valid_psnr /= validation_size
        valid_loss /= validation_size
        if valid_loss < best_loss:
            print('saving the best model of the best epoch %d, iters %d' % (epoch, total_iters))
            best_loss = valid_loss
            best_epoch = epoch
            #best_psnr = valid_psnr
        if valid_psnr > best_psnr:
            best_psnr = valid_psnr
            best_psnr_epoch = epoch
            model.save_networks('best')
            model.save_networks(best_psnr_epoch)
            
        print('epoch %d / %d \t Validation: %.4f dB Best_PSNR(Best epoch %d) : %.4f dB' % (epoch, opt.n_epochs + opt.n_epochs_decay, valid_psnr,best_psnr_epoch,best_psnr))
        print('epoch %d / %d \t Validation: %.4f dB Best_loss(Best epoch %d) : %.4f dB' % (epoch, opt.n_epochs + opt.n_epochs_decay, valid_loss,best_epoch,best_loss))        
        print('End of epoch %d / %d \t Time Taken: %d sec' % (epoch, opt.n_epochs + opt.n_epochs_decay, time.time() - epoch_start_time))
        model.update_learning_rate()                     # update learning rates at the end of every epoch.
