#!/usr/bin/env python3
"""
Setup Validation Script for ICLR 2026 Supplementary Material
Verifies that all dependencies and configurations are properly set up.
"""

import sys
import os
import importlib
from pathlib import Path

def check_python_version():
    """Check if Python version is compatible."""
    version = sys.version_info
    if version.major < 3 or (version.major == 3 and version.minor < 8):
        print("❌ Python 3.8+ required. Current version:", sys.version)
        return False
    print(f"✅ Python version: {version.major}.{version.minor}.{version.micro}")
    return True

def check_dependencies():
    """Check if all required packages are installed."""
    required_packages = [
        'requests', 'datasets', 'pandas', 'numpy', 'matplotlib', 
        'seaborn', 'sentence_transformers', 'nltk', 'scipy', 'scikit-learn'
    ]
    
    missing = []
    for package in required_packages:
        try:
            importlib.import_module(package.replace('-', '_'))
            print(f"✅ {package}")
        except ImportError:
            print(f"❌ {package} - Not installed")
            missing.append(package)
    
    if missing:
        print(f"\n🔧 Install missing packages with:")
        print(f"pip install {' '.join(missing)}")
        return False
    
    return True

def check_file_structure():
    """Check if all required files are present."""
    required_files = [
        'README.md',
        'requirements.txt', 
        'config.py',
        'experiments/aime24/parallel_template.py',
        'experiments/aime24/sequential_template.py',
        'experiments/aime25/parallel_template.py',
        'experiments/aime25/sequential_template.py',
        'experiments/gpqa_diamond/parallel_template.py',
        'experiments/gpqa_diamond/sequential_template.py',
        'experiments/creativity/creativity_experiment.py',
        'analysis/comprehensive_analysis.py'
    ]
    
    missing = []
    for file_path in required_files:
        if not Path(file_path).exists():
            print(f"❌ {file_path} - Missing")
            missing.append(file_path)
        else:
            print(f"✅ {file_path}")
    
    return len(missing) == 0

def check_api_configuration():
    """Check if API configuration is set up."""
    config_files = ['config.py', 'config_local.py']
    api_configured = False
    
    for config_file in config_files:
        if Path(config_file).exists():
            try:
                with open(config_file, 'r') as f:
                    content = f.read()
                    if 'YOUR_OPENROUTER_KEY' not in content:
                        print(f"✅ API key configured in {config_file}")
                        api_configured = True
                        break
                    else:
                        print(f"⚠️  {config_file} exists but API key not configured")
            except Exception as e:
                print(f"❌ Error reading {config_file}: {e}")
    
    if not api_configured:
        print("⚠️  API key not configured. Please set your OpenRouter API key.")
        print("   Either edit config.py or create config_local.py with your key.")
    
    return True  # Don't fail validation for API key

def main():
    """Run all validation checks."""
    print("🔍 ICLR 2026 Supplementary Material - Setup Validation")
    print("=" * 60)
    
    checks = [
        ("Python Version", check_python_version),
        ("Dependencies", check_dependencies), 
        ("File Structure", check_file_structure),
        ("API Configuration", check_api_configuration)
    ]
    
    all_passed = True
    for check_name, check_func in checks:
        print(f"\n📋 {check_name}:")
        print("-" * 40)
        result = check_func()
        all_passed = all_passed and result
    
    print("\n" + "=" * 60)
    if all_passed:
        print("🎉 All checks passed! Setup is ready for ICLR experiments.")
        print("\n📖 Next steps:")
        print("1. Configure your OpenRouter API key in config.py")
        print("2. Run a test experiment: python experiments/aime24/parallel_template.py")
        print("3. Analyze results: python analysis/comprehensive_analysis.py")
    else:
        print("❌ Some checks failed. Please address the issues above.")
        sys.exit(1)

if __name__ == "__main__":
    main()