"""Add runslow option and skip slow tests if not specified.

Taken from https://docs.pytest.org/en/latest/example/simple.html.
"""

from collections.abc import Iterable
from warnings import warn

import pytest
import torch
from pytest import Config, Item, Parser
from transformers import (
    PreTrainedModel,
    PreTrainedTokenizerBase,
)

from deception_detection.models import ModelName, get_model_and_tokenizer


@pytest.fixture(scope="session")
def mistral_tokenizer() -> PreTrainedTokenizerBase:
    _, tokenizer = get_model_and_tokenizer(ModelName.MISTRAL_3_1_24B, omit_model=True)
    return tokenizer


def pytest_addoption(parser: Parser) -> None:
    parser.addoption("--runslow", action="store_true", default=False, help="run slow tests")


def pytest_configure(config: Config) -> None:
    config.addinivalue_line("markers", "slow: mark test as slow to run")
    config.addinivalue_line("markers", "gpu: mark test as requiring gpu to run")


def pytest_collection_modifyitems(config: Config, items: Iterable[Item]) -> None:
    skip_slow = pytest.mark.skip(reason="need --runslow option to run")
    if not config.getoption("--runslow"):
        for item in items:
            if "slow" in item.keywords:
                item.add_marker(skip_slow)

    if not torch.cuda.is_available():
        skip_gpu = pytest.mark.skip(reason="need gpu to run")
        for item in items:
            if "gpu" in item.keywords:
                warn("GPU tests are skipped because no GPU is available", stacklevel=1)
                item.add_marker(skip_gpu)
