# Copyright 2024 The ProtEx Authors. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================

"""Converts BLAST scores to the same format as T5X inference output.

Output format should have rows like:
{"inputs": {"accession": "Q0HZU6", "label": "GO:0008150"}, "score": -0.04761}
"""

from absl import app
from absl import flags
import tensorflow as tf

from common import jsonl_utils


FLAGS = flags.FLAGS

flags.DEFINE_string("input", "", "Path to blast output tsv.")

flags.DEFINE_string("database_records", "", "Path to database records.")

flags.DEFINE_string("output", "", "Path to write generated scores.")

flags.DEFINE_bool("top", True, "Whether to restrict to top-1.")


def _extract_accession(original_string):
  return original_string.replace('accession="', "").replace('"', "")


def _read_blast_tsv(path):
  """Load TSV file generated by BLAST."""
  rows = []
  with tf.io.gfile.GFile(path, "r") as tsv_file:
    for line in tsv_file:
      line = line.rstrip()
      cols = line.split("\t")
      query_accession = _extract_accession(cols[0])
      neighbor_accession = _extract_accession(cols[1])
      # Column 11 is the bit score (i.e. alignment score).
      score = float(cols[11])
      rows.append((query_accession, neighbor_accession, score))
  print("Loaded %s rows from %s." % (len(rows), path))
  return rows


def _load_accession_to_labels_map(path):
  dataset = jsonl_utils.read(path)
  accession_to_labels = {}
  for record in dataset:
    accession_to_labels[record["accession"]] = record["labels"]
  return accession_to_labels


def main(unused_argv):
  blast_rows = _read_blast_tsv(FLAGS.input)
  accession_to_labels = _load_accession_to_labels_map(FLAGS.database_records)

  rows = []
  for query_accession, database_accession, score in blast_rows:
    labels = accession_to_labels[database_accession]
    for label in labels:
      json_dict = {
          "inputs": {
              "accession": query_accession,
              "label": label,
          },
          "score": score,
      }
      rows.append(json_dict)
  jsonl_utils.write(FLAGS.output, rows)


if __name__ == "__main__":
  app.run(main)
