import logging
from dataclasses import dataclass
from typing import List, Optional, Tuple

import torch
import torch.nn as nn
import torch.nn.functional as F

# pylint:disable=no-member


class ConvCompress(nn.Module):
    def __init__(self, dim, ratio=4):
        super().__init__()
        self.conv = nn.Conv1d(dim, dim, kernel_size=ratio, stride=ratio)

    def forward(self, mem):
        # mem shape: (mem_len, batch_size, dim_model)
        mem = mem.permute([1, 2, 0])
        compressed_mem = self.conv(mem)
        return compressed_mem.permute([2, 0, 1])


class PositionalEmbedding(nn.Module):
    def __init__(self, demb):
        super().__init__()

        self.demb = demb

        inv_freq = 1 / (10000**(torch.arange(0.0, demb, 2.0) / demb))
        self.register_buffer("inv_freq", inv_freq)

    def forward(self, pos_seq, bsz=None):
        sinusoid_inp = torch.ger(pos_seq, self.inv_freq)
        pos_emb = torch.cat([sinusoid_inp.sin(), sinusoid_inp.cos()], dim=-1)

        if bsz is not None:
            return pos_emb[:, None, :].expand(-1, bsz, -1)
        else:
            return pos_emb[:, None, :]


class PositionwiseFF(nn.Module):
    def __init__(self, d_model, d_inner, dropout, pre_lnorm=False, layer_norm_epsilon=1e-5):
        super().__init__()

        self.d_model = d_model
        self.d_inner = d_inner
        self.dropout = dropout

        self.CoreNet = nn.Sequential(
            nn.Linear(d_model, d_inner),
            nn.ReLU(inplace=True),
            nn.Dropout(dropout),
            nn.Linear(d_inner, d_model),
            nn.Dropout(dropout),
        )

        self.layer_norm = nn.LayerNorm(d_model, eps=layer_norm_epsilon)

        self.pre_lnorm = pre_lnorm

    def forward(self, inp):
        if self.pre_lnorm:
            # layer normalization + positionwise feed-forward
            core_out = self.CoreNet(self.layer_norm(inp))

            # residual connection
            output = core_out + inp
        else:
            # positionwise feed-forward
            core_out = self.CoreNet(inp)

            # residual connection + layer normalization
            output = self.layer_norm(inp + core_out)

        return output


class RelPartialLearnableMultiHeadAttn(nn.Module):
    def __init__(
        self,
        n_head,
        d_model,
        d_head,
        dropout,
        dropatt=0,
        tgt_len=None,
        ext_len=None,
        mem_len=None,
        pre_lnorm=False,
        r_r_bias=None,
        r_w_bias=None,
        layer_norm_epsilon=1e-5,
    ):
        super().__init__()

        self.n_head = n_head
        self.d_model = d_model
        self.d_head = d_head
        self.dropout = dropout

        self.qkv_net = nn.Linear(d_model, 3 * n_head * d_head, bias=False)

        self.drop = nn.Dropout(dropout)
        self.dropatt = nn.Dropout(dropatt)
        self.o_net = nn.Linear(n_head * d_head, d_model, bias=False)

        self.layer_norm = nn.LayerNorm(d_model, eps=layer_norm_epsilon)

        self.scale = 1 / (d_head**0.5)

        self.pre_lnorm = pre_lnorm

        if r_r_bias is None or r_w_bias is None:  # Biases are not shared
            self.r_r_bias = nn.Parameter(torch.FloatTensor(self.n_head, self.d_head))
            self.r_w_bias = nn.Parameter(torch.FloatTensor(self.n_head, self.d_head))
        else:
            self.r_r_bias = r_r_bias
            self.r_w_bias = r_w_bias

        self.r_net = nn.Linear(self.d_model, self.n_head * self.d_head, bias=False)

    def _rel_shift(self, x):
        zero_pad_shape = (x.size(0), 1) + x.size()[2:]
        zero_pad = torch.zeros(zero_pad_shape, device=x.device, dtype=x.dtype)
        x_padded = torch.cat([zero_pad, x], dim=1)

        x_padded_shape = (x.size(1) + 1, x.size(0)) + x.size()[2:]
        x_padded = x_padded.view(*x_padded_shape)

        x = x_padded[1:].view_as(x)

        return x

    def forward(self, w, r, attn_mask=None, mems=None, cmems=None):
        qlen, rlen, bsz = w.size(0), r.size(0), w.size(1)

        if mems is not None:
            assert cmems is not None
            #
            all_mems = torch.cat([cmems, mems], 0)
            cat = torch.cat([all_mems, w], 0)
            w_heads = self.qkv_net(self.layer_norm(cat))
            r_head_k = self.r_net(r)

            w_head_q, w_head_k, w_head_v = torch.chunk(w_heads, 3, dim=-1)
            w_head_q = w_head_q[-qlen:]
        else:
            w_heads = self.qkv_net(self.layer_norm(w))

            r_head_k = self.r_net(r)

            w_head_q, w_head_k, w_head_v = torch.chunk(w_heads, 3, dim=-1)

        klen = w_head_k.size(0)

        w_head_q = w_head_q.view(qlen, bsz, self.n_head, self.d_head)  # qlen x bsz x n_head x d_head
        w_head_k = w_head_k.view(klen, bsz, self.n_head, self.d_head)  # qlen x bsz x n_head x d_head
        w_head_v = w_head_v.view(klen, bsz, self.n_head, self.d_head)  # qlen x bsz x n_head x d_head

        r_head_k = r_head_k.view(rlen, self.n_head, self.d_head)  # qlen x n_head x d_head

        # compute attention score
        rw_head_q = w_head_q + self.r_w_bias  # qlen x bsz x n_head x d_head
        AC = torch.einsum("ibnd,jbnd->ijbn", (rw_head_q, w_head_k))  # qlen x klen x bsz x n_head

        rr_head_q = w_head_q + self.r_r_bias
        BD = torch.einsum("ibnd,jnd->ijbn", (rr_head_q, r_head_k))  # qlen x klen x bsz x n_head
        BD = self._rel_shift(BD)

        # [qlen x klen x bsz x n_head]
        attn_score = AC + BD
        attn_score.mul_(self.scale)

        # compute attention probability
        if attn_mask is not None and torch.sum(attn_mask).item():
            attn_mask = attn_mask == 1  # Switch to bool
            if attn_mask.dim() == 2:
                if next(self.parameters()).dtype == torch.float16:
                    attn_score = (
                        attn_score.float().masked_fill(attn_mask[None, :, :, None], -65000).type_as(attn_score)
                    )
                else:
                    attn_score = attn_score.float().masked_fill(attn_mask[None, :, :, None], -1e30).type_as(attn_score)
            elif attn_mask.dim() == 3:
                if next(self.parameters()).dtype == torch.float16:
                    attn_score = attn_score.float().masked_fill(attn_mask[:, :, :, None], -65000).type_as(attn_score)
                else:
                    attn_score = attn_score.float().masked_fill(attn_mask[:, :, :, None], -1e30).type_as(attn_score)
            elif attn_mask.dim() == 4:
                # [qlen x klen x bsz x n_head]
                attn_score = attn_score.float().masked_fill(attn_mask, -1e30).type_as(attn_score)

        # [qlen x klen x bsz x n_head]
        attn_prob = F.softmax(attn_score, dim=1)
        attn_prob = self.dropatt(attn_prob)

        # compute attention vector
        attn_vec = torch.einsum("ijbn,jbnd->ibnd", (attn_prob, w_head_v))

        # [qlen x bsz x n_head x d_head]
        attn_vec = attn_vec.contiguous().view(attn_vec.size(0), attn_vec.size(1), self.n_head * self.d_head)

        # linear projection
        attn_out = self.o_net(attn_vec)
        attn_out = self.drop(attn_out)

        if self.pre_lnorm:
            # residual connection
            outputs = [w + attn_out]
        else:
            # residual connection + layer normalization
            outputs = [self.layer_norm(w + attn_out)]

        return outputs


class RelPartialLearnableDecoderLayer(nn.Module):
    def __init__(self, n_head, d_model, d_head, d_inner, dropout, layer_norm_epsilon=1e-5, **kwargs):
        super().__init__()

        self.dec_attn = RelPartialLearnableMultiHeadAttn(
            n_head, d_model, d_head, dropout, layer_norm_epsilon=layer_norm_epsilon, **kwargs
        )
        self.pos_ff = PositionwiseFF(
            d_model, d_inner, dropout, pre_lnorm=True, layer_norm_epsilon=layer_norm_epsilon
        )

    def forward(self, dec_inp, r, dec_attn_mask=None, mems=None, cmems=None, head_mask=None, output_attentions=False):

        attn_outputs = self.dec_attn(
            dec_inp,
            r,
            attn_mask=dec_attn_mask,
            mems=mems,
            cmems=cmems,
        )
        ff_output = self.pos_ff(attn_outputs[0])

        outputs = [ff_output] + attn_outputs[1:]

        return outputs


class CompressiveTransformerModel(nn.Module):
    def __init__(self, config):
        super().__init__()
        self.config = config

        self.n_token = config.vocab_size
        self.same_length = config.same_length
        self.clamp_len = config.clamp_len

        self.d_embed = config.d_embed
        self.d_model = config.d_model
        self.n_head = config.n_head
        self.d_head = config.d_head
        self.c_ratio = config.c_ratio
        self.cmem_len = config.cmem_len

        self.word_emb = nn.Embedding(config.vocab_size, config.d_model)

        self.drop = nn.Dropout(config.dropout)

        self.n_layer = config.n_layer

        self.tgt_len = config.tgt_len
        self.mem_len = config.mem_len
        self.ext_len = config.ext_len
        self.max_klen = config.tgt_len + config.ext_len + config.mem_len

        self.attn_type = config.attn_type

        if not config.untie_r:
            self.r_w_bias = nn.Parameter(torch.FloatTensor(self.n_head, self.d_head))
            self.r_r_bias = nn.Parameter(torch.FloatTensor(self.n_head, self.d_head))

        self.layers = nn.ModuleList()
        for i in range(config.n_layer):
            self.layers.append(
                RelPartialLearnableDecoderLayer(
                    config.n_head,
                    config.d_model,
                    config.d_head,
                    config.d_inner,
                    config.dropout,
                    tgt_len=config.tgt_len,
                    ext_len=config.ext_len,
                    mem_len=config.mem_len,
                    dropatt=config.dropatt,
                    pre_lnorm=config.pre_lnorm,
                    r_w_bias=None if config.untie_r else self.r_w_bias,
                    r_r_bias=None if config.untie_r else self.r_r_bias,
                    layer_norm_epsilon=config.layer_norm_epsilon,
                )
            )

        self.pos_emb = PositionalEmbedding(self.d_model)
        self.compress_funcs = nn.ModuleList(
            [ConvCompress(self.d_model, ratio=self.c_ratio) for _ in range(self.n_layer)]
        )

        self.init_weights()

    def init_weights(self):
        self.apply(self._init_weights)

    def _init_weight(self, weight):
        nn.init.normal_(weight, 0.0, self.config.init_std)

    def _init_weights(self, m):
        """ Initialize the weights.
        """
        classname = m.__class__.__name__
        if classname.find("Linear") != -1:
            if hasattr(m, "weight") and m.weight is not None:
                self._init_weight(m.weight)
            if hasattr(m, "bias") and m.bias is not None:
                nn.init.constant_(m.bias, 0.0)
        elif classname.find("Conv") != -1:
            if hasattr(m, "weight") and m.weight is not None:
                self._init_weight(m.weight)
            if hasattr(m, "bias") and m.bias is not None:
                nn.init.constant_(m.bias, 0.0)
        elif classname.find("Embedding") != -1:
            if hasattr(m, "weight"):
                self._init_weight(m.weight)
        elif classname.find("LayerNorm") != -1:
            if hasattr(m, "weight"):
                nn.init.normal_(m.weight, 1.0, self.config.init_std)
            if hasattr(m, "bias") and m.bias is not None:
                nn.init.constant_(m.bias, 0.0)
        else:
            if hasattr(m, "r_emb"):
                self._init_weight(m.r_emb)
            if hasattr(m, "r_w_bias"):
                self._init_weight(m.r_w_bias)
            if hasattr(m, "r_r_bias"):
                self._init_weight(m.r_r_bias)
            if hasattr(m, "r_bias"):
                self._init_bias(m.r_bias)

    def init_mems(self, bsz):
        mems = []
        cmems = []
        param = next(self.parameters())
        for i in range(self.n_layer):
            empty = torch.zeros(self.mem_len, bsz, self.config.d_model, dtype=param.dtype, device=param.device)
            mems.append(empty)
            cempty = torch.zeros(self.cmem_len, bsz, self.config.d_model, dtype=param.dtype, device=param.device)
            cmems.append(cempty)

        return cmems, mems

    def _update_mems(self, hids, cmems, mems):
        
        # update mems
        with torch.no_grad():
            new_mems = []
            for i in range(len(hids)):
                cat = torch.cat([mems[i], hids[i]], dim=0)
                new_mems.append(cat[-self.mem_len:].detach())

        n_s = min(hids[0].shape[0], mems[0].shape[0])

        # update compress mems
        with torch.no_grad():
            new_cmems = []
            for i in range(len(hids)):
                old_mems = mems[i][:n_s]
                new_cm = self.compress_funcs[i](old_mems)
                cat = torch.cat([cmems[i], new_cm])
                new_cmems.append(cat[-self.cmem_len:].detach())

        new_all_mems = new_cmems, new_mems

        return new_all_mems

    def compute_attn_loss(self, hids, old_mems):
        """Attention reconstruction loss"""
        def attn(h, m, Q, K, V):
            batch_size = h.shape[1]
            q = F.linear(h, Q, bias=False).view(-1, batch_size, self.n_head, self.d_head)
            k = F.linear(m, K, bias=False).view(-1, batch_size, self.n_head, self.d_head)
            v = F.linear(m, V, bias=False).view(-1, batch_size, self.n_head, self.d_head)
            # shape: (qlen, klen, batch_size, num_heads)
            attn_logits = torch.einsum("ibnd,jbnd->ijbn", q, k)
            attn_logits = attn_logits / (self.d_head ** 0.5)
            attn_probs = torch.softmax(attn_logits, dim=1)
            result = torch.einsum("ijbn,jbnd->ibnd", (attn_probs, v))
            return result

        L_attn = 0.0
        for i in range(len(hids)):
            hidden = hids[i].detach()
            old_mem = old_mems[i].detach()
            Q, K, V = torch.chunk(self.layers[i].dec_attn.qkv_net.weight.detach(), 3, dim=0)
            new_cm = self.compress_funcs[i](old_mem)
            L_attn = L_attn + F.mse_loss(attn(hidden, old_mem, Q, K, V), attn(hidden, new_cm, Q, K, V))

        return L_attn / self.n_layer

    def forward(
        self,
        input_ids,
        all_mems=None,
    ):
        input_ids = input_ids.transpose(0, 1).contiguous()
        qlen, bsz = input_ids.size()

        if all_mems is None:
            cmems, mems = self.init_mems(bsz)
        else:
            cmems, mems = all_mems

        word_emb = self.word_emb(input_ids)

        mlen = self.cmem_len + self.mem_len
        klen = mlen + qlen
        all_ones = word_emb.new_ones((qlen, klen), dtype=torch.uint8)

        if self.same_length:
            mask_len = klen - mlen
            if mask_len > 0:
                mask_shift_len = qlen - mask_len
            else:
                mask_shift_len = qlen
            dec_attn_mask = (torch.triu(all_ones, 1 + mlen) + torch.tril(all_ones, -mask_shift_len))[:, :, None]  # -1
        else:
            dec_attn_mask = torch.triu(all_ones, diagonal=1 + mlen)[:, :, None]

        hids = []

        pos_seq = torch.arange(klen - 1, -1, -1.0, device=word_emb.device, dtype=word_emb.dtype)
        if self.clamp_len > 0:
            pos_seq.clamp_(max=self.clamp_len)
        pos_emb = self.pos_emb(pos_seq)

        core_out = self.drop(word_emb)
        pos_emb = self.drop(pos_emb)

        for i, layer in enumerate(self.layers):
            hids.append(core_out)
            layer_outputs = layer(
                core_out,
                pos_emb,
                dec_attn_mask=dec_attn_mask,
                mems=mems[i],
                cmems=cmems[i]
            )
            core_out = layer_outputs[0]
        core_out = self.drop(core_out)

        new_cmems, new_mems = self._update_mems(hids, cmems, mems)
        

        core_out = core_out.transpose(0, 1).contiguous()

        if self.training:
            attn_loss = self.compute_attn_loss(hids, mems)
        else:
            attn_loss = 0.0

        new_all_mems = new_cmems, new_mems

        return core_out, new_all_mems, attn_loss