# TResNet

A **TResNet** is a variant on a [ResNet](https://paperswithcode.com/method/resnet) that aim to boost accuracy while maintaining GPU training and inference efficiency.  They contain several design tricks including a SpaceToDepth stem, [Anti-Alias downsampling](https://paperswithcode.com/method/anti-alias-downsampling), In-Place Activated BatchNorm, Blocks selection and [squeeze-and-excitation layers](https://paperswithcode.com/method/squeeze-and-excitation-block).

## How do I use this model on an image?
To load a pretrained model:

```python
import external.timm.pytorch_image_models.timm

model = external.resnet_strikes.pytorch_image_models.timm.create_model('tresnet_l', pretrained=True)
model.eval()
```

To load and preprocess the image:
```python 
import urllib
from PIL import Image
from external.resnet_strikes.pytorch_image_models.timm.data import resolve_data_config
from external.resnet_strikes.pytorch_image_models.timm.data.transforms_factory import create_transform

config = resolve_data_config({}, model=model)
transform = create_transform(**config)

url, filename = ("https://github.com/pytorch/hub/raw/master/images/dog.jpg", "dog.jpg")
urllib.request.urlretrieve(url, filename)
img = Image.open(filename).convert('RGB')
tensor = transform(img).unsqueeze(0) # transform and add batch dimension
```

To get the model predictions:
```python
import torch
with torch.no_grad():
    out = model(tensor)
probabilities = torch.nn.functional.softmax(out[0], dim=0)
print(probabilities.shape)
# prints: torch.Size([1000])
```

To get the top-5 predictions class names:
```python
# Get imagenet class mappings
url, filename = ("https://raw.githubusercontent.com/pytorch/hub/master/imagenet_classes.txt", "imagenet_classes.txt")
urllib.request.urlretrieve(url, filename) 
with open("imagenet_classes.txt", "r") as f:
    categories = [s.strip() for s in f.readlines()]

# Print top categories per image
top5_prob, top5_catid = torch.topk(probabilities, 5)
for i in range(top5_prob.size(0)):
    print(categories[top5_catid[i]], top5_prob[i].item())
# prints class names and probabilities like:
# [('Samoyed', 0.6425196528434753), ('Pomeranian', 0.04062102362513542), ('keeshond', 0.03186424449086189), ('white wolf', 0.01739676296710968), ('Eskimo dog', 0.011717947199940681)]
```

Replace the model name with the variant you want to use, e.g. `tresnet_l`. You can find the IDs in the model summaries at the top of this page.

To extract image features with this model, follow the [timm feature extraction examples](https://rwightman.github.io/pytorch-image-models/feature_extraction/), just change the name of the model you want to use.

## How do I finetune this model?
You can finetune any of the pre-trained models just by changing the classifier (the last layer).
```python
model = external.resnet_strikes.pytorch_image_models.timm.create_model('tresnet_l', pretrained=True, num_classes=NUM_FINETUNE_CLASSES)
```
To finetune on your own dataset, you have to write a training loop or adapt [timm's training
script](https://github.com/rwightman/pytorch-image-models/blob/master/train.py) to use your dataset.

## How do I train this model?

You can follow the [timm recipe scripts](https://rwightman.github.io/pytorch-image-models/scripts/) for training a new model afresh.

## Citation

```BibTeX
@misc{ridnik2020tresnet,
      title={TResNet: High Performance GPU-Dedicated Architecture}, 
      author={Tal Ridnik and Hussam Lawen and Asaf Noy and Emanuel Ben Baruch and Gilad Sharir and Itamar Friedman},
      year={2020},
      eprint={2003.13630},
      archivePrefix={arXiv},
      primaryClass={cs.CV}
}
```

<!--
Type: model-index
Collections:
- Name: TResNet
  Paper:
    Title: 'TResNet: High Performance GPU-Dedicated Architecture'
    URL: https://paperswithcode.com/paper/tresnet-high-performance-gpu-dedicated
Models:
- Name: tresnet_l
  In Collection: TResNet
  Metadata:
    FLOPs: 10873416792
    Parameters: 53456696
    File Size: 224440219
    Architecture:
    - 1x1 Convolution
    - Anti-Alias Downsampling
    - Convolution
    - Global Average Pooling
    - InPlace-ABN
    - Leaky ReLU
    - ReLU
    - Residual Connection
    - Squeeze-and-Excitation Block
    Tasks:
    - Image Classification
    Training Techniques:
    - AutoAugment
    - Cutout
    - Label Smoothing
    - SGD with Momentum
    - Weight Decay
    Training Data:
    - ImageNet
    Training Resources: 8x NVIDIA 100 GPUs
    ID: tresnet_l
    LR: 0.01
    Epochs: 300
    Crop Pct: '0.875'
    Momentum: 0.9
    Image Size: '224'
    Weight Decay: 0.0001
    Interpolation: bilinear
  Code: https://github.com/rwightman/pytorch-image-models/blob/9a25fdf3ad0414b4d66da443fe60ae0aa14edc84/timm/models/tresnet.py#L267
  Weights: https://github.com/rwightman/pytorch-image-models/releases/download/v0.1-tresnet/tresnet_l_81_5-235b486c.pth
  Results:
  - Task: Image Classification
    Dataset: ImageNet
    Metrics:
      Top 1 Accuracy: 81.49%
      Top 5 Accuracy: 95.62%
- Name: tresnet_l_448
  In Collection: TResNet
  Metadata:
    FLOPs: 43488238584
    Parameters: 53456696
    File Size: 224440219
    Architecture:
    - 1x1 Convolution
    - Anti-Alias Downsampling
    - Convolution
    - Global Average Pooling
    - InPlace-ABN
    - Leaky ReLU
    - ReLU
    - Residual Connection
    - Squeeze-and-Excitation Block
    Tasks:
    - Image Classification
    Training Techniques:
    - AutoAugment
    - Cutout
    - Label Smoothing
    - SGD with Momentum
    - Weight Decay
    Training Data:
    - ImageNet
    Training Resources: 8x NVIDIA 100 GPUs
    ID: tresnet_l_448
    LR: 0.01
    Epochs: 300
    Crop Pct: '0.875'
    Momentum: 0.9
    Image Size: '448'
    Weight Decay: 0.0001
    Interpolation: bilinear
  Code: https://github.com/rwightman/pytorch-image-models/blob/9a25fdf3ad0414b4d66da443fe60ae0aa14edc84/timm/models/tresnet.py#L285
  Weights: https://github.com/rwightman/pytorch-image-models/releases/download/v0.1-tresnet/tresnet_l_448-940d0cd1.pth
  Results:
  - Task: Image Classification
    Dataset: ImageNet
    Metrics:
      Top 1 Accuracy: 82.26%
      Top 5 Accuracy: 95.98%
- Name: tresnet_m
  In Collection: TResNet
  Metadata:
    FLOPs: 5733048064
    Parameters: 41282200
    File Size: 125861314
    Architecture:
    - 1x1 Convolution
    - Anti-Alias Downsampling
    - Convolution
    - Global Average Pooling
    - InPlace-ABN
    - Leaky ReLU
    - ReLU
    - Residual Connection
    - Squeeze-and-Excitation Block
    Tasks:
    - Image Classification
    Training Techniques:
    - AutoAugment
    - Cutout
    - Label Smoothing
    - SGD with Momentum
    - Weight Decay
    Training Data:
    - ImageNet
    Training Resources: 8x NVIDIA 100 GPUs
    Training Time: < 24 hours
    ID: tresnet_m
    LR: 0.01
    Epochs: 300
    Crop Pct: '0.875'
    Momentum: 0.9
    Image Size: '224'
    Weight Decay: 0.0001
    Interpolation: bilinear
  Code: https://github.com/rwightman/pytorch-image-models/blob/9a25fdf3ad0414b4d66da443fe60ae0aa14edc84/timm/models/tresnet.py#L261
  Weights: https://github.com/rwightman/pytorch-image-models/releases/download/v0.1-tresnet/tresnet_m_80_8-dbc13962.pth
  Results:
  - Task: Image Classification
    Dataset: ImageNet
    Metrics:
      Top 1 Accuracy: 80.8%
      Top 5 Accuracy: 94.86%
- Name: tresnet_m_448
  In Collection: TResNet
  Metadata:
    FLOPs: 22929743104
    Parameters: 29278464
    File Size: 125861314
    Architecture:
    - 1x1 Convolution
    - Anti-Alias Downsampling
    - Convolution
    - Global Average Pooling
    - InPlace-ABN
    - Leaky ReLU
    - ReLU
    - Residual Connection
    - Squeeze-and-Excitation Block
    Tasks:
    - Image Classification
    Training Techniques:
    - AutoAugment
    - Cutout
    - Label Smoothing
    - SGD with Momentum
    - Weight Decay
    Training Data:
    - ImageNet
    Training Resources: 8x NVIDIA 100 GPUs
    ID: tresnet_m_448
    LR: 0.01
    Epochs: 300
    Crop Pct: '0.875'
    Momentum: 0.9
    Image Size: '448'
    Weight Decay: 0.0001
    Interpolation: bilinear
  Code: https://github.com/rwightman/pytorch-image-models/blob/9a25fdf3ad0414b4d66da443fe60ae0aa14edc84/timm/models/tresnet.py#L279
  Weights: https://github.com/rwightman/pytorch-image-models/releases/download/v0.1-tresnet/tresnet_m_448-bc359d10.pth
  Results:
  - Task: Image Classification
    Dataset: ImageNet
    Metrics:
      Top 1 Accuracy: 81.72%
      Top 5 Accuracy: 95.57%
- Name: tresnet_xl
  In Collection: TResNet
  Metadata:
    FLOPs: 15162534034
    Parameters: 75646610
    File Size: 314378965
    Architecture:
    - 1x1 Convolution
    - Anti-Alias Downsampling
    - Convolution
    - Global Average Pooling
    - InPlace-ABN
    - Leaky ReLU
    - ReLU
    - Residual Connection
    - Squeeze-and-Excitation Block
    Tasks:
    - Image Classification
    Training Techniques:
    - AutoAugment
    - Cutout
    - Label Smoothing
    - SGD with Momentum
    - Weight Decay
    Training Data:
    - ImageNet
    Training Resources: 8x NVIDIA 100 GPUs
    ID: tresnet_xl
    LR: 0.01
    Epochs: 300
    Crop Pct: '0.875'
    Momentum: 0.9
    Image Size: '224'
    Weight Decay: 0.0001
    Interpolation: bilinear
  Code: https://github.com/rwightman/pytorch-image-models/blob/9a25fdf3ad0414b4d66da443fe60ae0aa14edc84/timm/models/tresnet.py#L273
  Weights: https://github.com/rwightman/pytorch-image-models/releases/download/v0.1-tresnet/tresnet_xl_82_0-a2d51b00.pth
  Results:
  - Task: Image Classification
    Dataset: ImageNet
    Metrics:
      Top 1 Accuracy: 82.05%
      Top 5 Accuracy: 95.93%
- Name: tresnet_xl_448
  In Collection: TResNet
  Metadata:
    FLOPs: 60641712730
    Parameters: 75646610
    File Size: 224440219
    Architecture:
    - 1x1 Convolution
    - Anti-Alias Downsampling
    - Convolution
    - Global Average Pooling
    - InPlace-ABN
    - Leaky ReLU
    - ReLU
    - Residual Connection
    - Squeeze-and-Excitation Block
    Tasks:
    - Image Classification
    Training Techniques:
    - AutoAugment
    - Cutout
    - Label Smoothing
    - SGD with Momentum
    - Weight Decay
    Training Data:
    - ImageNet
    Training Resources: 8x NVIDIA 100 GPUs
    ID: tresnet_xl_448
    LR: 0.01
    Epochs: 300
    Crop Pct: '0.875'
    Momentum: 0.9
    Image Size: '448'
    Weight Decay: 0.0001
    Interpolation: bilinear
  Code: https://github.com/rwightman/pytorch-image-models/blob/9a25fdf3ad0414b4d66da443fe60ae0aa14edc84/timm/models/tresnet.py#L291
  Weights: https://github.com/rwightman/pytorch-image-models/releases/download/v0.1-tresnet/tresnet_l_448-940d0cd1.pth
  Results:
  - Task: Image Classification
    Dataset: ImageNet
    Metrics:
      Top 1 Accuracy: 83.06%
      Top 5 Accuracy: 96.19%
-->