#!/usr/bin/env python

import functools
import inspect
import sys
from abc import ABCMeta
from functools import partial, wraps
from inspect import Parameter, signature
from typing import Callable, Optional

PARAM_TYPES = {
    "KEYWORD_ONLY": Parameter.KEYWORD_ONLY,
    "POSITIONAL_ONLY": Parameter.POSITIONAL_ONLY,
    "POSITIONAL_OR_KEYWORD": Parameter.POSITIONAL_OR_KEYWORD,
    "VAR_KEYWORD": Parameter.VAR_KEYWORD,
    "VAR_POSITIONAL": Parameter.VAR_POSITIONAL,
}


def debug(func):
    @wraps(func)
    def wrapper(*func_args, **func_kwargs):
        print("\n" + "-" * 80 + "\n", flush=True)
        print(f">>> Entering {func}", flush=True)
        print(f"    {signature(func)=}", flush=True)
        print(f"    {func_args=}", flush=True)
        print(f"    {func_kwargs=}", flush=True)
        return_value = func(*func_args, **func_kwargs)
        print(f"    {return_value=}", flush=True)
        print(f"<<< Exiting  {func}", flush=True)
        return return_value

    return wrapper


#
#
# @debug
# def parametrized_decorator(decorator, *meta_args, **meta_kwargs):
#     print(dir(decorator))
#     @debug
#     def _autogenerated_decorator(*decorator_args, **decorator_kwargs):
#         @debug
#         def wrapper(decorator, *wrapper_args, **wrapper_kwargs):
#             result = decorator(*decorator_args, **decorator_kwargs)
#             return result
#         return wrapper
#     return _autogenerated_decorator
#
#
# @parametrized_decorator
# @debug
# def add(function: Callable, bias=0.0):
#     @wraps(function)
#     @debug
#     def wrapper(*function_args, **function_kwargs):
#         result = function(*function_args, **function_kwargs) + bias
#         return result
#     return wrapper


# @add(bias=3)
# def identity(x):
#     return x
# print(identity(3))


# @debug
# def multiplying(func=None, factor=1):
#     assert callable(func) or func is None
#
#     @debug
#     def _decorator(func):
#         @functools.wraps(func)
#         @debug
#         def wrapper(*args, **kwargs):
#             return factor * func(*args, **kwargs)
#         return wrapper
#
#     if func is None:
#         return _decorator
#     return _decorator(func)


# @debug
# def multiplying(func, *args, factor=1, **kwargs):
#     assert callable(func) or func is None
#
#     @debug
#     def _decorator(func):
#         @functools.wraps(func)
#         @debug
#         def wrapper(*args, **kwargs):
#             return factor * func(*args, **kwargs)
#         return wrapper
#
#     if func is None:
#         return _decorator
#     return _decorator(func)


# @debug
# def parametrized(decorator):
#     @debug
#     def generated_decorator(*args, **kwargs):
#         @debug
#         def wrapper(func):
#             return decorator(func, *args, **kwargs)
#         return wrapper
#     return generated_decorator


# @debug
# def parametrized_decorator(decorator):
#     """decorator needs to have the signature deco(func, *args, **kwargs)."""
#
#     @debug
#     def generated_decorator(func=None, **kwargs):
#         print(F"{func=}")
#         if func is None:
#             return partial(decorator, **kwargs)
#         return decorator(func, **kwargs)
#
#     return generated_decorator
#
#
# @parametrized_decorator
# def multiplying(func=None, factor=1):
#
#     @functools.wraps(func)
#     @debug
#     def wrapper(*args, **kwargs):
#         return factor * func(*args, **kwargs)
#
#     return wrapper


PARAM_KINDS = {
    "KEYWORD_ONLY",
    "POSITIONAL_ONLY",
    "POSITIONAL_OR_KEYWORD",
    "VAR_KEYWORD",
    "VAR_POSITIONAL",
}


@debug
def is_strict(fun):
    """Check whether the function signature is strict"""
    sig = signature(fun)
    print(sig)
    for param in sig.parameters.values():
        print(f"{param=}, kind={param.kind}", flush=True)
        if param.kind == param.POSITIONAL_OR_KEYWORD:
            return False
    return True


@debug
def parametrized_decorator(decorator):
    """We do not allow general parametrization.


    If the function definition contains positional-only parameters, or
    any positionalwithout default values, then

    @decorator
    def fun(...)

    is forbidden.

    If you have parameters that are not


    """

    """decorator needs to have the signature deco(func, *args, **kwargs)."""

    sig = signature(decorator)
    parameters = sig.parameters

    print(sig, flush=True)
    for param in parameters.values():
        print(f"{param=}, kind={param.kind}", flush=True)

    num_positionals = len(
        [
            param
            for param in parameters.values()
            if param.kind == Parameter.POSITIONAL_ONLY
        ]
    )
    print(f"{num_positionals=}")

    assert (
        num_positionals >= 1
    ), "The decorator needs to have at least 1 positional only argument __func__ in the the first position"

    if not is_strict(decorator):
        raise ValueError("Decorator is not strict!")

    print(f">>>>>{num_positionals=} <<<<<<", flush=True)

    if num_positionals > 1:
        # The decorator has positional arguments - they need to be called by keyword arguments!
        @debug
        def generated_decorator(*args, **kwargs):
            return partial(decorator, *args, **kwargs)

        return generated_decorator

    @debug
    def generated_decorator(__func__=None, **kwargs):
        print(f"{__func__=}")
        if __func__ is None:
            return partial(decorator, **kwargs)
        return decorator(__func__)

    return generated_decorator


@parametrized_decorator
@debug
def linear(func, w=1, /, *, bias=0):
    @functools.wraps(func)
    @debug
    def wrapper(*args, **kwargs):
        return w * func(*args, **kwargs) + bias

    return wrapper


print("\n" + "=" * 80)

is_strict(linear)

print("\n" + "=" * 80)


@linear
@debug
def summing(x):
    return sum(x)


print(summing(range(10)))
print("\n" + "=" * 80)


@linear()
@debug
def summing(x):
    return sum(x)


print(summing(range(10)))
print("\n" + "=" * 80)


@linear(bias=3)
@debug
def summing(x):
    return sum(x)


print(summing(range(10)))
print("\n" + "=" * 80)


@parametrized_decorator
@debug
def linear(func, w, /, *, bias=0):
    @functools.wraps(func)
    @debug
    def wrapper(*args, **kwargs):
        return w * func(*args, **kwargs) + bias

    return wrapper


is_strict(linear)

print("\n" + "=" * 80)


@linear(1)
@debug
def summing(x):
    return sum(x)


print(summing(range(10)))
print("\n" + "=" * 80)


@linear(2)
@debug
def summing(x):
    return sum(x)


print(summing(range(10)))
print("\n" + "=" * 80)


@linear(1, bias=3)
@debug
def summing(x):
    return sum(x)


print(summing(range(10)))
print("\n" + "=" * 80)
