import transformers
import torch
from .models_utils import BaseLM, find_layers
from transformers import LlamaForCausalLM, AutoTokenizer
import torch.nn.functional as F
from torch import nn
import torch
from tqdm import tqdm
from .quant import *
import re

class LLAMAClass(BaseLM):

    def __init__(self, args):

        super().__init__()

        self.args = args
        self._device = torch.device(
            "cuda" if torch.cuda.is_available() else "cpu")
        self.model_name = args.model
        self.batch_size_per_gpu = args.batch_size

        self.model = LlamaForCausalLM.from_pretrained(self.model_name,
                                                    torch_dtype='auto')
        self.seqlen = self.model.config.max_position_embeddings
        self.model.eval()

        # pretrained tokenizer for neo is broken for now so just hard-coding this to gpt2
        self.tokenizer = AutoTokenizer.from_pretrained(self.model_name,
                                                       use_fast=False)
        self.vocab_size = self.tokenizer.vocab_size
        print('LLAMA vocab size: ', self.vocab_size)

    @property
    def eot_token_id(self):
        # we use EOT because end of *text* is more accurate for what we're doing than end of *sentence*
        return self.tokenizer.eos_token_id

    @property
    def max_length(self):
        try:
            return self.gpt2.config.n_ctx
        except AttributeError:
            # gptneoconfig doesn't have n_ctx apparently
            return self.model.config.max_position_embeddings

    @property
    def max_gen_toks(self):
        print('max_gen_toks fn')
        return 256

    @property
    def batch_size(self):
        # TODO: fix multi-gpu
        return self.batch_size_per_gpu  # * gpus

    @property
    def device(self):
        # TODO: fix multi-gpu
        return self._device

    def tok_encode(self, string: str):
        return self.tokenizer.encode(string, add_special_tokens=False)

    def tok_decode(self, tokens):
        return self.tokenizer.decode(tokens)

    def _model_call(self, inps):
        """
        inps: a torch tensor of shape [batch, sequence]
        the size of sequence may vary from call to call
        returns: a torch tensor of shape [batch, sequence, vocab] with the
        logits returned from the model
        """
        with torch.no_grad():
            return self.model(inps)[0][:, :, :self.vocab_size]

    @torch.no_grad()
    def _model_logits_on_dataset(self, dataset_inps):
        #print('Evaluating ...')

        nsamples = len(dataset_inps)

        model = self.model
        dev = self.device

        use_cache = model.config.use_cache
        model.config.use_cache = False
        layers = model.model.layers

        model.model.embed_tokens = model.model.embed_tokens.to(dev)
        model.model.rotary_emb = model.model.rotary_emb.to(dev)
        layers[0] = layers[0].to(dev)

        dtype = next(iter(model.parameters())).dtype
        inps = []
        outs = []
        for batch_idx, batch in enumerate(dataset_inps):
            inps.append(
                torch.zeros(
                    (batch.shape[1], self.model.config.hidden_size),
                    dtype=dtype,
                ))
            outs.append(
                torch.zeros(
                    (batch.shape[1], self.model.config.hidden_size),
                    dtype=dtype,
                ))

        cache = {'i': 0, 'kwargs': []}

        class Catcher(nn.Module):

            def __init__(self, module):
                super().__init__()
                self.module = module

            def forward(self, inp, **kwargs):
                inps[cache['i']] = inp
                cache['i'] += 1
                cache['kwargs'].append(kwargs)
                raise ValueError

        layers[0] = Catcher(layers[0])
        for i in range(nsamples):
            batch = dataset_inps[i].to(dev)
            try:
                model(batch)
            except ValueError:
                pass
        layers[0] = layers[0].module

        layers[0] = layers[0].cpu()
        model.model.embed_tokens = model.model.embed_tokens.cpu()
        model.model.rotary_emb = model.model.rotary_emb.cpu()
        torch.cuda.empty_cache()

        layer_kwags = cache['kwargs']

        for i in tqdm(range(len(layers))):
            # print(i)
            layer = layers[i].to(dev)

            # if self.args.nearest:
            #     subset = find_layers(layer)
            #     for name in subset:
            #         quantizer = Quantizer()
            #         quantizer.configure(self.args.wbits,
            #                             perchannel=True,
            #                             sym=False,
            #                             mse=False)
            #         W = subset[name].weight.data
            #         quantizer.find_params(W, weight=True)
            #         subset[name].weight.data = quantize(
            #             W, quantizer.scale, quantizer.zero, quantizer.maxq).to(
            #                 next(iter(layer.parameters())).dtype)

            for j in range(nsamples):
                outs[j] = layer(inps[j].to(self.device), **layer_kwags[j])[0].detach().cpu()

            layers[i] = layer.cpu()
            del layer
            torch.cuda.empty_cache()
            inps, outs = outs, inps

        if model.model.norm is not None:
            model.model.norm = model.model.norm.to(dev)
        model.lm_head = model.lm_head.to(dev)

        if self.model.model.norm is not None:
            self.model.model.norm = self.model.model.norm.to(self.device)
        self.model.lm_head = self.model.lm_head.to(self.device)

        dataset_logits = []

        for i in range(nsamples):
            hidden_states = inps[i].unsqueeze(0).to(self.device)
            if self.model.model.norm is not None:
                hidden_states = self.model.model.norm(hidden_states)
            batch_logits = F.log_softmax(
                self.model.lm_head(hidden_states)[0][:, :, :self.vocab_size],
                dim=-1).cpu()
            dataset_logits.append(batch_logits)
        model.config.use_cache = use_cache
        return dataset_logits

    def model_batched_set(self, inps):
        import pdb
        pdb.set_trace()
        dataset_logits = []
        for batch in inps:
            multi_logits = F.log_softmax(
                self._model_call(batch),
                dim=-1).cpu()  # [batch, padding_length, vocab]
            dataset_logits.append(multi_logits)
        return dataset_logits

    def _model_generate(self, context, max_length, eos_token_id):
        return self.model.generate(context,
                                   max_length=max_length,
                                   eos_token_id=eos_token_id,
                                   do_sample=False)

    # @torch.no_grad()
    # def opt_sequential(self, dataloader):
    #     print('Starting ...')

    #     model = self.model
    #     dev = self.device

    #     use_cache = model.config.use_cache
    #     model.config.use_cache = False
    #     layers = model.model.decoder.layers

    #     model.model.decoder.embed_tokens = model.model.decoder.embed_tokens.to(
    #         dev)
    #     model.model.decoder.embed_positions = model.model.decoder.embed_positions.to(
    #         dev)
    #     if hasattr(model.model.decoder,
    #                'project_out') and model.model.decoder.project_out:
    #         model.model.decoder.project_out = model.model.decoder.project_out.to(
    #             dev)
    #     if hasattr(model.model.decoder,
    #                'project_in') and model.model.decoder.project_in:
    #         model.model.decoder.project_in = model.model.decoder.project_in.to(
    #             dev)
    #     layers[0] = layers[0].to(dev)

    #     dtype = next(iter(model.parameters())).dtype
    #     inps = torch.zeros(
    #         (self.args.nsamples, self.seqlen, model.config.hidden_size),
    #         dtype=dtype,
    #         device=dev)
    #     cache = {'i': 0, 'attention_mask': None}

    #     class Catcher(nn.Module):

    #         def __init__(self, module):
    #             super().__init__()
    #             self.module = module

    #         def forward(self, inp, **kwargs):
    #             inps[cache['i']] = inp
    #             cache['i'] += 1
    #             cache['attention_mask'] = kwargs['attention_mask']
    #             raise ValueError

    #     layers[0] = Catcher(layers[0])
    #     for batch in dataloader:
    #         try:
    #             model(batch[0].to(dev))
    #         except ValueError:
    #             pass
    #     layers[0] = layers[0].module

    #     layers[0] = layers[0].cpu()
    #     model.model.decoder.embed_tokens = model.model.decoder.embed_tokens.cpu(
    #     )
    #     model.model.decoder.embed_positions = model.model.decoder.embed_positions.cpu(
    #     )
    #     if hasattr(model.model.decoder,
    #                'project_out') and model.model.decoder.project_out:
    #         model.model.decoder.project_out = model.model.decoder.project_out.cpu(
    #         )
    #     if hasattr(model.model.decoder,
    #                'project_in') and model.model.decoder.project_in:
    #         model.model.decoder.project_in = model.model.decoder.project_in.cpu(
    #         )
    #     torch.cuda.empty_cache()

    #     outs = torch.zeros_like(inps)
    #     attention_mask = cache['attention_mask']

    #     print('Ready.')

    #     quantizers = {}
    #     for i in range(len(layers)):
    #         layer = layers[i].to(dev)

    #         subset = find_layers(layer)
    #         gptq = {}
    #         for name in subset:
    #             gptq[name] = GPTQ(subset[name])
    #             gptq[name].quantizer = Quantizer()
    #             gptq[name].quantizer.configure(self.args.wbits,
    #                                            perchannel=True,
    #                                            sym=False,
    #                                            mse=False)

    #         def add_batch(name):

    #             def tmp(_, inp, out):
    #                 gptq[name].add_batch(inp[0].data, out.data)

    #             return tmp

    #         handles = []
    #         for name in subset:
    #             handles.append(subset[name].register_forward_hook(
    #                 add_batch(name)))
    #         for j in range(self.args.nsamples):
    #             outs[j] = layer(inps[j].unsqueeze(0),
    #                             attention_mask=attention_mask)[0]
    #         for h in handles:
    #             h.remove()

    #         for name in subset:
    #             print(i, name)
    #             # print('Quantizing ...')
    #             gptq[name].fasterquant(percdamp=self.args.percdamp,
    #                                    groupsize=self.args.groupsize)
    #             quantizers['model.decoder.layers.%d.%s' %
    #                        (i, name)] = gptq[name].quantizer
    #             gptq[name].free()
    #         for j in range(self.args.nsamples):
    #             outs[j] = layer(inps[j].unsqueeze(0),
    #                             attention_mask=attention_mask)[0]

    #         layers[i] = layer.cpu()
    #         del layer
    #         del gptq
    #         torch.cuda.empty_cache()

    #         inps, outs = outs, inps

    #     model.config.use_cache = use_cache

    #     return quantizers


# for backwards compatibility
LLAMA = LLAMAClass
