"""Script for evaluation utilities
"""
import torch
import numpy as np


@torch.no_grad()
def evaluate_solver(model, valloader, device):
    """Evaluates the model and computes metrics

    Args:
        model (nn.Module): model to be evaluated
        valloader (DataLoader): validation data
        device (torch.device): device to send data to
    """
    predictions, groundtruths = [], []
    model.eval()
    for x, y in valloader:

        # send data to device
        x = x.to(device)
        y = y.to(device)

        # inference (evaluate on NN solution prediction)
        pred = model(x)

        # store metrics
        groundtruths.append(y)
        predictions.append(pred)

    groundtruths = torch.cat(groundtruths).detach().cpu().numpy()
    predictions = torch.cat(predictions).detach().cpu().numpy()

    # compute metrics
    outputs = {'groundtruth': groundtruths, 'prediction': predictions}
    metrics = {
        'RMSE': np.sqrt(((predictions - groundtruths)**2).mean()),
        'MAE': np.abs(predictions - groundtruths).mean(),
        'Max-AE': np.abs(predictions - groundtruths).max(),
        'Relative Error': np.linalg.norm(predictions-groundtruths)/np.linalg.norm(groundtruths)
    }

    return metrics, outputs
