from pipeline.evaluator.abstract_evaluator import Evaluator
import transformers
import torch
import re
import json

class Llama3Evaluator(Evaluator):
    """
    Evaluator using Meta-Llama-3-70B-Instruct model
    """
    def __init__(self, model_id = "Meta-Llama-3-70B-Instruct", cache_dir = None):
        self.temperature = 0
        model_kwargs = {"torch_dtype": torch.bfloat16}
        if cache_dir is not None:
            model_kwargs["cache_dir"] = cache_dir
            
        if model_id == "Meta-Llama-3-70B-Instruct":
            model_id = "meta-llama/Meta-Llama-3-70B-Instruct"
            
        self.model_pipeline = transformers.pipeline(
            "text-generation",
            model=model_id,
            model_kwargs=model_kwargs,
            device_map="auto",
        )
        #TODO: pass model config (model, cache_dir, temperature) as config dict

    def decision_extract(self, last_msg):
        matches = re.findall(r'\{[^{}]*\}', last_msg)
        if matches:
            try:
                decision_json = json.loads(matches[-1])
                decision = decision_json["decision"]
                if decision == "success":
                    return "success"
                elif decision == "partially":  # use 'partial' to match 'partially'
                    return "partially"
                elif decision == "failed":  # use 'fail' to match 'failed'
                    return "failed"
            except:
                print("decision parsing error")
        print("no matching decision")
        print(last_msg)
        return None

    def _generate_response(self, system_msg, user_msg):
        input_message = [
            {"role": "system", "content": system_msg},
            {"role": "user", "content": user_msg},
        ]
        
        prompt = self.model_pipeline.tokenizer.apply_chat_template(
                input_message, 
                tokenize=False, 
                add_generation_prompt=True
        )

        outputs = self.model_pipeline(
            prompt,
            eos_token_id=[
                self.model_pipeline.tokenizer.eos_token_id,
                self.model_pipeline.tokenizer.convert_tokens_to_ids("<|eot_id|>")
            ],
            do_sample=True,
            temperature=0.9,
        )
        return outputs[0]["generated_text"][len(prompt):], 0
        

