from langchain.pydantic_v1 import BaseModel, Field
from langchain.tools import tool
from langchain_core.tools import ToolException
from typing import Optional, Type
from zipfile import ZipFile
import shutil
import os
from langchain.tools import ShellTool
from langchain.tools.retriever import create_retriever_tool
from .vector_store import VectorStore
from langchain.chains import create_retrieval_chain
from langchain_openai import ChatOpenAI
from langchain.chains.combine_documents import create_stuff_documents_chain
from langchain_core.prompts import ChatPromptTemplate
from langchain.vectorstores import Weaviate
import weaviate
from weaviate.embedded import EmbeddedOptions


working_dir = "."

root = (os.path.dirname(os.path.realpath(".")))

# Build repo retriever


index_path = os.path.join(root, ".github/vector_store_faiss-v2")
print("index path: ", index_path)
vector = VectorStore("pipeline/models/lanchain/knowledgebase", "index")
retriever = vector.get_vector_store(True)

        
# Create the prompt template
prompt = ChatPromptTemplate.from_template("""You should answer following problem strictly follow the documents provided. Answer the following question based only on the provided context in a format which is suitable for Github comment:

<context>
{context}
</context>

Question: {input}""")

# Create the document chain
llm = ChatOpenAI(model_name="gpt-4-0125-preview", temperature=0)
document_chain = create_stuff_documents_chain(llm, prompt)

retrieval_chain = create_retrieval_chain(retriever, document_chain)
    


# RetrieverTool = create_retriever_tool(
#     retriever,
#     "guide_information_search",
#     "Search for information about how to contribute/review dataset. For any question, you must use this tool first!",
# )

# retriever two

# from langchain.document_loaders import TextLoader

# documents = []
# for doc_path in ['./GLI-Test/Docs/doc.txt', './GLI-Test/Docs/doc2.txt', './GLI-Test/Docs/doc3.md']:
#     loader = TextLoader(os.path.join(root, doc_path))
#     documents += loader.load()

# print(len(documents))
# from langchain.text_splitter import CharacterTextSplitter
# text_splitter = CharacterTextSplitter(chunk_size=500, chunk_overlap=50)
# chunks = text_splitter.split_documents(documents)

# client = weaviate.Client(
#   embedded_options = EmbeddedOptions()
# )

# vectorstore = Weaviate.from_documents(
#     client = client,    
#     documents = chunks,
#     embedding = OpenAIEmbeddings(),
#     by_text = False
# )

# retriever = vectorstore.as_retriever()

# RetrieverTool_predefined = create_retriever_tool(
#     retriever,
#     "guide_information_search",
#     "Search for information about how to contribute/review dataset. For any question, you must use this tool first!",
# )

####


shell_tool = ShellTool()

class ScriptConversionInput(BaseModel):
    pythonCode: str = Field(description="should be the Python code")
    scriptName: str = Field(description="should be the Python Script Name with '.py' suffix")
    append_text: bool = Field(description="False: cover the existing code.py context; True: append context to existing code.py")
        
class fileConversionInput(BaseModel):
    fileContent: str = Field(description="should be any valid file content according to its suffix")
    fileName: str = Field(description="should be assigned a proper name with corresponding suffix")
    
class fileLoaderInput(BaseModel):
    filePath: str = Field(description="where the file is stored")

class ZipExtractionInput(BaseModel):
    zipFilePath: str = Field(description="The path to the zip file to be extracted")
    extractTo: str = Field(default="extracted", description="The directory to extract the files into")
    
class CreateToolInput(BaseModel):
    Usage: str = Field(description="The usage of the tool you want to create")
    SavePath: str = Field(description="The path you want to save your created .py tool to")
    
class ReadFileStructureInput(BaseModel):
    WorkDir: str = working_dir

class ExecutePythonCodeInput(BaseModel):
    CodePath: str = Field(description="The path of the python code to execute")
    
class FileWriteInput(BaseModel):
    filePath: str = Field(description="where the file is stored")
    fileContent: str = Field(description="content to write into the file")
    
class FileFinderInput(BaseModel):
    fileName: str = Field(description="the name of the file or directory")
    
class RetrievalInput(BaseModel):
    query: str = Field(description="questions you want to ask the retrieval database")
    
class CreateDirInput(BaseModel):
    dir_name: str = Field(description="the name of the directory you want to create")
    parent_name: str = Field(description="the parent directory under which you will create a new directory")

# python_repl = PythonREPL()

# PythonCodeInterpreter = Tool(
#     name="python_repl",
#     description="A Python shell. Use this to execute python commands. Input should be a valid python command. If you want to see the output of a value, you should print it out with `print(...)`.",
#     func=python_repl.run,
# )

@tool("retrieval_tool", args_schema=RetrievalInput, return_direct=False)
def RetrievalTool(query: str) -> str:
    """retrieve the vector database to get the answer to the query"""
    
    # response = retrieval_chain.invoke({"input": "How to review an uploaded dataset? If you are asked to design some tests, what are the tests you will design?"})["answer"]
    response = retrieval_chain.invoke({"input": query})["answer"]

    return response


@tool("custom_python_code_script_conversion_tool", args_schema=ScriptConversionInput, return_direct=False)
def PythonScriptSaverTool( pythonCode: str, scriptName: str, append_text: bool) -> str:
    """
    Save Python code to a script, useful whenever you are going to generate the Python code
    If append_text = True, then the original code.py (if exists) will be covered; if append_text = False, then this tool will append text to code.py
    """

    mode = "w+"
    
    if append_text:
        mode = "a+"

    if not os.path.exists('scripts'):
        print("Creating scripts directory")
        os.makedirs('scripts')
        
    file_path = f"scripts/{scriptName}"

    try:
        with open(file_path,mode) as f:
            f.write(pythonCode)
    except Exception as e:
        # raise ToolException(f"An error occurred while saving the code: {e}")
        return "An error has been detected! Examine whether your datapath is correct!"
    return f"Code has been saved to {file_path}"


@tool("file_store_tool", args_schema=fileConversionInput, return_direct=False)
def FileSaverTool( fileContent: str, fileName: str) -> str:
    """Save Files to local, but for python code you should use custom_python_code_script_conversion_tool. You are forbid to use this tool to save python code"""
    # file_path = f"localStorage/{fileName}"
    # if not os.path.exists('localStorage'):
    #     print("Creating localStorage directory")
    #     os.makedirs('localStorage')
    
    file_path = f"{fileName}"
    try:
        with open(file_path,"w+") as f:
            f.write(fileContent)
    except Exception as e:
        raise ToolException(f"An error occurred while saving the file: {e}")
    return f"Code has been saved to {file_path}"

@tool("file_write_tool", args_schema=FileWriteInput, return_direct=False) 
def FileWriteTool(fileContent: str, filePath: str, append_text: bool) -> str:
    """Write fileContent to file"""
    # filePath = os.path.join("localStorage", filePath)
    if not os.path.exists(filePath):
        try:
            with open(filePath, "w+") as f:
                f.write(fileContent)
        except Exception as e:
            raise ToolException(f"An error occurred while writing the file: {e}")
    else:
        try:
            with open(filePath, "a+") as f:
                f.write(fileContent)
        except Exception as e:
            return "Retry this tool with different filePath!!! "
    return f"{fileContent} has been written to {filePath}"


@tool("file_loader_tool", args_schema=fileLoaderInput, return_direct=False)
def FileLoaderTool(filePath: str) -> str:
    """load file from filePath, return its content. This tool is only designed for file with proper suffix. 
    If you find that this tool cannot load the content of a certain file, then you should write your own loading tools, 
    use `creat_tool` tool"""
    if not os.path.exists(filePath):
        return (f"The file {filePath} does not exist!")
    try:
        with open(filePath,'r') as f:
            result = f.read()
            return result
    except Exception as e:
        raise ToolException(f"An error occurred while loading the file: {e}")


@tool("zip_extractor_tool", args_schema=ZipExtractionInput, return_direct=False)
def ZipExtractor(zipFilePath: str, extractTo: str = "extracted") -> str:
    """Extracts all files from a zip file to a specified directory. Better not change extractTo"""
    # if os.path.exists(extractTo):
    #     print(f"Clearing directory {extractTo}")
    #     shutil.rmtree(extractTo)
    os.makedirs(extractTo, exist_ok=True)

    try:
        with ZipFile(zipFilePath, 'r') as zip_ref:
            zip_ref.extractall(extractTo)
            extracted_files = zip_ref.namelist()
            dataset_path = f"{extractTo}/{zipFilePath}".split(".")[0]
            return f"Extracted {len(extracted_files)} files to {extractTo}. You should take {dataset_path} as dataset path"
    except Exception as e:
        raise ToolException(f"An error occurred while extracting the zip file: {e}")


def ManualZipExtractor( zipFilePath: str, extractTo: str = "extracted") -> str:

    os.makedirs(extractTo, exist_ok=True)

    try:
        with ZipFile(zipFilePath, 'r') as zip_ref:
            zip_ref.extractall(extractTo)
            extracted_files = zip_ref.namelist()
            dataset_path = f"{extractTo}/{zipFilePath}".split(".")[0]
            # return f"Extracted {len(extracted_files)} files to {extractTo}. You should take {dataset_path} as dataset path"
            return dataset_path
    except Exception as e:
        raise ToolException(f"An error occurred while extracting the zip file: {e}")



@tool("create_tool", args_schema=CreateToolInput, return_direct=False)
def CreateTool( usage: str, savePath: str):
    """Creat your own tool for Usage and store it to savePath. You should:
    1. Write your own tool python code
    2. Store it to local use FileSaverTool"""

def read_directory_structure_as_string( dir_path, indent=''):
    """Recursively reads a directory structure and returns it as a string with indentation."""
    lines = []
    for item in sorted(os.listdir(dir_path)):
        item_path = os.path.join(dir_path, item)
        lines.append(f"{indent}|- {item}")
        if os.path.isdir(item_path):
            lines.append(read_directory_structure_as_string(item_path, indent + '  '))
    return '\n'.join(lines)


@tool("create_dir", args_schema=CreateDirInput, return_direct=False)
def CreateDirTool(dir_name: str, parent_name: str):
    "Create a dir_name directory under parent_name directory"
    os.makedirs(os.path.join(parent_name,dir_name),exist_ok=True)


@tool("read_file_structure_tool", args_schema=ReadFileStructureInput, return_direct=False)
def ReadFileStructureTool( WorkDir: str = working_dir):
    """Read the full file structure of the working directory"""
    # return read_directory_structure_as_string(".")
    return read_directory_structure_as_string(WorkDir)

@tool("execute_python_code_tool", args_schema=ExecutePythonCodeInput, return_direct=False)
def ExecutePythonCodeTool(CodePath: str):
    """Execute the python code at CodePath"""
    return shell_tool.run({"commands":[f"python -W ignore {CodePath}"]})

@tool("file_finder_tool", args_schema=FileFinderInput, return_direct=False)
def FindFileTool(fileName: str) -> str:
    """Get the path of a file/directory start from current working directory "." """
    for root, dirs, files in os.walk("."):
        if fileName in files:
            return os.path.join(root, fileName)
    return "Not Found!"

def get_toolkit( option = "default"):
    if option == "split-test-no-unzip":
        return [RetrievalTool, PythonScriptSaverTool, FileLoaderTool, ExecutePythonCodeTool,ReadFileStructureTool, FindFileTool]
    elif option == "test-design":
        return [RetrievalTool, FileSaverTool]
    elif option == "test-design-predefined-docs":
        return [FileSaverTool, RetrievalTool]
    elif option == "script-executor":
        return [ReadFileStructureTool, FileLoaderTool]
    else:
        return [RetrievalTool, PythonScriptSaverTool, FileLoaderTool, ExecutePythonCodeTool, ZipExtractor, ReadFileStructureTool]



