from typing import Union
from langchain.vectorstores import FAISS
from langchain.embeddings import OpenAIEmbeddings
import os
import warnings
import os
from langchain.chains import create_retrieval_chain
from langchain.chains.combine_documents import create_stuff_documents_chain
from langchain_openai import ChatOpenAI, OpenAIEmbeddings
from langchain_core.prompts import ChatPromptTemplate
from langchain_community.vectorstores import FAISS

warnings.filterwarnings('ignore')

class VectorStore:
    """A class for managing vector stores, supporting operations such as building and retrieving vectors."""

    def __init__(self, index_path: str, index_name: str, store_type: str = "FAISS") -> None:
        """Initialize the VectorStore instance.
        
        Args:
            index_path (str): The filesystem path where the index is stored.
            index_name (str): The name of the index file.
            store_type (str, optional): The type of vector store. Defaults to "FAISS".
        """
        self.__index_path = index_path
        self.__index_name = index_name
        self.__store_type = store_type
        self.__embeddings = OpenAIEmbeddings() 

    def build(self) -> None:
        """TODO Build the vector store. This method is to be implemented as needed. Use"""
        pass

    def get_vector_store(self, as_retriever: bool = True):
        """Retrieve the vector store, optionally as a retriever object.
        
        Args:
            as_retriever (bool, optional): Whether to return the store as a retriever object. Defaults to True.
        
        Returns:
            Union[FAISS, 'RetrieverType']: The vector store or retriever object.
            
        Raises:
            FileNotFoundError: If the index file does not exist.
            NotImplementedError: If the store type is not supported.
        """
        
        if self.__store_type == "FAISS":
            vector_store = FAISS.load_local(self.__index_path, self.__embeddings, index_name=self.__index_name,allow_dangerous_deserialization = True)
        else:
            raise NotImplementedError(f"Store type '{self.__store_type}' is not implemented.")
        
        if as_retriever:
            return vector_store.as_retriever()
        else:
            return vector_store
        

if __name__ == "__main__":
    
    root = os.path.dirname(os.path.dirname(os.path.realpath(__file__)))
    index_path = os.path.join(root, ".github/vector_store_faiss-v2")
    index_name = "my_vector_index"
    
    vector = VectorStore(index_path, index_name, "FAISS")
    retriever = vector.get_vector_store()
    
    # Create the prompt template
    prompt = ChatPromptTemplate.from_template("""You should answer following problem strictly follow the documents provided. Answer the following question based only on the provided context in a format which is suitable for Github comment:

    <context>
    {context}
    </context>

    Question: {input}""")

    # Create the document chain
    llm = ChatOpenAI(model_name="gpt-4-0125-preview", temperature=0)
    document_chain = create_stuff_documents_chain(llm, prompt)
    
    retrieval_chain = create_retrieval_chain(retriever, document_chain)
        
    response = retrieval_chain.invoke({"input": "How to review an uploaded dataset? If you are asked to design some tests, what are the tests you will design?"})["answer"]
    
    print(response)


