from cProfile import label
from logging import root
from pathlib import Path
import re
from tkinter import font
import uuid
from pprint import pprint
from termcolor import colored
import os
import json
import numpy as np
from openai import OpenAI
import tiktoken
import sys
root_path = os.path.abspath(os.path.join(os.path.dirname(__file__), ".."))
print("root_path", root_path)
sys.path.append(root_path)
import pandas as pd
from matplotlib import pyplot as plt

def get_user_input(prompt_message):
    return input(prompt_message)

def print_colored(message, color='white'):
    print(colored(message, color))
    
def div_by_zero(a, b): 
    if (b == 0): 
        return a
    return a / b

def load_human_labels(path = None):
    """
    This function is to load the 92 human labeled test cases, in a set of 2-tuple (file_id, hint_level)
    The label is in the format of json, please see `label.json` under the same directory for the format
    """
    if path is None:
        assert False, "Please provide the path to the label file"
        labels = set()
        
    path = os.path.join(root_path, path)
    with open(path, "r") as f:
        labels = json.load(f)
    
    res = set()
    
    for hint_level, item in labels.items():
        for _, case_list in item.items():
            for case in case_list:
                res.add((case["id"], hint_level))
    
    return res
    
def get_openai_api_key():
    if os.environ["OPENAI_API_KEY"]:
        return os.environ["OPENAI_API_KEY"]
    else:
        return get_user_input("Please input your OpenAI API Key: ")
    
def get_user_input_multiline(prompt):
    print_colored(prompt + " (Please Input [E] in a newline to finish)", 'cyan')
    lines = []
    while True:
        line = input()
        if line == "[E]":
            break
        lines.append(line)
    return '\n'.join(lines) 

def flat_directory(directory_path):
    files = []
    for root, _, filenames in os.walk(directory_path): # os.walk() generates the file names in a directory tree by walking either top-down or bottom-up
        for filename in filenames:
            # if filename.startswith("."):
            #     continue
            files.append(os.path.join(root, filename))
    return files

def get_pdf_context(pdf_file_path):
    """
    Get the length of the content in a PDF file.
    """
    import fitz
    text = ""
    with fitz.open(pdf_file_path) as doc:
        for page in doc:
            text += page.get_text()
            
    return text

def get_csv_context(csv_file_path):
    """
    Get the length of the content in a CSV file.
    """
    context = ""
    with open(csv_file_path, 'r', encoding='utf-8', errors='ignore') as f:
        content = f.read()
        context += content
    return context

def find_not_exist_label_in_res(log_path, label_path):
    """_summary_

    Args:
        log_path (_type_): _description_
        label_path (_type_): _description_
        
    Example:
    
        log_path = "pipeline/eval_log/gpt-4-0125-preview_res_run.json"
        data_eval = analyze_result(log_path)
        label_path = "pipeline/label.json"
        find_not_exist_label_in_res(log_path, label_path)
    
    """
    
    with open(log_path, 'r') as f:
        res = json.load(f)
    with open(label_path, 'r') as f:
        labels = json.load(f)
        
    label_id = []
    res_id = []
    for hint in labels:
        for scene in labels[hint]:
            for item in labels[hint][scene]:
                label_id.append(f"{item['id']}-{hint}")
    
    for scene in res:
        for id in res[scene]:
            for i in range(-1,3):
                res_id.append(f"{id}-{i}")
            
    label_id = set(label_id)
    res_id = set(res_id)
    
    print(len(label_id))
    print(len(res_id))
    
    print(label_id.difference(res_id))

def calculate_token_len(sentence, model="gpt-4-0125-preview"):
    encoding = tiktoken.encoding_for_model(model)
    tokens = encoding.encode(sentence, disallowed_special=()) # we allow special tokens like <|start_of_text|>, since we are counting the token length for the prompt
    token_len = len(tokens)
    return token_len

def gen_uid():
    return str(uuid.uuid4())


def generate_hints(content, hint_writer_prompt_path = "prompts/gen_hints.txt", model="gpt-4-0125-preview"):
    """
    Use LLM model to generate three levels hints 
    """
    content = str(content).replace(r'\n', '\n').replace('\\',"")
    client = OpenAI()
    with open (hint_writer_prompt_path, "r") as file:
        prompt = file.read()
    prompt = prompt.replace("<ISSUE>", content)
    response = client.chat.completions.create(
        model=model,
        messages=[
            {"role": "user", "content": prompt },
        ]
    )
    hints = response.choices[0].message.content
    hints = hints.replace("```json", "").replace("```", "")
    # parse string into dict
    hints = json.loads(hints)
    return [hints["general"], hints["medium"], hints["specific"]]


############# ANALYSIS START #############

def analyze_result_detailed(log, visualize = False):
    """
    This function is used to evaluate and visualize the results of the evaluation, from all the meta info in task_meta.json, including:
    - platform
    - hint level
    - file number
    - file content length
    - type
    - tags
    """
    from benchmark.api import BenchmarkManager
    manager = BenchmarkManager()
    task_meta = manager.get_task_meta()
    log = os.path.join(root_path, log)
    with open(log, 'r') as f:
        res = json.load(f)
        
    report = {
        "hint_level": {
            "-1": {"failed":0, "success":0, "partially":0},
            "0": {"failed":0, "success":0, "partially":0},
            "1": {"failed":0, "success":0, "partially":0},
            "2": {"failed":0, "success":0, "partially":0},
        },
        "platform": {
            "BigBench": {"failed":0, "success":0, "partially":0},
            "Kaggle": {"failed":0, "success":0, "partially":0},
            "open-data-registry": {"failed":0, "success":0, "partially":0},
            "TensorFlow": {"failed":0, "success":0, "partially":0},
            "FiveThirtyEight": {"failed":0, "success":0, "partially":0},
            "HuggingFace": {"failed":0, "success":0, "partially":0},
            "OpenML": {"failed":0, "success":0, "partially":0},
            "GLI": {"failed":0, "success":0, "partially":0},
        },
        "type":{
            "single-issue-single-file": {"failed":0, "success":0, "partially":0},
            "single-issue-multi-file": {"failed":0, "success":0, "partially":0},
            "multi-issue-single-file": {"failed":0, "success":0, "partially":0},
            "multi-issue-multi-file": {"failed":0, "success":0, "partially":0},
        },
        "tags": {
            "typo": {"failed":0, "success":0, "partially":0},
            "wrong-format": {"failed":0, "success":0, "partially":0},
            "inappropriate-file": {"failed":0, "success":0, "partially":0},
            "ethical-legal-risk": {"failed":0, "success":0, "partially":0},
            "internal-discrepancy": {"failed":0, "success":0, "partially":0},
            "cross-file-discrepancy": {"failed":0, "success":0, "partially":0},
            "data-problem": {"failed":0, "success":0, "partially":0},
            "wrong-value": {"failed":0, "success":0, "partially":0},
            "missing-value": {"failed":0, "success":0, "partially":0},
            "data-leakage": {"failed":0, "success":0, "partially":0},
            "apparent-corruption": {"failed":0, "success":0, "partially":0},
            "hidden-corruption": {"failed":0, "success":0, "partially":0},
            "document-problem": {"failed":0, "success":0, "partially":0},
            "wrong-info": {"failed":0, "success":0, "partially":0},
            "insufficient-info": {"failed":0, "success":0, "partially":0},
            "infrastructure-problem": {"failed":0, "success":0, "partially":0},
            "data-access": {"failed":0, "success":0, "partially":0},
            "script-code": {"failed":0, "success":0, "partially":0},
        },
        "file_number": [],
        "content_length": [],
    }
    
    for scene in res:
        for id in res[scene]:
            # get the meta info
            meta = {}
            for item in task_meta:
                if item["id"] == id:
                    meta = item
                    break
            type = meta["type"][0] + "-" + meta["type"][1]
            tag_list = []
            for tag in meta["tags"]:
                tag_tmp_list = tag.split("/")
                tag_list.extend(tag_tmp_list)
            file_number = meta["file_number"]
            content_length = meta["file_content_length"]

            for level in res[scene][id]:
                level = str(level)
                if res[scene][id][level] == "failed":
                    report["hint_level"][level]["failed"] += 1
                    report["platform"][scene]["failed"] += 1
                    report["type"][type]["failed"] += 1
                    for tag in tag_list:
                        report["tags"][tag]["failed"] += 1
                    report["file_number"].append((file_number, -1))
                    report["content_length"].append((content_length, -1))
                elif res[scene][id][level] == "success":
                    report["hint_level"][level]["success"] += 1
                    report["platform"][scene]["success"] += 1
                    report["type"][type]["success"] += 1
                    for tag in tag_list:
                        report["tags"][tag]["success"] += 1
                    report["file_number"].append((file_number, 1))
                    report["content_length"].append((content_length, 1))
                elif res[scene][id][level] == "partially":
                    report["hint_level"][level]["success"] += 1
                    report["platform"][scene]["success"] += 1
                    report["type"][type]["success"] += 1
                    for tag in tag_list:
                        report["tags"][tag]["success"] += 1
                    report["file_number"].append((file_number, 1))
                    report["content_length"].append((content_length, 1))
                else:
                    raise ValueError("Invalid value in the result")
    
    tags_result = report["tags"]
    for tag in tags_result:
        report["tags"][tag]["success_rate"] = calculate_success_rate(tags_result[tag])
    
    with open("report.json", 'w') as f:
        json.dump(report, f)
    
    # begin visualization
    # for scene, type, level, we need to calculate the success rate, and draw the bar chart
    # for tags, since we have tag that has subtag, we need to calculate the success rate for each tag, and draw the bar chart, and annotate the subtag with its parent tag in axis
    # for file number and content length, we need to draw the histogram

    if visualize:
        visualize_report(report)

def calculate_success_rate(data):
    total = sum(data.values())
    return data["success"] / total if total > 0 else 0
    # return total * total / data["success"]  / 364

def visualize_report(report, dir="visualization"):
    os.makedirs(dir, exist_ok=True)
    
    # Visualize hint level, platform, and type
    for category in ['hint_level', 'platform', 'type']:
        data = {k: calculate_success_rate(v) for k, v in report[category].items()}
        categories = list(data.keys())
        success_rates = list(data.values())
        
        plt.figure(figsize=(12, 6))
        bars = plt.bar(categories, success_rates)
        plt.title(f'Success Rate by {category.capitalize()}')
        plt.xlabel(category.capitalize())
        plt.ylabel('Success Rate')
        # plt.ylim(0, 1)  # Set y-axis limit from 0 to 1
        
        # Add text labels on top of each bar
        for bar in bars:
            height = bar.get_height()
            plt.text(bar.get_x() + bar.get_width()/2., height,
                     f'{height:.2f}',
                     ha='center', va='bottom')
        
        plt.tight_layout()
        plt.savefig(os.path.join(dir, f'{category}_success_rate.pdf'))
        plt.close()

    # Visualize main tags
    main_tags = [
        "typo", "wrong-format", "inappropriate-file", "ethical-legal-risk",
        "internal-discrepancy", "cross-file-discrepancy", "data-problem",
        "document-problem", "infrastructure-problem"
    ]
    
    main_tag_data = {tag: calculate_success_rate(report['tags'][tag]) for tag in main_tags}
    tags = list(main_tag_data.keys())
    success_rates = list(main_tag_data.values())

    plt.figure(figsize=(12, 6))
    bars = plt.bar(tags, success_rates)
    plt.title('Success Rate by Main Tags')
    plt.xlabel('Main Tags')
    plt.ylabel('Success Rate')
    plt.ylim(0, 1)
    plt.xticks(rotation=45, ha='right')
    
    for bar in bars:
        height = bar.get_height()
        plt.text(bar.get_x() + bar.get_width()/2., height,
                 f'{height:.2f}',
                 ha='center', va='bottom')
    
    plt.tight_layout()
    plt.savefig(os.path.join(dir, 'main_tags_success_rate.pdf'))
    plt.close()

    # Visualize sub tags
    sub_tags = [
        "wrong-value", "missing-value", "data-leakage", "apparent-corruption",
        "hidden-corruption", "wrong-info", "insufficient-info", "data-access", "script-code"
    ]
    
    sub_tag_data = {tag: calculate_success_rate(report['tags'][tag]) for tag in sub_tags}
    tags = list(sub_tag_data.keys())
    success_rates = list(sub_tag_data.values())

    plt.figure(figsize=(12, 6))
    bars = plt.bar(tags, success_rates)
    plt.title('Success Rate by Sub Tags')
    plt.xlabel('Sub Tags')
    plt.ylabel('Suceess Rate')
    plt.ylim(0, 1)
    plt.xticks(rotation=45, ha='right')
    
    for bar in bars:
        height = bar.get_height()
        plt.text(bar.get_x() + bar.get_width()/2., height,
                 f'{height:.2f}',
                 ha='center', va='bottom')
    
    plt.tight_layout()
    plt.savefig(os.path.join(dir, 'sub_tags_success_rate.pdf'))
    plt.close()

    def map2index(res):
        if res == -1:
            return 0
        elif res == 1:
            return 1
    # Visualize file number and content length
    for category in ['file_number', 'content_length']:
        data = np.array(report[category])
        plt.figure(figsize=(10, 6))




        if category == 'content_length':
            bins = np.logspace(np.log10(data[:, 0].min()), np.log10(data[:, 0].max()), 20)
            
            # Separate data by success and failed
            success_data = data[data[:, 1] == 1, 0]
            failed_data = data[data[:, 1] == -1, 0]
            
            # Calculate histogram counts for success and failed data
            success_counts, _ = np.histogram(success_data, bins=bins)
            failed_counts, _ = np.histogram(failed_data, bins=bins)
            
            # Get the bin centers for bar placement
            bin_centers = (bins[:-1] + bins[1:]) / 2
            
            # Calculate success rate
            total_counts = success_counts + failed_counts
            success_rate = np.where(total_counts != 0, success_counts / total_counts, 0)
            
            # Create a new figure with two y-axes
            fig, ax1 = plt.subplots(figsize=(12, 8))
            ax2 = ax1.twinx()
            
            # Plot bars for success data (upward)
            ax1.bar(bin_centers, success_counts, width=np.diff(bins), align='center', color='#2ecc71', alpha=0.7, label='Success')
            
            # Plot bars for failed data (downward)
            ax1.bar(bin_centers, -failed_counts, width=np.diff(bins), align='center', color='#e74c3c', alpha=0.7, label='Failed')

            # Draw a black line at y = 0
            ax1.axhline(0, color='black', linewidth=0.5)

            # Plot success rate line
            ax2.plot(bin_centers, success_rate, color='#3498db', linewidth=2, label='Success Rate', marker='o', markersize=6)

            # Set x-axis to log scale
            ax1.set_xscale('log')
            
            # Determine the y-axis limit for bar plot with extra space
            max_count = max(success_counts.max(), failed_counts.max())
            y_margin = max_count * 0.1  # 10% margin
            ax1.set_ylim(-max_count - y_margin, max_count + y_margin)
            
            # Customizing y-ticks to be symmetric without negative labels
            y_ticks = np.linspace(0, max_count, 5)
            ax1.set_yticks(list(-y_ticks[::-1]) + list(y_ticks)[1:])
            ax1.set_yticklabels([str(abs(int(i))) for i in list(-y_ticks[::-1]) + list(y_ticks)[1:]], fontsize=14)
            
            ax1.tick_params(axis='x', labelsize=13)

            # Set y-axis for success rate with margin
            ax2.set_ylim(-0.05, 1.05)  # 5% margin on top and bottom
            ax2.set_yticks(np.arange(0, 1.1, 0.2))
            ax2.set_yticklabels([f'{i:.0%}' for i in np.arange(0, 1.1, 0.2)], fontsize=13)

            # Title and labels with increased font sizes
            plt.title('Number of Success & Failure and Success Rate Versus Different Content Length', fontsize=16, pad=20)
            ax1.set_xlabel('Content Length (log scale)', fontsize=14, labelpad=10)
            ax1.set_ylabel('Count', fontsize=14, labelpad=10)
            ax2.set_ylabel('Success Rate', fontsize=14, labelpad=10)

            # Combine legends with adjusted location
            lines1, labels1 = ax1.get_legend_handles_labels()
            lines2, labels2 = ax2.get_legend_handles_labels()
            ax1.legend(lines1 + lines2, labels1 + labels2, fontsize=14, loc='upper right', bbox_to_anchor=(0.99, 0.99))

            # Add grid for better readability
            ax1.grid(True, linestyle='--', alpha=0.3)

            plt.tight_layout()
            plt.savefig(os.path.join(dir, f'{category}_success_rate.pdf'))
            plt.show()


        elif category == 'file_number':
            # Calculate success rate for each file number
            unique_files = np.unique(data[:, 0])
            success_rates = [np.mean(data[data[:, 0] == fnum, 1] == 1) for fnum in unique_files]
            
            # Plotting bar chart for success rates with annotations
            bars = plt.bar(unique_files, success_rates, color='#1f77b4')  # Aesthetic blue color
            plt.title('Success Rate by File Number')
            plt.xlabel('File Number')
            plt.ylabel('Success Rate')

            # Adding text labels on top of each bar
            for bar in bars:
                height = bar.get_height()
                plt.text(bar.get_x() + bar.get_width() / 2, height,
                         f'{height:.2f}',
                         ha='center', va='bottom')

        plt.tight_layout()
        plt.savefig(os.path.join(dir, f'{category}_success_rate.pdf'))
        plt.close()
        
def analyze_result(log, visualize = False, label_path = None):
    report = {}
    log = os.path.join(root_path, log)
    if isinstance(log, str):
        with open(log, 'r') as f:
            res = json.load(f)
    else:
        res = log
        
    if label_path:
        labels = load_human_labels(label_path)
    else:
        labels = set()

    for scene in res:
        report[scene] = {}
        for i in range(-1,3):
            report[scene][str(i)] = {"failed":0, "success":0, "partially":0}
        for id in res[scene]:
            for level in res[scene][id]:
                level = str(level)
                if len(labels) > 0 and (id, level) not in labels:
                    continue
                if res[scene][id][level] == "failed":
                    report[scene][level]["failed"] += 1
                elif res[scene][id][level] == "success":
                    report[scene][level]["success"] += 1
                elif res[scene][id][level] == "partially":
                    report[scene][level]["partially"] += 1
                else:
                    raise ValueError("Invalid value in the result")
                    
    print_colored("Analysis of the result:", 'green')
    print("=====================================")
    res_hint = {str(i):{} for i in range(-1,3)}
    for scene in report:
        total = len(res[scene])
        for i in range(-1,3):
            failed = report[scene][str(i)]["failed"]
            partially =  report[scene][str(i)]["partially"]
            success = report[scene][str(i)]["success"]
            res_hint[str(i)]["failed"] = res_hint[str(i)].get("failed", 0) + failed
            res_hint[str(i)]["partially"] = res_hint[str(i)].get("partially", 0) + partially
            res_hint[str(i)]["success"] = res_hint[str(i)].get("success", 0) + success
            # print_colored(f"In {scene}, with hint level {i} | failed: {failed / total: .3f}, success: {success / total : .3f}, partially: {partially / total : .3f}, soft success: {(success + partially) / total } total: {total}", 'yellow')

    # print("=====================================")
    print_colored("summary by hint level:", 'green')
    for i in range(-1,3):
        total = res_hint[str(i)]["failed"] + res_hint[str(i)]["partially"] + res_hint[str(i)]["success"]
        print_colored(f"Hint Level: {i} | failed: {res_hint[str(i)]['failed'] / total: .3f}, partially: {res_hint[str(i)]['partially'] / total : .3f},  success: {res_hint[str(i)]['success'] / total : .3f}, total: {total}", 'yellow')
    
    print_colored("total summary:", 'green')
    total = 0
    total_success = 0
    total_strict_success = 0
    for i in range(-1,3):
        total += res_hint[str(i)]["failed"] + res_hint[str(i)]["partially"] + res_hint[str(i)]["success"]
        total_success += res_hint[str(i)]["success"] + res_hint[str(i)]["partially"]
        total_strict_success = res_hint[str(i)]["success"]
    
    print_colored(f"Total: {total}, Success: {total_success}, Success Rate: {total_success / total: .3f}, Strict Success Rate: {total_strict_success / total: .3f}", 'yellow')
    print_colored("===================================== END \n", 'green')
    
    succ_rate = total_success / total
    if visualize:
        scenes = list(report.keys())
        hint_levels = [str(i) for i in range(-1, 3)]
        
        # # fig 1
        # fig, ax = plt.subplots(figsize=(len(scenes) * 2, 6))

        # bar_width = 0.1
        # x = np.arange(len(scenes))

        # for i, level in enumerate(hint_levels):
        #     success_rates = [
        #         (report[scene][level]["success"] + report[scene][level]["partially"]) / len(res[scene]) # the success rate is calculated within a scene
        #         for scene in scenes
        #     ]
        #     ax.bar(x + i * bar_width, success_rates, width=bar_width, label=f"Hint Level {level}")

        # ax.set_xticks(x + (len(hint_levels) - 1) * bar_width / 2)
        # ax.set_xticklabels(scenes, rotation=45, ha='right')
        # # Remove x-axis label
        # ax.set_xlabel('')
        # ax.set_ylabel('Success Rate')
        # ax.legend()

        # # Remove title
        # plt.title('')

        # # Remove top and right border
        # ax.spines['top'].set_visible(False)
        # ax.spines['right'].set_visible(False)

        # # Add thicker border to the plot
        # ax.spines['left'].set_linewidth(1.5)
        # ax.spines['bottom'].set_linewidth(1.5)

        # # Add grey gridlines behind the bars
        # ax.yaxis.grid(True, color='grey', linestyle='-', linewidth=0.5)
        # ax.set_axisbelow(True)

        # plt.tight_layout()
        
        # fig 2
        
        success_rates = []
        for level in hint_levels:
            total_success = res_hint[level]["success"] + res_hint[level]["partially"]
            total_count = res_hint[level]["success"] + res_hint[level]["failed"] + res_hint[level]["partially"]
            success_rate = total_success / total_count
            success_rates.append(success_rate)

        fig, ax = plt.subplots(figsize=(5, 3))

        x = np.arange(len(hint_levels))
        bar_width = 0.2

        bars = ax.bar(x, success_rates, width=bar_width)

        for bar in bars:
            height = bar.get_height()
            ax.text(bar.get_x() + bar.get_width() / 2, height,
                    f'{height:.2%}', ha='center', va='bottom', size=10)

        ax.set_xticks(x)
        from matplotlib import font_manager

        bold_font = font_manager.FontProperties(weight=545)
        ax.set_xticklabels([f"Hint Level {i}" for i in range(len(hint_levels))], fontproperties=bold_font)
        ax.tick_params(axis='x', labelsize=10)
        ax.tick_params(axis='y', labelsize=12, width=2, length=0, pad=10)

        # Remove title
        plt.title('')

        # Remove top and right border
        ax.spines['top'].set_visible(False)
        ax.spines['right'].set_visible(False)

        # Add thicker border to the plot
        ax.spines['left'].set_linewidth(1.5)
        ax.spines['bottom'].set_linewidth(1.5)

        # Add grey gridlines behind the bars
        ax.yaxis.grid(True, color='gray', linestyle='-', linewidth=0.5, alpha=0.7)
        ax.set_axisbelow(True)

        plt.tight_layout()
        plt.savefig("success_rate.pdf")
        plt.show()

    return res_hint, succ_rate
    
def analyze_result_with_human_label(log_path, label_path, verbose = False):
    """
    compare the evaluation results by evaluator with human labels

    Args:
        log_path (_type_): _description_
        label_path (_type_): _description_
    """    
    print("log_path: ", log_path)
    matches = re.findall(r'(eval|test)=([^/]+)', log_path)

    result = {}
    for match in matches:
        key, value = match
        result[key] = value

    eval_model = result.get('eval')
    test_model = result.get('test')

    log_path = os.path.join(root_path, log_path)
    label_path = os.path.join(root_path, label_path)
    with open(log_path, 'r') as f:
        res = json.load(f)
    with open(label_path, 'r') as f:
        labels = json.load(f)
        
    report = {}

    for scene in res:
        report[scene] = {}
        for i in range(-1,3):
            report[scene][str(i)] = {"failed":0, "success":0, "partially":0}
        for id in res[scene]:
            for level in res[scene][id]:
                level = str(level)
                if find_id_in_label(labels, scene, id, level):
                    if res[scene][id][level] == "failed":
                        report[scene][level]["failed"] += 1
                    elif res[scene][id][level] == "success":
                        report[scene][level]["success"] += 1
                    else:
                        report[scene][level]["partially"] += 1
    
    res_hint = {str(i):{} for i in range(-1,3)}
    for scene in report:
        total = len(res[scene])
        for i in range(-1,3):
            failed = report[scene][str(i)]["failed"]
            partially =  report[scene][str(i)]["partially"]
            success = report[scene][str(i)]["success"]
            res_hint[str(i)]["failed"] = res_hint[str(i)].get("failed", 0) + failed
            res_hint[str(i)]["partially"] = res_hint[str(i)].get("partially", 0) + partially
            res_hint[str(i)]["success"] = res_hint[str(i)].get("success", 0) + success

    correct = 0
    tp = 0 # we define failed as positive
    fp = 0
    fn = 0
    total = 0
    soft_correct = 0
    
    # cal kapa
    soft_kapa_matrix = np.zeros((2,2)) # fail, success
    strict_kapa_matrix = np.zeros((3,3)) # fail, success, partially
    
    strict_kapa_map = {"failed":0, "success":1, "partially":2}
    soft_kapa_map = {"failed":0, "success":1, "partially":1}
    
    
    wrong = {}
    
    # we deem success as postive
    for scene in res:
        for id in res[scene]:
            if len(res[scene][id].keys()) > 0:
                for level in res[scene][id]:
                    human_label = find_id_in_label(labels, scene, id, level, bool=False)
                    if human_label == None:
                        continue
                    answer = res[scene][id][level]
                    
                    soft_kapa_matrix[soft_kapa_map[human_label], soft_kapa_map[answer]] += 1
                    strict_kapa_matrix[strict_kapa_map[human_label], strict_kapa_map[answer]] += 1
                    
                    if human_label == answer:
                        correct += 1
                    elif answer == "success" and human_label == "partially":
                        soft_correct += 1
                    elif answer == "partially" and human_label == "success":
                        soft_correct += 1
                    else:
                        if scene not in wrong:
                            wrong[scene] = []
                        wrong[scene].append({"id":id, "level":level, "answer":answer, "human_label":human_label})
                    
                    if human_label in ["success", "partially"] and answer in ["success", "partially"]:
                        tp += 1
                    if human_label in ["success", "partially"] and answer == "failed":
                        fn += 1
                    if human_label == "failed" and answer in ["success", "partially"]:
                        fp += 1
                    
                    total += 1
    print_colored("============ Analyze the result with human label:", 'green')
    print_colored(f"Eval Model: {eval_model}, Test Model: {test_model}", 'green')
    print_colored(f"Strict Accuracy: {correct / total}, Soft Accuracy: {(soft_correct + correct) / total}, Total: {total}", 'green')
    
    soft_kapa_po = div_by_zero(soft_kapa_matrix[0,0] + soft_kapa_matrix[1,1], np.sum(soft_kapa_matrix))
    soft_kapa_pe = div_by_zero((soft_kapa_matrix[0,0] + soft_kapa_matrix[0,1]) * (soft_kapa_matrix[0,0] + soft_kapa_matrix[1,0]) + (soft_kapa_matrix[1,0] + soft_kapa_matrix[1,1]) * (soft_kapa_matrix[0,1] + soft_kapa_matrix[1,1]), np.sum(soft_kapa_matrix) ** 2)
    
    soft_kapa = div_by_zero(soft_kapa_po - soft_kapa_pe, 1 - soft_kapa_pe)
    
    strict_kapa_po = div_by_zero(strict_kapa_matrix[0,0] + strict_kapa_matrix[1,1] + strict_kapa_matrix[2,2], np.sum(strict_kapa_matrix))
    strict_kapa_pe = div_by_zero((strict_kapa_matrix[0,0] + strict_kapa_matrix[0,1] + strict_kapa_matrix[0,2]) * (strict_kapa_matrix[0,0] + strict_kapa_matrix[1,0] + strict_kapa_matrix[2,0]) + (strict_kapa_matrix[1,0] + strict_kapa_matrix[1,1] + strict_kapa_matrix[1,2]) * (strict_kapa_matrix[0,1] + strict_kapa_matrix[1,1] + strict_kapa_matrix[2,1]) + (strict_kapa_matrix[2,0] + strict_kapa_matrix[2,1] + strict_kapa_matrix[2,2]) * (strict_kapa_matrix[0,2] + strict_kapa_matrix[1,2] + strict_kapa_matrix[2,2]), np.sum(strict_kapa_matrix) ** 2)
    
    strict_kapa = div_by_zero(strict_kapa_po - strict_kapa_pe, 1 - strict_kapa_pe)
    
    print_colored(f"Strict Kappa: {strict_kapa}, Soft Kappa: {soft_kapa}", 'green')
    
    # calculate precision, recall, F1 Score for correct and soft correct seperately
    precision = div_by_zero(tp, tp + fp)
    recall = div_by_zero(tp, tp + fn)
    fpr = div_by_zero(fp, fp + tp)
    f1_score = div_by_zero(2 * precision * recall, precision + recall)
    print_colored(f"Precision: {precision}, Recall: {recall}, FPR: {fpr},  F1 Score: {f1_score}", 'green')
    
    if verbose:
        print_colored("Wrong List:", 'red')
        pprint(wrong)
    print_colored("============ End \n", 'green')
    
    res_dict = {
        "precision":precision,
        "acc": (soft_correct + correct) / total,
        "recall":recall,
        "fpr":fpr,
        "f1_score":f1_score,
        "strict_kapa":strict_kapa,
        "soft_kapa":soft_kapa,
    }
    
    return res_hint, res_dict

def filter_a_according_b(res1, res2):
    """
    filter res1 ids according to the results in res2

    Args:
        res1 (_type_): _description_
        res2 (_type_): _description_

    Returns:
        _type_: _description_
    """
    id_hints = []
    for platform in res2:
        for id, hints in res2[platform].items():
            for hint in hints:
                id_hints.append((id, hint))
    
    res1_ = {}
    for platform in res1:
        res1_[platform] = {}
        for id,hints in res1[platform].items():
            for hint in hints:
                if (id, hint) in id_hints:
                    res1_[platform][id] = res1_[platform].get(id, {})
                    res1_[platform][id][hint] = res1[platform][id][hint]
    return res1_
    

def analyze_human_label(label_path):
    res = {}
    for hint_level in ["-1", "0", "1", "2"]:
        res[hint_level] = {}
        label_path = os.path.join(root_path, label_path)
        with open(label_path, 'r') as f:
            labels = json.load(f)
        total = 0
        for scene in labels[hint_level]:
            for item in labels[hint_level][scene]:
                res[hint_level]["failed"] = res[hint_level].get("failed", 0)
                res[hint_level]["partially"] = res[hint_level].get("partially", 0)
                res[hint_level]["success"] = res[hint_level].get("success", 0)
                if item["human_label"] == "failed":
                    res[hint_level]["failed"] = res[hint_level].get("failed", 0) + 1
                elif item["human_label"] == "partially":
                    res[hint_level]["partially"] = res[hint_level].get("partially", 0) + 1
                else:
                    res[hint_level]["success"] = res[hint_level].get("success", 0) + 1
                total += 1
        res[hint_level]["total"] = total
            
    print_colored("============ Analyze the human label:", 'green')           
    for hint_level in res:
        print_colored(f"Hint Level: {hint_level} | failed: {res[hint_level].get('failed', 0) / res[hint_level].get('total',0) : 3f}, partially: {res[hint_level].get('partially', 0) / res[hint_level].get('total',0) : 3f}, success: {res[hint_level].get('success', 0) / res[hint_level].get('total',0) : 3f}", 'yellow')
    print_colored("============ End \n", 'green')
    return res
           
############# ANALYSIS END #############  
    
def get_shuffle_hint_example(shuff_dict):
    from benchmark.api import BenchmarkManager
    import numpy as np
    seed = 0
    np.random.seed(seed)
    manager = BenchmarkManager()
    hint_example = {"-1":{}, "0":{}, "1":{}, "2":{}}
    for scene in shuff_dict:
        number = shuff_dict[scene]
        task_meta = manager.get_scenario_task_meta(scene)
        for i in range(-1,3):
            hint_example[str(i)][scene] = []
            samples = [sample["id"] for sample in (np.random.choice(task_meta, number, replace=False))]
            hint_example[str(i)][scene].extend(samples)
    return hint_example

def find_id_in_label(labels, scenario, id, hint_level, bool=True):
    data = labels[hint_level][scenario]
    if bool:
        for item in data:
            if item["id"] == id:
                return True
        return False
    else:
        for item in data:
            if item["id"] == id:
                return item["human_label"]
        return None

def get_all_failed_sample(log_path, number = 32, seed = 10):
    """
    get hardeset samples
    """
    with open(log_path, 'r') as f:
        res = json.load(f)
    failed_samples = []
    for scene in res:
        for id in res[scene]:
            for level in res[scene][id]:
                if res[scene][id][level] == "failed":
                    failed_samples.append((scene, id, level))
                    
    # randomly select samples
    import numpy as np
    # set seed
    np.random.seed(seed)
    sample_indx = np.random.choice(len(failed_samples), number, replace=False)
    failed_samples = np.array(failed_samples)[sample_indx].tolist()
    # write into json
    with open("failed_samples.json", 'w') as f:
        json.dump(failed_samples, f)
        
def is_non_text_file(file_name):
    non_text_file_extensions = ('.jpg', '.jpeg', '.png', '.gif', '.bmp', 
                                '.mp3', '.wav', '.ogg', '.flac', 
                                '.mp4', '.avi', '.mov', '.mkv')
    return file_name.endswith(non_text_file_extensions)



def process_other_information(row):
    """
    Process other information from a row in a CSV file.

    Args:
        row (dict): A dictionary containing the row data.

    Returns:
        tuple: A tuple containing the original issue URL, platform license, and dataset license.
    """
    orginal_issue_url = row["Issue URL"]
    platform_name = row["Source"]
    plarform_license = row["Platform License"]
    dataset_license = row["Dataset License"]
    if not dataset_license:
        dataset_license = "Not Specified"
    return orginal_issue_url, plarform_license, dataset_license, platform_name

def process_types(row):
    """
    Process the issue types from a row in a CSV file.

    Args:
        row (dict): A dictionary containing the row data.

    Returns:
        list: A list of issue types.
    """
    types = []
    if row["Issue Number"] == "single-issue":
        types.append("single-issue")
    elif row["Issue Number"] == "multi-issue":
        types.append("multi-issue")
    else:
        raise ValueError(f"Unknown issue number: {row['Issue Number']}")
    
    if row["File Number"] == "single-file":
        types.append("single-file")
    elif row["File Number"] == "multi-file":
        types.append("multi-file")
    else:
        raise ValueError(f"Unknown file number: {row['File Number']}")
        
    return types
        

def process_tags(row):
    """
    Process the issue tags from a row in a CSV file.

    Args:
        row (dict): A dictionary containing the row data.

    Returns:
        list: A list of issue tags.
    """
    tags = []
    
    # Process simple tags
    simple_tags = ['typo', 'wrong-format', 'inappropriate-file', 'ethical/legal-risk', 'cross-file-discrepancy', 'internal-discrepancy', 'data-problem', 'document-problem', 'infrastructure-problem']
    for tag in simple_tags:
        
        if row[tag]:
            if tag == "ethical/legal-risk":
                tag = "ethical-legal-risk"
            tags.append(tag)
    
    # Process data-problem tags
    data_problem_tags = ['wrong-value', 'missing-value', 'data-leakage', 'apparent-corruption', 'hidden-corruption']
    for tag in data_problem_tags:
        if row[tag]:
            tags.append(f'data-problem/{tag}')
            if 'data-problem' in tags:
                tags.remove('data-problem')
    
    # Process document-problem tags
    document_problem_tags = ['wrong-info', 'insufficient-info']
    for tag in document_problem_tags:
        if row[tag]:
            tags.append(f'document-problem/{tag}')
            if 'document-problem' in tags:
                tags.remove('document-problem')
    
    # Process infrastructure-problem tags
    infrastructure_problem_tags = ['data-access', 'script-code']
    for tag in infrastructure_problem_tags:
        if row[tag]:
            tags.append(f'infrastructure-problem/{tag}')
            if 'infrastructure-problem' in tags:
                tags.remove('infrastructure-problem')
    
    return tags



def get_length_bias_analysis(test_model_id, test_stamp, scenarios):
    eval_res_path = f"pipeline/eval_log/eval=gpt-4-0125-preview/test={test_model_id}/{test_stamp}/res.json"
    if test_stamp:
        test_output_dir = f"pipeline/output/{test_model_id}/{test_stamp}"
    else:
        test_output_dir = f"pipeline/output/{test_model_id}"
    
    eval_res_path = os.path.join(root_path, eval_res_path)
    test_output_dir = os.path.join(root_path, test_output_dir)
    
    with open(eval_res_path, "r") as f:
        eval_res = json.load(f)
        
    eval_res_accross_answer_len = []
    total_file_cnt = 0
    total_tokens_cnt = 0
    for scenario in scenarios:
        test_output_scenario_dir = f"{test_output_dir}/{scenario}"
        # the output from the test model
        if not os.path.exists(test_output_scenario_dir):
            continue
        ids = [id for id in os.listdir(test_output_scenario_dir) if not id.startswith(".")]
        id_bar = ids
                    
        for id in id_bar:
            for hint_level in range(-1,3):
                test_model_output_path = f"{test_output_scenario_dir}/{id}/hint_level_{hint_level}/output.txt"

                if os.path.exists(test_model_output_path):
                    with open(test_model_output_path, "r") as f:
                        agent_answer = f.read()
                else:
                    continue
                
                total_file_cnt += 1
                ans_length = calculate_token_len(agent_answer)
                if id in eval_res[scenario]:
                    if isinstance(eval_res[scenario][id], dict):
                        if str(hint_level) in eval_res[scenario][id]:
                            eval_res_value = eval_res[scenario][id][str(hint_level)]
                            eval_res_accross_answer_len.append((eval_res_value, ans_length))
                    else:
                        print(f"eval_res[scenario][id] is not a dict, it's a {type(eval_res[scenario][id])}")

                total_tokens_cnt += ans_length
                
    data = np.array([(len, 1 if res == "success" or res == "partially" else -1)
                    for res, len in eval_res_accross_answer_len], dtype=[('length', np.int32), ('result', np.int32)])

    success_rate = len(data[data['result'] == 1]) / len(data)
    print("test_model_id: ", test_model_id)
    print("test_stamp: ",test_stamp)
    print("=== Length Bias Analysis ===")
    print("average token len", total_tokens_cnt/total_file_cnt)
    print("success_rate", success_rate)
    print("====== \n \n")
    
