import requests
from tqdm import tqdm
import multiprocessing as mp
import csv
from functools import partial
import traceback
import sys
import time

DISCUSSION_THRESHOLD = 3
SIZE_LIMIT = 512
WANTED_MODALITY = ["modality:text", "modality:tabular"]
WANTED_SIZE = ["size_categories:10K<n<100K", "size_categories:1K<n<10K"]
MAX_RETRIES = 3
RETRY_DELAY = 1  # seconds

def valid_modality(tags):
    return any(modality in tags for modality in WANTED_MODALITY)

def valid_size(tags):
    return any(size in tags for size in WANTED_SIZE)

def turnbyte2mb(byte):
    return byte / 1024 / 1024

BOT_ACTION = ["Convert dataset to Parquet", "[bot] Conversion to Parquet", "Convert dataset sizes from base 2 to base 10 in the dataset card", "Support streaming", "Reorder split names", "Replace YAML keys from int to str", "Deprecate dataset", "Delete trailing empty example", "Dataset Viewer issue", "Replace YAML keys from int to str", "Convert dataset to Parquet", "Delete legacy JSON metadata"]

raw_url = "https://huggingface.co/datasets"

API_URL = "https://huggingface.co/api/datasets"

def extract_valid_discuss(response):
    cnt = 0
    for item in response.get("discussions", []):
        isHF = item.get("author", {}).get("isHf", False)
        title = item.get("title", "")
        
        if title in BOT_ACTION or isHF:
            continue
        cnt += 1
    return cnt

def calculate_size(dataset_info):
    if isinstance(dataset_info, dict):
        try:
            bytes = dataset_info["download_size"]
            return turnbyte2mb(bytes)
        except Exception:
            raise ValueError("Missing download size information")
    elif isinstance(dataset_info, list):
        return sum(calculate_size(item) for item in dataset_info)

def process_dataset(dataset_info):
    try:
        tags = dataset_info.get("tags", [])
        if not (valid_modality(tags) and valid_size(tags)):
            return None

        id = dataset_info.get("id")
        if not id:
            return None

        detailed_url = f"{API_URL}/{id}"

        detailed_dataset_info = make_request(detailed_url).json()
        if not detailed_dataset_info:
            return None

        try:
            size = calculate_size(detailed_dataset_info["cardData"]["dataset_info"])
        except Exception:
            print(detailed_dataset_info["cardData"]["dataset_info"])
            raise ValueError("Missing download size information")

        if size > SIZE_LIMIT:
            return None

        discussion_url = f"{API_URL}/{id}/discussions"
        try:
            discussion_response = make_request(discussion_url).json()
            if not discussion_response:
                return None
            counts = extract_valid_discuss(discussion_response)
            if counts < DISCUSSION_THRESHOLD:
                return None
        except Exception:
            raise ValueError("Missing discussion information")

        dataset_url = f"{raw_url}/{id}"
        return {
            "id": id,
            "url": dataset_url,
            "discussions_num": counts,
            "size": size
        }
    except Exception as e:
        error_info = sys.exc_info()
        error_location = traceback.extract_tb(error_info[2])[-1]
        print(f"Error at {error_location.filename}:{error_location.lineno}")
        print(f"Error processing dataset {dataset_info.get('id', 'Unknown')}: {str(e)}")
        return None

def make_request(url, params=None):
    for attempt in range(MAX_RETRIES):
        try:
            response = requests.get(url, params=params)
            response.raise_for_status()
            return response
        except requests.RequestException as e:
            if attempt == MAX_RETRIES - 1:
                print(f"Failed to fetch data from {url} after {MAX_RETRIES} attempts: {str(e)}")
                return None
            time.sleep(RETRY_DELAY)

def fetch_datasets(limit=100000, batch_size=1000):
    all_datasets = []
    params = {
        "full": "true",
        "limit": batch_size,
        "sort": "downloads",
        "direction": "-1",  # Sort in descending order
    }
    API_URL = "https://huggingface.co/api/datasets"
    with tqdm(total=limit, desc="Fetching datasets") as pbar:
        while len(all_datasets) < limit:
            response = make_request(API_URL, params)
            if not response:
                break
            
            page_data = response.json()
            all_datasets.extend(page_data)
            pbar.update(len(page_data))
            
            if len(page_data) < batch_size:
                break
            
            # Check for next page in Link header
            next_link = response.links.get('next', {}).get('url')
            if not next_link:
                break
            
            params = {}  # Clear params as they are included in the next URL
            API_URL = next_link
    
    return all_datasets[:limit]

def main():
    all_datasets = fetch_datasets()
    
    process_pbar = tqdm(total=len(all_datasets), desc="Processing datasets")
    
    with mp.Pool(processes=mp.cpu_count()) as pool:
        results = []
        for result in pool.imap_unordered(process_dataset, all_datasets):
            if result:
                results.append(result)
            process_pbar.update(1)
    
    process_pbar.close()
    
    with open("datasets.csv", "w", newline='') as f:
        writer = csv.DictWriter(f, fieldnames=["id", "url", "discussions_num", "size"])
        writer.writeheader()
        for result in results:
            writer.writerow(result)
    
    total_processed = len(all_datasets)
    total_filtered = len(results)
    
    print(f"\nTotal datasets processed: {total_processed}")
    print(f"Datasets matching criteria: {total_filtered}")
    print(f"Results written to datasets.csv")

if __name__ == "__main__":
    mp.freeze_support()
    main()