import requests
from kaggle.api.kaggle_api_extended import KaggleApi
import json
from tqdm import tqdm
import concurrent.futures
from bs4 import BeautifulSoup
import time
import csv

DISCUSSION_THRESHOLD = 5
SIZE_LIMIT = 512  # MB
MAX_RETRIES = 3
BATCH_SIZE = 1000  # Number of results to write in each batch

def get_discussion_count(dataset_ref):
    url = f"https://www.kaggle.com/datasets/{dataset_ref}"
    for _ in range(MAX_RETRIES):
        try:
            response = requests.get(url)
            if response.status_code == 200:
                soup = BeautifulSoup(response.text, 'html.parser')
                script_tag = soup.find('script', type='application/ld+json')
                if script_tag:
                    json_data = json.loads(script_tag.string)
                    interaction_statistics = json_data.get('interactionStatistic', [])
                    for stat in interaction_statistics:
                        if stat.get('interactionType') == 'http://schema.org/CommentAction':
                            return stat.get('userInteractionCount', 0)
            time.sleep(1)  # Wait before retrying
        except Exception as e:
            print(f"Error getting discussion count for {dataset_ref}: {str(e)}")
            time.sleep(1)  # Wait before retrying
    return 0

def conver2mb(byte):
    return byte / 1024 / 1024

def process_dataset(dataset):
    api = KaggleApi()
    api.authenticate()
    
    for _ in range(MAX_RETRIES):
        try:
            url = f"https://www.kaggle.com/api/v1/datasets/list/{dataset['ref']}"
            response = requests.get(url)
            if response.status_code != 200:
                print(f"Error: Status code {response.status_code} for dataset {dataset['ref']}")
                time.sleep(1)
                continue

            dataset_info = response.json()
            if not isinstance(dataset_info, dict) or 'datasetFiles' not in dataset_info:
                print(f"Error: Unexpected data format for dataset {dataset['ref']}")
                time.sleep(1)
                continue

            files = dataset_info["datasetFiles"]
            total_size = sum(conver2mb(file["totalBytes"]) for file in files)

            if total_size > SIZE_LIMIT:
                return None
            
            discussion_count = get_discussion_count(dataset['ref'])
            if discussion_count >= DISCUSSION_THRESHOLD:
                return {
                    "id": dataset['ref'],
                    "url": f"https://www.kaggle.com/datasets/{dataset['ref']}",
                    "discussions_num": discussion_count,
                    "size_mb": total_size
                }
            return None  # Dataset doesn't meet criteria, but no error occurred
        except Exception as e:
            print(f"Error processing dataset {dataset['ref']}: {str(e)}")
            time.sleep(1)
    
    print(f"Failed to process dataset {dataset['ref']} after {MAX_RETRIES} attempts")
    return None

def write_batch_to_csv(results, batch_number):
    filename = f"kaggle_datasets_batch_{batch_number}.csv"
    with open(filename, "w", newline='') as f:
        writer = csv.DictWriter(f, fieldnames=["id", "url", "discussions_num", "size_mb"])
        writer.writeheader()
        for result in results:
            writer.writerow(result)
    print(f"Batch {batch_number} written to {filename}")

def main():
    api = KaggleApi()
    api.authenticate()
    
    all_datasets = []
    page = 1
    MAX_DATASETS = 500000
    with tqdm(total=MAX_DATASETS, desc="Fetching datasets") as pbar:
        while len(all_datasets) < MAX_DATASETS:
            datasets = list(api.datasets_list(sort_by='hottest', page=page, max_size=SIZE_LIMIT * 1024 * 1024))
            if not datasets:
                break  # Exit loop if no more datasets
            all_datasets.extend(datasets)
            pbar.update(len(datasets))
            page += 1
    
    results = []
    batch_number = 1
    with tqdm(total=len(all_datasets), desc="Processing datasets") as pbar:
        with concurrent.futures.ThreadPoolExecutor(max_workers=10) as executor:
            future_to_dataset = {executor.submit(process_dataset, dataset): dataset for dataset in all_datasets}
            for future in concurrent.futures.as_completed(future_to_dataset):
                result = future.result()
                if result:
                    results.append(result)
                    if len(results) >= BATCH_SIZE:
                        write_batch_to_csv(results, batch_number)
                        results = []
                        batch_number += 1
                pbar.update(1)
    
    # Write any remaining results
    if results:
        write_batch_to_csv(results, batch_number)
    
    print(f"\nTotal datasets processed: {len(all_datasets)}")
    print(f"Datasets matching criteria: {sum(batch_number - 1 for _ in range(batch_number - 1)) + len(results)}")
    print(f"Results written to CSV files: kaggle_datasets_batch_1.csv to kaggle_datasets_batch_{batch_number}.csv")

if __name__ == "__main__":
    main()