import argparse
import os.path
from argparse import ArgumentParser
from typing import Any

import numpy as np
from imageio import imwrite


def load_from_folder(folder: str, num_samples: int, start_index: int = 0) -> np.ndarray:
    return np.array([np.load(f'{folder}/{i + start_index}.npy') for i in range(num_samples)])


def run(
        image_folders: list[str],
        save_path: str,
        num_samples: int = 50,
        start_index: list[int] = None
) -> None:
    if start_index is None:
        start_index: list[int] = [0 for _ in range(len(image_folders))]

    samples: np.ndarray = np.array([
        load_from_folder(image_folders[i], num_samples, start_index[i]) for i in range(len(image_folders))])

    print(samples.shape)
    _, _, channels, height, width = samples.shape
    grid: np.ndarray = np.zeros((channels, height * num_samples, width * len(image_folders)))
    for i in range(len(image_folders)):
        for j in range(num_samples):
            grid[:, j * height:(j + 1) * height, i * width:(i + 1) * width] = samples[i, j]

    grid_converted: np.ndarray = (np.clip(np.transpose(
        grid * 0.5 + 0.5, (1, 2, 0)), 0, 1) * 255).round().astype(np.uint8)

    if os.path.dirname(save_path) != '':
        os.makedirs(os.path.dirname(save_path), exist_ok=True)
    imwrite(save_path, grid_converted)


def run_from_config(config: dict[str, Any]) -> None:
    run(**config)


def parse_args() -> argparse.Namespace:
    parser: ArgumentParser = ArgumentParser()
    parser.add_argument('--image_folders', type=str, nargs='+', required=True)
    parser.add_argument('--save_path', type=str, required=True)
    parser.add_argument('--num_samples', type=int, default=50)
    parser.add_argument('--start_index', type=int, nargs='+', default=None)
    return parser.parse_args()


def get_config_from_args(args: argparse.Namespace) -> dict[str, Any]:
    return vars(args)


def main() -> None:
    args: argparse.Namespace = parse_args()
    config: dict[str, Any] = get_config_from_args(args)
    run_from_config(config)


if __name__ == '__main__':
    main()
