import argparse
import os.path
import pickle
from typing import Any, Optional, Callable

import numpy as np
import torch.nn
from torch import inference_mode

from datasets.numpy import NumpyDataset, FolderNumpyDataset
from src.datasets.noise import NumpyNoiseDataset
from src.datasets.noise_label import NoiseLabelDataset, NumpyNoiseLabelDataset
from src.fid.utils import calculate_fid_for_dataset_and_model, RandomNoiseDataset, RandomLabelDataset
from src.models.models.edm import create_edm_model
from src.utils.edm import get_sigmas
from src.utils.load import load_from_state_dict
from src.utils.utils import create_one_hot_torch
from torch_utils.distributed.distributed_manager import DistributedManager
from torch_utils.stats import get_torch_stats
from torch_utils.utils import get_default_device
from utils.logger.logger import Logger
from utils.utils import get_class_name, get_object_name


@torch.inference_mode()
def multi_step_euler_sampler(
        num_steps: int,
        time_steps: list[int],
        net: torch.nn.Module,
        noises: torch.Tensor,
        class_labels: torch.Tensor = None,
        randn_like: Callable[[torch.Tensor], torch.Tensor] = torch.randn_like,
        sigma_min: float = 0.002,
        sigma_max: float = 80,
        rho: float = 7,
        s_churn: float = 0,
        s_min: float = 0,
        s_max: float = float('inf'),
        s_noise: float = 1,
        multiply_noises: bool = True,
        same_noise: bool = True,
        device: str = None
) -> torch.Tensor:
    Logger.debug(
        f'{get_class_name(multi_step_euler_sampler)} - '
        f'num_steps: {num_steps}, '
        f'time_steps: {time_steps}, '
        f'net: {get_object_name(net)}, '
        f'noises: {get_torch_stats(noises)}, '
        f'class_labels: {get_torch_stats(class_labels) if class_labels is not None else None}, '
        f'multiply_noises: {multiply_noises}, '
        f'sigma_min: {sigma_min}, '
        f'sigma_max: {sigma_max}, '
        f'rho: {rho}, '
        f's_churn: {s_churn}, '
        f's_min: {s_min}, '
        f's_max: {s_max}, '
        f's_noise: {s_noise}, '
        f'same_noise: {same_noise}, '
        f'device: {device}'
    )
    assert multiply_noises or not same_noise, 'same_noise is only relevant when multiply_noises is True'
    assert time_steps[0] == 0, 'the first time step must be 0'
    assert time_steps[-1] == num_steps, 'the last time step must be the number of steps'

    if device is None:
        device: str = get_default_device()

    sigmas: torch.Tensor = torch.from_numpy(get_sigmas(
        num_steps=num_steps,
        sigma_min=sigma_min,
        sigma_max=sigma_max,
        rho=rho
    ))

    batch_size: int = noises.shape[0]

    x_next: torch.Tensor = noises.to(torch.float64) * sigmas[0] if multiply_noises else noises.to(torch.float64)
    for i in range(len(time_steps) - 1):
        t_cur = sigmas[time_steps[i]].to(device)
        t_next = sigmas[time_steps[i + 1]].to(device)

        x_cur: torch.Tensor = x_next

        gamma: float = min(s_churn / num_steps, np.sqrt(2) - 1) if s_min <= t_cur <= s_max else 0
        t_hat: torch.Tensor = torch.as_tensor(t_cur + gamma * t_cur)
        x_hat: torch.Tensor = x_cur + torch.sqrt(t_hat ** 2 - t_cur ** 2) * s_noise * randn_like(x_cur)

        if torch.max(torch.abs(x_hat - x_cur)) != 0:
            Logger.warning('warning: sampling is not deterministic.')

        denoised: torch.Tensor = net(
            x_hat,
            torch.ones(batch_size).to(device) * t_hat,
            class_labels
        ).to(torch.float64)

        d_cur: torch.Tensor = (x_hat - denoised) / t_hat
        x_next: torch.Tensor = x_hat + (t_next - t_hat) * d_cur

    return x_next


@inference_mode()
def inference_multi_step_euler(
        model: torch.nn.Module,
        num_steps: int,
        time_steps: list[int],
        noises: torch.Tensor,
        labels: torch.Tensor = None,
        num_classes: Optional[int] = None,
        sigma_min: float = 0.002,
        sigma_max: float = 80,
        rho: float = 7,
        s_churn: float = 0,
        s_min: float = 0,
        s_max: float = float('inf'),
        s_noise: float = 1,
        multiply_noises: bool = True,
        same_noise: bool = True,
        device: str = None
) -> torch.Tensor:
    assert labels is None or num_classes is not None, 'labels must be None or num_classes must not be None'
    if device is None:
        device: str = get_default_device()
    return multi_step_euler_sampler(
        num_steps=num_steps,
        time_steps=time_steps,
        net=model,
        noises=noises.to(device),
        class_labels=create_one_hot_torch(labels, num_classes).to(device) if labels is not None else None,
        sigma_min=sigma_min,
        sigma_max=sigma_max,
        rho=rho,
        s_churn=s_churn,
        s_min=s_min,
        s_max=s_max,
        s_noise=s_noise,
        multiply_noises=multiply_noises,
        same_noise=same_noise
    )


def run(
        num_steps: int,
        time_steps: list[int],
        image_height: int,
        image_width: int,
        image_channels: int,
        reference_path: str,
        model_name: str,
        model_path: str,
        noise_folder: str = None,
        label_folder: str = None,
        noise_start_index: int = 0,
        label_start_index: int = 0,
        num_classes: Optional[int] = None,
        model_load_keys: list[str] = None,
        num_samples: int = 50_000,
        batch_size: int = 50,
        output_save_path: str = None,
        sigma_min: float = 0.002,
        sigma_max: float = 80,
        rho: float = 7,
        s_churn: float = 0,
        s_min: float = 0,
        s_max: float = float('inf'),
        s_noise: float = 1,
        multiply_noises: bool = True,
        same_noise: bool = True,
        dims: int = 2048,
        data_loader_workers: int = 8,
        distributed: bool = False,
        distributed_backend: str = 'nccl'
) -> None:
    Logger.debug(
        f'{get_class_name(run)} - '
        f'num_steps: {num_steps}, '
        f'time_steps: {time_steps}, '
        f'image_height: {image_height}, '
        f'image_width: {image_width}, '
        f'image_channels: {image_channels}, '
        f'reference_path: {reference_path}, '
        f'model_name: {model_name}, '
        f'model_path: {model_path}, '
        f'noise_folder: {noise_folder}, '
        f'label_folder: {label_folder}, '
        f'noise_start_index: {noise_start_index}, '
        f'label_start_index: {label_start_index}, '
        f'num_classes: {num_classes}, '
        f'model_load_keys: {model_load_keys}, '
        f'num_samples: {num_samples}, '
        f'batch_size: {batch_size}, '
        f'output_save_path: {output_save_path}, '
        f'sigma_min: {sigma_min}, '
        f'sigma_max: {sigma_max}, '
        f'rho: {rho}, '
        f's_churn: {s_churn}, '
        f's_min: {s_min}, '
        f's_max: {s_max}, '
        f's_noise: {s_noise}, '
        f'multiply_noises: {multiply_noises}, '
        f'same_noise: {same_noise}, '
        f'dims: {dims}, '
        f'data_loader_workers: {data_loader_workers}, '
        f'distributed: {distributed}, '
        f'distributed_backend: {distributed_backend}'
    )
    if distributed:
        DistributedManager.init(backend=distributed_backend)

    conditional: bool = num_classes is not None
    Logger.debug(f'conditional: {conditional}')

    Logger.debug('creating model')
    model: torch.nn.Module = create_edm_model(model_name)
    Logger.debug('loading model')
    model: torch.nn.Module = load_from_state_dict(model, model_path, model_load_keys)
    model.to(get_default_device())
    model.eval()

    noise_dataset: NumpyDataset = FolderNumpyDataset(
        data_shape=(image_channels, image_height, image_width),
        data_type=np.float64,
        folder=noise_folder,
        num_samples=num_samples,
        start_index=noise_start_index
    ) if noise_folder is not None else RandomNoiseDataset(
        num_samples=num_samples,
        noise_shape=(image_channels, image_height, image_width)
    )
    label_dataset: NumpyDataset = FolderNumpyDataset(
        data_shape=(),
        data_type=np.int64,
        folder=label_folder,
        num_samples=num_samples,
        start_index=label_start_index
    ) if label_folder is not None else RandomLabelDataset(
        num_samples=num_samples,
        num_classes=num_classes
    )

    dataset: NoiseLabelDataset = NumpyNoiseLabelDataset(
        noise_dataset=noise_dataset,
        label_dataset=label_dataset
    ) if conditional else NumpyNoiseDataset(
        noise_dataset=noise_dataset
    )

    Logger.debug('calculating fid')
    fid: float = calculate_fid_for_dataset_and_model(
        model=model,
        dataset=dataset,
        reference_path=reference_path,
        inference_batch_func=lambda m, n, l: inference_multi_step_euler(
            model=m,
            num_steps=num_steps,
            time_steps=time_steps,
            noises=n,
            labels=l if conditional else None,
            num_classes=num_classes,
            sigma_min=sigma_min,
            sigma_max=sigma_max,
            rho=rho,
            s_churn=s_churn,
            s_min=s_min,
            s_max=s_max,
            s_noise=s_noise,
            multiply_noises=multiply_noises,
            same_noise=same_noise
        ),
        batch_size=batch_size,
        dims=dims,
        data_loader_workers=data_loader_workers
    )
    Logger.debug(f'fid: {fid}')
    if DistributedManager.is_main():
        if output_save_path is not None:
            Logger.debug(f'saving fid to {output_save_path}')
            if os.path.dirname(output_save_path) != '':
                os.makedirs(os.path.dirname(output_save_path), exist_ok=True)
            with open(output_save_path, 'wb') as file:
                pickle.dump(fid, file)

    if distributed:
        DistributedManager.destroy()


def run_from_config(config: dict[str, Any]) -> None:
    run(**config)


def parse_args() -> argparse.Namespace:
    parser: argparse.ArgumentParser = argparse.ArgumentParser()
    parser.add_argument('--num_steps', type=int, required=True)
    parser.add_argument('--time_steps', type=int, nargs='+', required=True)
    parser.add_argument('--image_height', type=int, required=True)
    parser.add_argument('--image_width', type=int, required=True)
    parser.add_argument('--image_channels', type=int, required=True)
    parser.add_argument('--reference_path', type=str, required=True)
    parser.add_argument('--model_name', type=str, required=True)
    parser.add_argument('--model_path', type=str, required=True)
    parser.add_argument('--noise_folder', type=str, default=None)
    parser.add_argument('--label_folder', type=str, default=None)
    parser.add_argument('--noise_start_index', type=int, default=0)
    parser.add_argument('--label_start_index', type=int, default=0)
    parser.add_argument('--num_classes', type=int, default=None)
    parser.add_argument('--model_load_keys', type=str, nargs='+', default=None)
    parser.add_argument('--num_samples', type=int, default=50_000)
    parser.add_argument('--batch_size', type=int, default=50)
    parser.add_argument('--output_save_path', type=str, default=None)
    parser.add_argument('--sigma_min', type=float, default=0.002)
    parser.add_argument('--sigma_max', type=float, default=80)
    parser.add_argument('--rho', type=float, default=7)
    parser.add_argument('--s_churn', type=float, default=0)
    parser.add_argument('--s_min', type=float, default=0)
    parser.add_argument('--s_max', type=float, default=float('inf'))
    parser.add_argument('--s_noise', type=float, default=1)
    parser.add_argument('--multiply_noises', type=bool, default=True)
    parser.add_argument('--same_noise', type=bool, default=True)
    parser.add_argument('--dims', type=int, default=2048)
    parser.add_argument('--data_loader_workers', type=int, default=8)
    parser.add_argument('--distributed', action='store_true')
    parser.add_argument('--distributed_backend', type=str, default='nccl')
    return parser.parse_args()


def get_config_from_args(args: argparse.Namespace) -> dict[str, Any]:
    return vars(args)


def main() -> None:
    args: argparse.Namespace = parse_args()
    config: dict[str, Any] = get_config_from_args(args)
    run_from_config(config)


if __name__ == '__main__':
    main()
