import io
import re
import sys
import os
import gc 
import subprocess
import unittest
import tempfile
from typing import List, Dict, Tuple
from unittest.runner import TextTestRunner
import signal
from contextlib import contextmanager
from utils import test_code_preprocess, get_code_from_response, get_case_from_response, get_result_from_code, get_code_from_case
from pathlib import Path
import multiprocessing
from multiprocessing import Manager

class Unittest_python:
    def __init__(self):
        pass
    
    @contextmanager
    def timeout(self, seconds: int):
        """设置代码执行的超时上下文管理器"""
        def signal_handler(signum, frame):
            raise TimeoutError(f"测试执行超时（{seconds}秒）")
        
        original_handler = signal.getsignal(signal.SIGALRM)
        # 设置信号处理
        try:
            signal.signal(signal.SIGALRM, signal_handler)
            signal.alarm(seconds)
            yield
        except TimeoutError as e:
            # 超时异常直接向上抛出
            raise e
        except Exception as e:
            # 捕获其他异常并重新抛出
            raise e
        finally:
            # 重置闹钟
            signal.alarm(0)
            signal.signal(signal.SIGALRM, original_handler)

    def unittest_action(self, data: list[dict], response_list: List[str], print_error: bool=False) -> Tuple[List[str], List[str]]:
        """
        执行回复中的单元测试代码并收集结果
        
        参数:
            data: 字典列表
            response_list: 消息响应列表
            print_error: 是否打印错误信息
            
        返回:
            action_log_list: 记录每个操作过程的日志列表
            action_res_list: 每个测试结果的列表
        """
        action_log_list = []
        action_res_list = []


        test_code_list = get_code_from_response(response_list)
        for test_code in test_code_list:
            try:
                # 创建临时测试模块
                idx = test_code_list.index(test_code)+1
                print(f"正在执行测试第{idx}个代码")

                task_data = data[idx-1]
                origin_code = task_data.get("code", None)
                if origin_code is not None:
                    origin_code = test_code_preprocess(origin_code)
                test_code = origin_code + "\n\n" + test_code
                module_name = "temp_test_module"
                module = type(sys)(module_name)
                module.__dict__['unittest'] = unittest
                sys.modules[module_name] = module
                timeout_error = False
                unknown_error = False
                error_str = None
                
                # 执行测试代码
                old_stdout = sys.stdout
                new_stdout = io.StringIO()
                sys.stdout = new_stdout
                
                try:
                    with timeout(30):
                        exec(test_code, module.__dict__)
                        
                        # 发现并运行测试
                        test_loader = unittest.TestLoader()
                        test_suite = test_loader.loadTestsFromModule(module)
                        test_runner = unittest.TextTestRunner(stream=new_stdout, verbosity=2)
                        test_runner.run(test_suite)
                except TimeoutError as e:
                    # print(f"执行测试时发生超时错误: {str(e)}")
                    print(f"出错代码为：\n{test_code}")
                    timeout_error = True
                    error_str = str(e)
                    new_stdout.write(f"执行测试时发生超时错误: {str(e)}\n")    
                except Exception as e:
                    # print(f"执行测试时发生未知错误: {str(e)}")
                    print(f"出错代码为：\n{test_code}")
                    unknown_error = True
                    error_str = str(e)
                    new_stdout.write(f"执行测试时发生未知错误: {str(e)}\n")
                finally:
                    sys.stdout = old_stdout
                    del sys.modules[module_name]
                    
                # 获取测试输出
                test_output = new_stdout.getvalue()
                action_log_list.append(test_output)

                if timeout_error:
                    if print_error:
                        print(test_output)
                    timeout_error = False
                    error_str = None
                elif unknown_error:
                    if print_error:
                        print(test_output)
                    unknown_error = False
                    error_str = None
                
                # 分析结果
                lines = test_output.strip().split('\n')
                if len(lines) >= 2:
                    last_line = lines[-1].strip()
                    second_last_line = lines[-2].strip()
                    
                    if second_last_line.startswith('OK'):
                        action_res_list.append('OK')
                    elif second_last_line.startswith('FAILED'):
                        action_res_list.append('FAILED')
                    else:
                        # 处理可能的不同输出格式
                        if 'FAILED' in test_output:
                            action_res_list.append('FAILED')
                        elif 'OK' in test_output:
                            action_res_list.append('OK')
                        else:
                            action_res_list.append('UNKNOWN')
                else:
                    action_res_list.append('FAILED')
                    
            except Exception as e:
                print(f"执行测试时主代码发生未知错误: {str(e)}")
                action_log_list.append(f"执行测试时发生未知错误: {str(e)}")
                action_res_list.append(["UNKNOWN"]*len(data[idx-1]["code_list"]))
        
        return action_log_list, action_res_list


    def unittest_case_action(self, data: list[dict], response_list: list[list[str]], print_error: bool=False, is_response:bool=True) -> Tuple[List[str], List[str]]:
        """
        执行回复中的单元测试代码并针对每个测试用例收集结果
        
        参数:
            data: 数据字典列表
            response_list: 消息响应列表
            print_error: 是否打印错误信息
            
        返回:
            action_log_list: 记录每个操作过程的日志列表
            action_res_list: 每个测试结果的列表
        """
        action_log_list = []
        action_res_list = []

        if is_response:
            response_list = get_code_from_response(response_list, action_type="case")
            response_list = get_case_from_response(response_list)

        for task_idx, task_cases in enumerate(response_list):
            # print(f"正在执行任务{task_idx+1}个代码")
            task_data = data[task_idx-1]
            origin_code = task_data.get("code", None)
            if origin_code is not None:
                origin_code = test_code_preprocess(origin_code)
            action_task_log_list = []
            action_task_res_list = []
            for case_idx, test_case in enumerate(task_cases):
                try:
                    # 创建临时测试模块
                    # 准备测试代码，将测试用例插入到测试类中

                    front_code = "import unittest\nfrom io import StringIO\nimport sys\nclass TestBottleOpening(unittest.TestCase):\n    def test_1(self):\n"
                    back_code = "\n        old_stdin = sys.stdin\n        sys.stdin = StringIO(input_str)\n        capturedOutput = StringIO()\n        sys.stdout = capturedOutput\n        try:\n            mycode()  # 调用test_code\n        finally:\n            sys.stdin = old_stdin\n            sys.stdout = sys.__stdout__\n        self.assertEqual(capturedOutput.getvalue(), output_str) \n\nif __name__ == '__main__':\n    unittest.main()"
                    test_class_code = origin_code + "\n\n" + front_code + test_case + back_code
                    # 执行测试
                    module_name = f"temp_test_module_{task_idx}_{case_idx}"
                    module = type(sys)(module_name)
                    module.__dict__['unittest'] = unittest
                    sys.modules[module_name] = module
                    timeout_error = False
                    unknown_error = False
                    error_str = None
                    
                    # 执行测试代码
                    old_stdout = sys.stdout
                    new_stdout = io.StringIO()
                    sys.stdout = new_stdout
                    
                    try:
                        with timeout(30):
                            exec(test_class_code, module.__dict__)
                            
                            # 发现并运行测试
                            test_loader = unittest.TestLoader()
                            test_suite = test_loader.loadTestsFromModule(module)
                            test_runner = unittest.TextTestRunner(stream=new_stdout, verbosity=2)
                            test_runner.run(test_suite)
                    except TimeoutError as e:
                        print(f"执行测试时发生超时错误: {str(e)}")
                        print(f"出错代码为：\n{test_class_code}")
                        timeout_error = True
                        error_str = str(e)
                        new_stdout.write(f"执行测试时发生超时错误: {str(e)}\n")    
                    except Exception as e:
                        print(f"执行测试时发生未知错误: {str(e)}")
                        print(f"出错代码为：\n{test_class_code}")
                        unknown_error = True
                        error_str = str(e)
                        new_stdout.write(f"执行测试时发生未知错误: {str(e)}\n")
                    finally:
                        sys.stdout = old_stdout
                        del sys.modules[module_name]
                        
                    # 获取测试输出
                    test_output = new_stdout.getvalue()
                    action_task_log_list.append(test_output)

                    if timeout_error:
                        if print_error:
                            print(test_output)
                        timeout_error = False
                        error_str = None
                    elif unknown_error:
                        if print_error:
                            print(test_output)
                        unknown_error = False
                        error_str = None
                    
                    # 分析结果
                    lines = test_output.strip().split('\n')
                    if len(lines) >= 2:
                        last_line = lines[-1].strip()
                        second_last_line = lines[-2].strip()
                        
                        if second_last_line.startswith('OK'):
                            action_task_res_list.append('OK')
                        elif second_last_line.startswith('FAILED'):
                            action_task_res_list.append('FAILED')
                        else:
                            # 处理可能的不同输出格式
                            if 'FAILED' in test_output:
                                action_task_res_list.append('FAILED')
                            elif 'OK' in test_output:
                                action_task_res_list.append('OK')
                            else:
                                action_task_res_list.append('UNKNOWN')
                    else:
                        action_res_list.append('UNKNOWN')
    
                except Exception as e:
                    print(f"执行测试用例时发生未知错误: {str(e)}")
                    action_log_list.append(f"执行测试用例时发生未知错误: {str(e)}")
                    action_res_list.append("UNKNOWN")
            if 'UNKNOWN' in action_task_res_list:
                action_res_list.append('UNKNOWN')
            elif 'FAILED' in action_task_res_list:
                action_res_list.append('FAILED')
            else:
                action_res_list.append('OK')
            str_temp = ""
            for i in range(len(action_task_res_list)):
                if action_task_res_list[i] != 'OK':
                    str_temp  = str_temp + "/n" + action_task_log_list[i]
            action_log_list.append(str_temp)
        
        return action_log_list, action_res_list



class Pytest_python:
    def __init__(self):
        pass

    def pytest_action(self, data: list[dict], response_list: List[str], print_error: bool=False) -> Tuple[List[str], List[str]]:
        """
        执行回复中的单元测试代码并收集结果
        
        参数:
            data: 字典列表
            response_list: 消息响应列表
            print_error: 是否打印错误信息
            
        返回:
            action_log_list: 记录每个操作过程的日志列表
            action_res_list: 每个测试结果的列表
            cov_dict: 每个测试的覆盖率数据字典
        """
        action_log_list = []
        action_res_list = []
        cov_line_list = []
        cov_branch_list = []
        cov_line_all = 1e-12
        cov_line_covered = 0
        cov_branch_all = 1e-12
        cov_branch_covered = 0


        test_code_list = get_code_from_response(response_list)
        for test_code in test_code_list:
            try:
                # 创建临时测试模块
                idx = test_code_list.index(test_code)+1
                print(f"正在执行测试第{idx}个代码")
                if test_code == "代码提取错误，没有提取到有效信息":
                    print(test_code)
                    action_log_list.append(test_code)
                    action_res_list.append("UNKNOWN")
                    continue

                task_data = data[idx-1]
                origin_code = task_data.get("code", None)
                if origin_code is not None:
                    origin_code = test_code_preprocess(origin_code)
                else:
                    origin_code = ""
                with tempfile.TemporaryDirectory() as temp_dir:
                    # 创建 mycode.py 文件
                    code_path = os.path.join(temp_dir, "tempcode.py")
                    with open(code_path, "w", encoding="utf-8") as f:
                        f.write(origin_code)

                    # 创建 test_mycode.py 文件
                    test_path = os.path.join(temp_dir, "test_tempcode.py")
                    with open(test_path, "w", encoding="utf-8") as f:
                        f.write(test_code)

                    cmd = [
                        "pytest", 
                        "--cov=tempcode", 
                        "--cov-branch",
                        "--cov-report=term-missing",
                        "-v",
                        test_path
                    ]
                    try:
                        result = subprocess.run(
                            cmd,
                            cwd=temp_dir,
                            capture_output=True,
                            text=True,
                            timeout=30
                        )
                    except subprocess.TimeoutExpired as e:
                        print(f"出错代码为：\n{origin_code}")
                        print(f"对应单测代码为：\n{test_code}")
                        print(f"执行测试时发生超时错误: {str(e)}")
                        result = f"执行测试时发生超时错误: {str(e)}"
                    except Exception as e:
                        print(f"出错代码为：\n{origin_code}")
                        print(f"对应单测代码为：\n{test_code}")
                        print(f"执行测试时发生未知错误: {str(e)}")
                        result = f"执行测试时发生未知错误: {str(e)}"
                    
                # 获取测试输出
                if isinstance(result, subprocess.CompletedProcess):
                    # 测试正常执行完成，获取标准输出和错误输出
                    test_output = result.stdout + result.stderr
                else:
                    # 测试出错，result 是错误信息字符串
                    test_output = result
                action_log_list.append(test_output)
                
                # 分析结果
                lines = test_output.strip().split('\n')
                for line in lines:
                    if line.startswith("tempcode.py "):
                        parts = line.split()
                        numbers = parts[1:5]
                        numbers = [float(num) for num in numbers]
                        # print(numbers)
                        cov_line = 100 if numbers[0] == 0 else (1 - numbers[1]/numbers[0]) * 100
                        cov_branch = 100 if numbers[2] == 0 else (1 - numbers[3]/numbers[2]) * 100
                        cov_line_all += numbers[0]
                        cov_line_covered += numbers[1]
                        cov_branch_all += numbers[2]
                        cov_branch_covered += numbers[3]
                        cov_line_list.append(cov_line)
                        cov_branch_list.append(cov_branch)


                last_line = lines[-1].strip()
                if "failed" in last_line:
                    action_res_list.append('FAILED')
                elif "passed" in last_line:
                    action_res_list.append('OK')
                else:
                    print(test_output)
                    print(f"出错代码:{test_code}")
                    action_res_list.append('UNKNOWN')
                    
                    
            except Exception as e:
                print(f"执行测试时主代码发生未知错误: {str(e)}")
                print(test_output)
                action_log_list.append(f"执行测试时发生未知错误: {str(e)}")
                action_res_list.append("UNKNOWN")
        cov_dict = {
            "cov_line": 100.00 if cov_line_all == 0 else round((1 - cov_line_covered / cov_line_all) * 100, 2),
            "cov_branch": 100.00 if cov_branch_all == 0 else round((1 - cov_branch_covered / cov_branch_all) * 100, 2),
            "cov_line_list": cov_line_list,
            "cov_branch_list": cov_branch_list
        }
        
        return action_log_list, action_res_list, cov_dict  


    def pytest_action_pair(self, data: list[dict], response_list: List[str], print_error: bool=False) -> Tuple[List[str], List[str]]:
        """
        执行回复中的单元测试代码并收集结果
        
        参数:
            data: 字典列表
            response_list: 消息响应列表
            print_error: 是否打印错误信息
            
        返回:
            action_log_list: 记录每个操作过程的日志列表
            action_res_list: 每个测试结果的列表
            cov_dict: 每个测试的覆盖率数据字典
        """
        action_log_list = []
        action_res_list = []
        cov_line_list = []
        cov_branch_list = []
        cov_line_error_list = []
        cov_branch_error_list = []
        cov_line_all = 1e-12
        cov_line_covered = 0
        cov_branch_all = 1e-12
        cov_branch_covered = 0


        test_code_list = get_code_from_response(response_list)
        for test_code in test_code_list:
            try:
                # 创建临时测试模块
                idx = test_code_list.index(test_code)+1
                print(f"正在执行测试第{idx}个单测代码")
                if test_code == "代码提取错误，没有提取到有效信息":
                    print(test_code)
                    action_log_list.append(test_code)
                    action_res_list.append(["REQUEST ERROR"]*len(data[idx-1]["code_list"]))
                    continue

                task_data = data[idx-1]
                action_log_temp = []
                cov_line_temp = []
                cov_branch_temp = []
                action_res_temp = []
                result_list = get_result_from_code(task_data, test_code)
                for result in result_list:
                    # 获取测试输出
                    if isinstance(result, subprocess.CompletedProcess):
                        # 测试正常执行完成，获取标准输出和错误输出
                        test_output = result.stdout + result.stderr
                    else:
                        # 测试出错，result 是错误信息字符串
                        test_output = result
                    action_log_temp.append(test_output)
                    # 分析结果
                    lines = test_output.strip().split('\n')
                    for line in lines:
                        if line.startswith("tempcode.py "):
                            parts = line.split()
                            numbers = parts[1:5]
                            numbers = [float(num) for num in numbers]
                            # print(numbers)
                            cov_line = 100 if (numbers[0]-15) == 0 else (1 - numbers[1]/(numbers[0]-15)) * 100
                            cov_branch = 100 if numbers[2] == 0 else (1 - numbers[3]/numbers[2]) * 100
                            cov_line_all += (numbers[0]-15)
                            # print(f"{numbers[1]}/{numbers[0]}")
                            cov_line_covered += numbers[1]
                            cov_branch_all += numbers[2]
                            cov_branch_covered += numbers[3]
                            cov_line_temp.append(cov_line)
                            cov_branch_temp.append(cov_branch)

                    last_line = lines[-1].strip()
                    if "failed" in last_line:
                        action_res_temp.append('FAILED')
                    elif "passed" in last_line:
                        action_res_temp.append('OK')
                    elif "超时错误" in test_output:
                        print(test_output)
                        action_res_temp.append('TIMEOUT')
                    else:
                        # print(f"出错代码为：\n{task_data['code_list']}")
                        # print(f"对应单测代码为：\n{test_code}")
                        print(test_output)
                        action_res_temp.append('COMPILATION_ERROR')
                s_temp = ""
                for i, action_log in enumerate(action_log_temp):
                    s_temp += f"第{i+1}次测试结果:\\n" + action_log + '\\n'
                action_log_list.append(s_temp)
                cov_line_list.append(cov_line_temp)
                cov_branch_list.append(cov_branch_temp)
                action_res_list.append(action_res_temp)
                    
            except Exception as e:
                print(f"执行测试时主代码发生未知错误: {str(e)}")
                action_log_list.append(f"执行测试时发生未知错误: {str(e)}")
                action_res_list.append(["UNKNOWN"]*len(data[idx-1]["code_list"]))

        cov_dict = {
            "cov_line": 100.00 if cov_line_all == 0 else round((1 - cov_line_covered / cov_line_all) * 100, 2),
            "cov_branch": 100.00 if cov_branch_all == 0 else round((1 - cov_branch_covered / cov_branch_all) * 100, 2),
            "cov_line_list": cov_line_list,
            "cov_branch_list": cov_branch_list,
        }
        
        return action_log_list, action_res_list, cov_dict





    def pytest_case_action(self, data: list[dict], response_list: list[list[str]], print_error: bool=False, is_response:bool=True) -> Tuple[List[str], List[str]]:
        """
        执行回复中的单元测试代码并针对每个测试用例收集结果
        
        参数:
            data: 数据字典列表
            response_list: 消息响应列表
            print_error: 是否打印错误信息
            
        返回:
            action_log_list: 记录每个操作过程的日志列表
            action_res_list: 每个测试结果的列表
        """

        action_log_list = []
        action_res_list = []
        cov_line_list = []
        cov_branch_list = []
        cov_line_error_list = []
        cov_branch_error_list = []
        cov_line_all = 1e-12
        cov_line_covered = 0
        cov_branch_all = 1e-12
        cov_branch_covered = 0

        if is_response:
            code_list = get_code_from_response(response_list, action_type="case")
            case_list = get_case_from_response(code_list)

        for idx, task_cases in enumerate(case_list):     
            try:
                print(f"正在执行任务{idx+1}个代码")
                task_data = data[idx]
                if "代码提取错误，没有提取到有效信息" in task_cases:
                    print("代码提取错误，没有提取到有效信息")
                    action_log_list.append("代码提取错误，没有提取到有效信息")
                    action_res_list.append(["REQUEST ERROR"]*len(data[idx]["code_list"]))
                    continue
                test_code = "from tempcode import run_method\nimport pytest\nclass TestCode():\n"
                for case_idx, test_case in enumerate(task_cases):
                    # 准备测试代码，将测试用例插入到测试类中   
                    test_code += f"    def test_{case_idx+1}(self):\n"
                    test_code += test_case
                    test_code += "\n        assert run_method(input_str) == output_str\n\n"        
                # 创建临时测试模块
                action_log_temp = []
                cov_line_temp = []
                cov_branch_temp = []
                action_res_temp = []
                result_list = get_result_from_code(task_data, test_code)
                for result in result_list:
                    # 获取测试输出
                    if isinstance(result, subprocess.CompletedProcess):
                        # 测试正常执行完成，获取标准输出和错误输出
                        test_output = result.stdout + result.stderr
                    else:
                        # 测试出错，result 是错误信息字符串
                        test_output = result
                    action_log_temp.append(test_output)
                    # 分析结果
                    lines = test_output.strip().split('\n')
                    for line in lines:
                        if line.startswith("tempcode.py "):
                            parts = line.split()
                            numbers = parts[1:5]
                            numbers = [float(num) for num in numbers]
                            # print(numbers)
                            cov_line = 100 if (numbers[0]-15) == 0 else (1 - numbers[1]/(numbers[0]-15)) * 100
                            cov_branch = 100 if numbers[2] == 0 else (1 - numbers[3]/numbers[2]) * 100
                            cov_line_all += (numbers[0]-15)
                            cov_line_covered += numbers[1]
                            cov_branch_all += numbers[2]
                            cov_branch_covered += numbers[3]
                            cov_line_temp.append(cov_line)
                            cov_branch_temp.append(cov_branch)

                    last_line = lines[-1].strip()
                    if "failed" in last_line:
                        action_res_temp.append('FAILED')
                    elif "passed" in last_line:
                        action_res_temp.append('OK')
                    elif "超时错误" in test_output:
                        print(test_output)
                        action_res_temp.append('TIMEOUT')
                    else:
                        # print(f"出错代码为：\n{task_data['code_list']}")
                        print(test_output)
                        # if "执行测试时发生超时错误" not in test_output:
                        #     print(f"对应单测代码为：\n{test_code}")
                        #     print(f"对应respon为： \n{response_list[idx]}")
                        action_res_temp.append('COMPILATION_ERROR')
                s_temp = ""
                for i, action_log in enumerate(action_log_temp):
                    s_temp += f"第{i+1}次测试结果:\\n" + action_log + '\\n'
                action_log_list.append(s_temp)
                cov_line_list.append(cov_line_temp)
                cov_branch_list.append(cov_branch_temp)
                action_res_list.append(action_res_temp)
                    
            except Exception as e:
                print(f"执行测试时主代码发生未知错误: {str(e)}")
                action_log_list.append(f"执行测试时发生未知错误: {str(e)}")
                action_res_list.append(["UNKNOWN"]*len(data[idx-1]["code_list"]))

        cov_dict = {
            "cov_line": 100.00 if cov_line_all == 0 else round((1 - cov_line_covered / cov_line_all) * 100, 2),
            "cov_branch": 100.00 if cov_branch_all == 0 else round((1 - cov_branch_covered / cov_branch_all) * 100, 2),
            "cov_line_list": cov_line_list,
            "cov_branch_list": cov_branch_list,
        }
        
        return action_log_list, action_res_list, cov_dict


    def pytest_action_pass(self, data: list[dict], response_list: List[str], print_error: bool=False) -> Tuple[List[str], List[str]]:
        """
        执行回复中的单元测试代码并收集结果
        
        参数:
            data: 字典列表
            response_list: 消息响应列表
            print_error: 是否打印错误信息
            
        返回:
            action_log_list: 记录每个操作过程的日志列表
            action_res_list: 每个测试结果的列表
            cov_dict: 每个测试的覆盖率数据字典
        """
        action_log_list = []
        action_res_list = []


        code_list = get_code_from_response(response_list)
        for code in code_list:
            try:
                idx = code_list.index(code)+1
                print(f"正在执行测试第{idx}个单测代码")
                if code == "代码提取错误，没有提取到有效信息":
                    print(code)
                    action_log_list.append(code)
                    action_res_list.append("UNKNOWN")
                    continue

                task_data = data[idx-1]

                test_cases = task_data.get('test_cases', None)
                test_code = get_code_from_case(test_cases)
                if code is not None:
                    origin_code = test_code_preprocess(code, add_test_code=True)
                else:
                    origin_code = ""
                with tempfile.TemporaryDirectory() as temp_dir:
                    # 创建 mycode.py 文件
                    code_path = os.path.join(temp_dir, "tempcode.py")
                    with open(code_path, "w", encoding="utf-8") as f:
                        f.write(origin_code)

                    # 创建 test_mycode.py 文件
                    test_path = os.path.join(temp_dir, "test_tempcode.py")
                    with open(test_path, "w", encoding="utf-8") as f:
                        f.write(test_code)

                    cmd = [
                        "pytest", 
                        "--cov=tempcode", 
                        "--cov-branch",
                        "--cov-report=term-missing",
                        "-v",
                        test_path
                    ]
                    try:
                        result = subprocess.run(
                            cmd,
                            cwd=temp_dir,
                            capture_output=True,
                            text=True,
                            timeout=30
                        )
                    except subprocess.TimeoutExpired as e:
                        result = f"执行测试时发生超时错误: {str(e)}"
                    except Exception as e:
                        result = f"执行测试时发生未知错误: {str(e)}"
                # 获取测试输出
                if isinstance(result, subprocess.CompletedProcess):
                    # 测试正常执行完成，获取标准输出和错误输出
                    test_output = result.stdout + result.stderr
                else:
                    # 测试出错，result 是错误信息字符串
                    test_output = result
                action_log_list.append(test_output)
                # 分析结果
                lines = test_output.strip().split('\n')
                last_line = lines[-1].strip()
                if "failed" in last_line:
                    action_res_list.append('FAILED')
                elif "passed" in last_line:
                    action_res_list.append('OK')
                else:
                    print(f"出错代码为：\n{origin_code}")
                    print(f"对应response为： \n{response_list[idx]}")
                    print(test_output)
                    action_res_temp.append('UNKNOWN')
                    
            except Exception as e:
                print(f"执行测试时主代码发生未知错误: {str(e)}")
                action_log_list.append(f"执行测试时主代码发生未知错误: {str(e)}")
                action_res_list.append("UNKNOWN")
        
        return action_log_list, action_res_list




class Gtest_cpp:
    def __init__(self, cleanup=True, print_output=False, print_error=False):
        """初始化分析器，设置清理标志"""
        self.cleanup = cleanup
        self.temp_dir = None
        self.result = "FAILED"
        self.cov_line = 0.0
        self.cov_line_count = 0
        self.print_output = print_output
        self.log = ""
        self.print_error = print_error

    def run_command(self, command, cwd=None, shell=False, add_log=False, timeout=None):
        """执行命令并返回输出"""
        try:
            # print(f"执行命令: {' '.join(command) if isinstance(command, list) else command}")
            process = subprocess.Popen(
                command,
                cwd=cwd or self.temp_dir.name,
                shell=shell,
                text=True,
                stdout=subprocess.PIPE,
                stderr=subprocess.STDOUT,  # 合并标准输出和错误输出
                bufsize=1,  # 行缓冲
                universal_newlines=True
            )

            stdout, stderr = process.communicate(timeout=timeout)
            output_lines = [line.rstrip() for line in stdout.splitlines()]
            # 打印输出
            if self.print_output:
                for line in output_lines:
                    print(line)
                
            output_str = '\n'.join(output_lines)
            if add_log:
                # print(output_str)
                self.log += output_str
                self.log += '\n'

            return_code = process.returncode
            if return_code != 0:
                raise subprocess.CalledProcessError(return_code, command, 
                                                output='\n'.join(output_lines))
            return output_str
        except subprocess.CalledProcessError as e:
            # print(f"命令执行失败: {e.output}", file=sys.stderr)
            raise
        except subprocess.TimeoutExpired as e:
            # 超时处理：终止进程并记录错误
            process.kill()
            stdout, stderr = process.communicate()
            raise

    def setup_temp_environment(self, source_code, test_code):
        """创建临时环境并写入代码"""
        self.temp_dir = tempfile.TemporaryDirectory()
        # print(f"创建临时目录: {self.temp_dir.name}")
        
        # 写入源代码
        source_file = os.path.join(self.temp_dir.name, "source.cpp")
        with open(source_file, 'w') as f:
            f.write(source_code)
        
        # 写入测试代码
        test_file = os.path.join(self.temp_dir.name, "test.cpp")
        with open(test_file, 'w') as f:
            f.write(test_code)
        
        # 创建CMakeLists.txt
        cmake_file = os.path.join(self.temp_dir.name, "CMakeLists.txt")
        with open(cmake_file, 'w') as f:
            f.write("""
cmake_minimum_required(VERSION 3.10)
project(Gtest_cpp)

set(CMAKE_CXX_STANDARD 17)
set(CMAKE_CXX_STANDARD_REQUIRED ON)

# ===== 手动配置本地GTest =====
# 设置本地GTest的源代码目录（根据实际路径调整）
set(GTEST_SOURCE_DIR /home/disk1/kuangzhaoqi/program_file/TestJudge/third_party/gtest/src)

# 添加GTest子目录（会自动构建gtest和gtest_main）
add_subdirectory(${GTEST_SOURCE_DIR} ${CMAKE_CURRENT_BINARY_DIR}/gtest_build)

# 构建动态库（包含全局变量和核心逻辑）
add_library(source SHARED 
    source.cpp  # 包含mycode()和run_mycode()的实现
)

# 添加测试可执行文件
add_executable(my_test 
    test.cpp  # 包含Google Test测试用例
)

# 链接测试程序需要的库
target_link_libraries(my_test 
    gtest_main 
    dl  # Unix系统动态库加载所需
)

# 启用测试
enable_testing()
add_test(NAME my_test COMMAND my_test)

# 确保动态库在测试程序运行时可被找到（Unix系统）
set_target_properties(my_test PROPERTIES
    BUILD_WITH_INSTALL_RPATH TRUE
    INSTALL_RPATH "$ORIGIN"
    BUILD_RPATH_USE_ORIGIN TRUE
)
    
    
""")

    def build_project(self):
        """编译项目"""
        # 创建build目录
        build_dir = os.path.join(self.temp_dir.name, "build")
        os.makedirs(build_dir, exist_ok=True)
        
        # 配置CMake
        cmake_cmd = [
            "cmake",
            "-DCMAKE_BUILD_TYPE=Debug",
            '-DCMAKE_CXX_FLAGS=-g -O0 -fprofile-arcs -ftest-coverage',
            "-DCMAKE_C_COMPILER=/usr/local/gcc-11.2.0/bin/gcc",
            "-DCMAKE_CXX_COMPILER=/usr/local/gcc-11.2.0/bin/g++",
            ".."
        ]
        self.run_command(cmake_cmd, cwd=build_dir)
        
        # 编译项目
        build_cmd = ["cmake", "--build", "."]
        self.run_command(build_cmd, cwd=build_dir)
        
        return os.path.join(build_dir, "my_test")

    def run_tests(self, test_executable):
        """运行测试并解析结果"""
        test_cmd = [test_executable, "--gtest_output=xml:test_results.xml"]
        try:
            output = self.run_command(test_cmd, add_log=True, timeout=10)
        except subprocess.CalledProcessError as e:
            output = e.output  # 获取测试输出
            self.result = "FAILED"  # 标记测试失败
            return output
        
        # 改进测试结果判断逻辑 - 优先检查通过的用例数
        pass_match = re.search(r'\[  PASSED  \]\s+(\d+)\s+test', output)
        fail_match = re.search(r'\[  FAILED  \]\s+(\d+)\s+test', output)
        
        if pass_match and (not fail_match or int(fail_match.group(1)) == 0):
            self.result = "OK"
        else:
            self.result = "FAILED"
        
        return output

    def calculate_coverage(self):
        """计算代码覆盖率"""
        build_dir = os.path.join(self.temp_dir.name, "build")
        
        # 收集覆盖率数据
        self.run_command([
            "lcov", 
            "--capture", 
            "--directory", ".", 
            "--output-file", "coverage.info"
        ], cwd=build_dir)
        
        # 改进过滤条件，确保正确排除不需要的文件
        self.run_command([
            "lcov", 
            "--remove", "coverage.info", 
            "/usr/*", 
            "*/_deps/googletest/*", 
            "*/test.cpp",
            "--output-file", "coverage_filtered.info"
        ], cwd=build_dir, shell=False)
        
        # 生成覆盖率报告
        report = self.run_command([
            "lcov", 
            "--list", "coverage_filtered.info"
        ], cwd=build_dir)
        
        # 改进覆盖率提取逻辑 - 只关注source.cpp
        source_match = re.search(r'source\.cpp\s*\|\s*(\d+(?:\.\d+)?)%\s+(\d+(?:,\d+)*)\s*\|', report)
        if source_match:
            self.cov_line = float(source_match.group(1))
            self.cov_line_count = int(source_match.group(2).replace(',', ''))
        else:
            print("警告: 未能提取行覆盖率", file=sys.stderr)
        
        return report

    def cleanup_temp_environment(self):
        """清理临时环境"""
        if self.cleanup and self.temp_dir:
            # print(f"清理临时目录: {self.temp_dir.name}")
            self.temp_dir.cleanup()
            self.temp_dir = None

    def gtest_action(self, source_code, test_code, cal_cov=True):
        """执行完整的分析流程"""
        try:
            # 设置临时环境
            self.setup_temp_environment(source_code, test_code)
            
            # 编译项目
            # print("编译项目...")
            test_executable = self.build_project()
        except subprocess.CalledProcessError as e:
            print(f"\n编译过程中发生错误:\n {e.output}", file=sys.stderr)
            return {
                "result": "COMPILATION_ERROR",
                "log": f"编译过程中发生错误:\n {e.output}",
                "cov_line": 0.0,
                "covered_line_count": 0,
                "cov_line_count": 0
            }
            
        try:
            # 运行测试
            # print("运行测试...")
            self.run_tests(test_executable)
        except subprocess.CalledProcessError as e:
            print(f"\n执行过程中发生错误:\n {e.output}", file=sys.stderr)
            return {
                "result": "UNKNOWN",
                "log": f"执行过程中发生错误:\n {e.output}",
                "cov_line": 0.0,
                "covered_line_count": 0,
                "cov_line_count": 0
            }
        except subprocess.TimeoutExpired as e:
            # 超时处理：终止进程并记录错误
            print(f"\n测试执行超时: {str(e)}", file=sys.stderr)
            return {
                "result": "TIMEOUT",
                "log": f"测试执行超时: {str(e)}",
                "cov_line": 0.0,
                "covered_line_count": 0,
                "cov_line_count": 0
            }
        if not cal_cov:
            return {
                "result": self.result,
                "log": self.log,
                "cov_line": 0.0,
                "covered_line_count": 0,
                "cov_line_count": 0
            }
        elif cal_cov:
            try:
                # 计算代码覆盖率
                # print("计算代码覆盖率...")
                self.calculate_coverage()
                return {
                    "result": self.result,
                    "log": self.log,
                    "cov_line": self.cov_line,
                    "covered_line_count": int(self.cov_line * self.cov_line_count / 100),
                    "cov_line_count": self.cov_line_count
                }
            except Exception as e:
                print(f"\n执行过程中发生错误: {str(e)}", file=sys.stderr)
                return {
                    "result": "UNKNOWN",
                    "log": f"执行过程中发生错误: {str(e)}",
                    "cov_line": 0.0,
                    "covered_line_count": 0,
                    "cov_line_count": 0
                }
        # 清理临时环境
        self.cleanup_temp_environment()

    def gtest_case_action(self, data: list[dict], response_list: list[list[str]], print_error: bool=False, is_response:bool=True) -> Tuple[List[str], List[str]]:
        """
        执行回复中的单元测试代码并针对每个测试用例收集结果
        
        参数:
            data: 数据字典列表
            response_list: 消息响应列表
            print_error: 是否打印错误信息
            is_response: 是否是消息响应
            
        返回:
            action_log_list: 记录每个操作过程的日志列表
            action_res_list: 每个测试结果的列表
            cov_dict: 行覆盖率字典
        """

        action_log_list = []
        action_res_list = []
        cov_line_list = []
        cov_line_count_list = []
        covered_line_count_list = []

        if is_response:
            code_list = get_code_from_response(response_list, start_str='```cpp', end_str='```', action_type="case")
            case_list = get_case_from_response(code_list, return_type="set")
        else:
            case_list = response_list

        for idx, task_cases in enumerate(case_list):     
            try:
                # print(f"正在执行任务{idx+1}个代码")
                task_data = data[idx]
                if "代码提取错误，没有提取到有效信息" in task_cases:
                    print("代码提取错误，没有提取到有效信息")
                    action_log_list.append("代码提取错误，没有提取到有效信息")
                    action_res_list.append(["UNKNOWN"]*len(data[idx]["code_list"]))
                    continue

                action_log_temp = []
                action_res_temp = []
                cov_line_temp = []
                cov_line_count_temp = 0
                covered_line_count_temp = 0
                test_code = "#include <gtest/gtest.h>\n#include <sstream>\n#include <string>\n#include <cctype>\n#include <dlfcn.h>\ntypedef std::string (*RunMyCodeFunc)(const std::string&);\n"
                test_code += 'void* load_library(RunMyCodeFunc& run_func) {\n    void* handle = dlopen("./build/libsource.so", RTLD_LAZY);\n    if (!handle) {\n        std::cerr << "无法加载动态库: " << dlerror() << std::endl;\n        return nullptr;\n    }\n    dlerror();\n    run_func = (RunMyCodeFunc)dlsym(handle, "run_mycode");\n    const char* dlsym_error = dlerror();\n    if (dlsym_error) {\n        std::cerr << "无法找到函数: " << dlsym_error << std::endl;\n        dlclose(handle);\n        return nullptr;\n    }\n    return handle;\n}\n'
                test_code += 'std::string trimTrailingWhitespace(const std::string &str) {\n    size_t end = str.find_last_not_of(" \\t\\n\\r");\n    if (end != std::string::npos) {\n        return str.substr(0, end + 1);\n    } else {\n        return "";\n    }\n}\n'
                test_code += 'TEST(ColorDetectionTest, AllCases) {\n    RunMyCodeFunc run_func;\n'
                for case_idx, test_case in enumerate(task_cases):
                    test_code += f'    void* handle{case_idx} = load_library(run_func);\n    ASSERT_NE(handle{case_idx}, nullptr);\n'
                    test_code += f'    EXPECT_EQ(trimTrailingWhitespace(run_func("{test_case[0]}")), "{test_case[1]}");\n'
                    test_code += f'    dlclose(handle{case_idx});\n'
                test_code += "}\n"
                test_code += "int main(int argc, char **argv) {\n    ::testing::InitGoogleTest(&argc, argv);\n    return RUN_ALL_TESTS();\n}\n"
                origin_code_list = task_data["code_list"]
                for origin_code in origin_code_list:
                    # print(origin_code)
                    origin_code = test_code_preprocess(origin_code, add_test_code=True, code_type="cpp")
                    # print(origin_code)
                    # print(test_code)
                    result_dict = self.gtest_action(origin_code, test_code)
                    action_log_temp.append(result_dict["log"])
                    action_res_temp.append(result_dict["result"])
                    cov_line_temp.append(result_dict["cov_line"])
                    covered_line_count_temp+=result["covered_line_count"]
                    cov_line_count_temp+=result_dict["cov_line_count"]
                s_temp = ""
                for i, action_log in enumerate(action_log_temp):
                    s_temp += f"第{i+1}次测试结果:\\n" + action_log + '\\n'
                action_log_list.append(s_temp)
                cov_line_list.append(cov_line_temp)
                cov_line_count_list.append(cov_line_count_temp)
                covered_line_count_list.appedn(covered_line_count_temp)
                action_res_list.append(action_res_temp)
                    
            except Exception as e:
                print(f"执行测试时主代码发生未知错误: {str(e)}")
                action_log_list.append(f"执行测试时发生未知错误: {str(e)}")
                action_res_list.append(["UNKNOWN"]*len(data[idx]["code_list"]))
        cov_dict = {
            "cov_line": 100.00 if sum(cov_line_count_list) == 0 else round((sum(covered_line_count_list) / sum(cov_line_count_list)) * 100, 2),
            "cov_line_list": cov_line_list,
        }

        return action_log_list, action_res_list, cov_dict

    def gtest_action_pair_single(self, args):
        """处理单个测试用例的子进程函数"""
        idx, test_code, data_item, pre_code, post_code, print_error, cal_cov = args
        try:
            if test_code == "代码提取错误，没有提取到有效信息":
                if print_error:
                    print(f"任务{idx+1}：代码提取错误，没有提取到有效信息")
                return (
                    idx, 
                    "代码提取错误，没有提取到有效信息", 
                    ["UNKNOWN"]*len(data_item["code_list"]),
                    [], 0, 0
                )

            action_log_temp = []
            action_res_temp = []
            cov_line_temp = []
            covered_line_count_temp = 0
            cov_line_count_temp = 0
            
            # 构建完整测试代码
            test_code_full = pre_code + test_code + post_code

            # 处理每个原始代码
            origin_code_list = data_item["code_list"]
            for origin_code in origin_code_list:
                origin_code = test_code_preprocess(origin_code, code_type="cpp")
                result_dict = self.gtest_action(origin_code, test_code_full, cal_cov=cal_cov)  # 假设这是一个独立函数
                
                action_log_temp.append(result_dict["log"])
                action_res_temp.append(result_dict["result"])
                if cal_cov:
                    cov_line_temp.append(result_dict["cov_line"])
                    covered_line_count_temp += result_dict["covered_line_count"]
                    cov_line_count_temp += result_dict["cov_line_count"]

            # 格式化日志
            s_temp = ""
            for i, action_log in enumerate(action_log_temp):
                s_temp += f"第{i+1}次测试结果:\\n" + action_log + '\\n'

            return (
                idx, 
                s_temp, 
                action_res_temp,
                cov_line_temp, 
                covered_line_count_temp, 
                cov_line_count_temp
            )
            
        except Exception as e:
            error_msg = f"执行测试时发生未知错误: {str(e)}"
            if print_error:
                print(f"任务{idx+1}：{error_msg}")
            return (
                idx, 
                error_msg, 
                ["UNKNOWN"]*len(data_item["code_list"]),
                [], 0, 0
            )

    def gtest_action_pair(self, data: list[dict], response_list: list[list[str]], print_error: bool=False) -> Tuple[List[str], List[str], Dict]:
        """
        执行回复中的单元测试代码并针对每个测试用例收集结果（多进程并行版本）
        
        参数:
            data: 数据字典列表
            response_list: 消息响应列表
            print_error: 是否打印错误信息
            
        返回:
            action_log_list: 记录每个操作过程的日志列表
            action_res_list: 每个测试结果的列表
            cov_dict: 行覆盖率字典
        """
        # 初始化存储结果的列表
        action_log_list = ["" for _ in range(len(data))]
        action_res_list = [[] for _ in range(len(data))]
        cov_line_list = [[] for _ in range(len(data))]
        cov_line_count_list = [0]*len(data)
        covered_line_count_list = [0]*len(data)
        
        # 预处理代码片段
        pre_code = '\n#include <gtest/gtest.h>\n#include <sstream>\n#include <string>\n#include <cctype>\n#include <dlfcn.h>\n#include <cstdio>\n#include <cstring>\n#include <streambuf>\n#include <iostream>\n#include <unistd.h>\n#include <sys/types.h>\n#include <sys/stat.h>\n#include <fcntl.h>\nusing namespace std;\n\ntypedef int (*RunMyCodeFunc)();\nvoid* load_library(RunMyCodeFunc& run_func) {\n    void* handle = dlopen("./build/libsource.so", RTLD_LAZY);\n    if (!handle) {\n        std::cerr << "Unable to load dynamic library: " << dlerror() << std::endl;\n        return nullptr;\n    }\n    dlerror();\n    run_func = (RunMyCodeFunc)dlsym(handle, "mycode");\n    const char* dlsym_error = dlerror();\n    if (dlsym_error) {\n        std::cerr << "Unable to find function: " << dlsym_error << std::endl;\n        dlclose(handle);\n        return nullptr;\n    }\n    return handle;\n}\n\nstd::string run_mycode(const std::string& input_str) {\n    FILE* orig_stdin = stdin;\n    FILE* orig_stdout = stdout;\n    std::streambuf* orig_cin = std::cin.rdbuf();\n    std::streambuf* orig_cout = std::cout.rdbuf();\n    \n    std::string combined_output;\n    \n    std::istringstream iss(input_str);\n    \n    int pipefd[2];\n    pipe(pipefd);\n    int stdout_copy = dup(fileno(stdout));\n    \n    std::cin.rdbuf(iss.rdbuf());\n    FILE* strin = fmemopen(const_cast<char*>(input_str.c_str()), \n                            input_str.size(), "r");\n    stdin = strin;  //\n    \n    dup2(pipefd[1], fileno(stdout));\n    close(pipefd[1]);\n    \n    std::ostringstream oss;\n    std::cout.rdbuf(oss.rdbuf());\n    \n    try {\n        RunMyCodeFunc run_func;\n        void* handle0 = load_library(run_func);\n        run_func();\n        dlclose(handle0);\n        \n        fflush(stdout);\n        std::cout.flush();\n    } catch (...) {\n        std::cin.rdbuf(orig_cin);\n        std::cout.rdbuf(orig_cout);\n        stdin = orig_stdin;\n        dup2(stdout_copy, fileno(stdout));\n        close(stdout_copy);\n        close(pipefd[0]);\n        fclose(strin);\n        throw;\n    }\n    \n    std::cin.rdbuf(orig_cin);\n    std::cout.rdbuf(orig_cout);\n    stdin = orig_stdin;\n    dup2(stdout_copy, fileno(stdout));\n    close(stdout_copy);\n    \n    combined_output = oss.str();\n    \n    close(pipefd[1]);\n    char buffer[4096];\n    ssize_t bytes_read;\n    while ((bytes_read = read(pipefd[0], buffer, sizeof(buffer))) > 0) {\n        combined_output.append(buffer, bytes_read);\n    }\n    close(pipefd[0]);\n    \n    fclose(strin);\n    \n    while (!combined_output.empty() && \n            (combined_output.back() == \'\\n\' || combined_output.back() == \'\\r\')) {\n        combined_output.pop_back();\n    }\n    \n    return combined_output;\n}\n\nstd::string format(const std::string &str) {\n    size_t end = str.find_last_not_of(" \\t\\n\\r");\n    if (end != std::string::npos) {\n        return str.substr(0, end + 1);\n    } else {\n        return "";\n    }\n}\n\n'
        post_code = '\nint main(int argc, char **argv) {\n    ::testing::InitGoogleTest(&argc, argv);\n    return RUN_ALL_TESTS();\n}\n'

        # 提取测试代码列表
        test_code_list = get_code_from_response(response_list, start_str="```cpp")
        total_tasks = len(test_code_list)
        completed_tasks = 0

        # 准备进程参数
        process_args = []
        cal_cov = True
        for idx in range(total_tasks):
            process_args.append((
                idx,
                test_code_list[idx],
                data[idx],
                pre_code,
                post_code,
                print_error,
                cal_cov
            ))

        # 使用多进程池处理
        print(f"CPU核心数:{multiprocessing.cpu_count()}")
        #max_processes = multiprocessing.cpu_count()  # 使用与CPU核心数相同的进程数
        max_processes = 6
        print(f"执行并发数:{max_processes}")
        print(f"开始处理长度为{len(response_list)}的响应列表")
        with multiprocessing.Pool(processes=max_processes) as pool:
            # 处理所有任务
            for result in pool.imap_unordered(self.gtest_action_pair_single, process_args):
                idx, log, res, cov_line, covered_count, total_count = result
                
                # 保存结果
                action_log_list[idx] = log
                action_res_list[idx] = res
                cov_line_list[idx] = cov_line
                cov_line_count_list[idx] = total_count
                covered_line_count_list[idx] = covered_count
                
                # 更新进度
                completed_tasks += 1
                print(f"已执行代码：{completed_tasks}/{total_tasks}")

        # 计算覆盖率
        total_lines = sum(cov_line_count_list)
        covered_lines = sum(covered_line_count_list)
        
        cov_dict = {
            "cov_line": 100.00 if total_lines == 0 else round((covered_lines / total_lines) * 100, 2),
            "cov_line_list": cov_line_list,
        }

        return action_log_list, action_res_list, cov_dict

    def gtest_action_pass(self, data: list[dict], response_list: List[str], print_error: bool=False) -> Tuple[List[str], List[str]]:
        """
        执行回复中的单元测试代码并收集结果
        
        参数:
            data: 字典列表
            response_list: 消息响应列表
            print_error: 是否打印错误信息
            
        返回:
            action_log_list: 记录每个操作过程的日志列表
            action_res_list: 每个测试结果的列表
        """
        action_log_list = ["" for _ in range(len(data))]
        action_res_list = [-1 for _ in range(len(data))]


        code_list = get_code_from_response(response_list, start_str="```cpp")
        total_tasks = len(code_list)
        completed_tasks = 0

        pre_code = '\n#include <gtest/gtest.h>\n#include <sstream>\n#include <string>\n#include <cctype>\n#include <dlfcn.h>\n#include <cstdio>\n#include <cstring>\n#include <streambuf>\n#include <iostream>\n#include <unistd.h>\n#include <sys/types.h>\n#include <sys/stat.h>\n#include <fcntl.h>\nusing namespace std;\n\ntypedef int (*RunMyCodeFunc)();\nvoid* load_library(RunMyCodeFunc& run_func) {\n    void* handle = dlopen("./build/libsource.so", RTLD_LAZY);\n    if (!handle) {\n        std::cerr << "Unable to load dynamic library: " << dlerror() << std::endl;\n        return nullptr;\n    }\n    dlerror();\n    run_func = (RunMyCodeFunc)dlsym(handle, "mycode");\n    const char* dlsym_error = dlerror();\n    if (dlsym_error) {\n        std::cerr << "Unable to find function: " << dlsym_error << std::endl;\n        dlclose(handle);\n        return nullptr;\n    }\n    return handle;\n}\n\nstd::string run_mycode(const std::string& input_str) {\n    FILE* orig_stdin = stdin;\n    FILE* orig_stdout = stdout;\n    std::streambuf* orig_cin = std::cin.rdbuf();\n    std::streambuf* orig_cout = std::cout.rdbuf();\n    \n    std::string combined_output;\n    \n    std::istringstream iss(input_str);\n    \n    int pipefd[2];\n    pipe(pipefd);\n    int stdout_copy = dup(fileno(stdout));\n    \n    std::cin.rdbuf(iss.rdbuf());\n    FILE* strin = fmemopen(const_cast<char*>(input_str.c_str()), \n                            input_str.size(), "r");\n    stdin = strin;  //\n    \n    dup2(pipefd[1], fileno(stdout));\n    close(pipefd[1]);\n    \n    std::ostringstream oss;\n    std::cout.rdbuf(oss.rdbuf());\n    \n    try {\n        RunMyCodeFunc run_func;\n        void* handle0 = load_library(run_func);\n        run_func();\n        dlclose(handle0);\n        \n        fflush(stdout);\n        std::cout.flush();\n    } catch (...) {\n        std::cin.rdbuf(orig_cin);\n        std::cout.rdbuf(orig_cout);\n        stdin = orig_stdin;\n        dup2(stdout_copy, fileno(stdout));\n        close(stdout_copy);\n        close(pipefd[0]);\n        fclose(strin);\n        throw;\n    }\n    \n    std::cin.rdbuf(orig_cin);\n    std::cout.rdbuf(orig_cout);\n    stdin = orig_stdin;\n    dup2(stdout_copy, fileno(stdout));\n    close(stdout_copy);\n    \n    combined_output = oss.str();\n    \n    close(pipefd[1]);\n    char buffer[4096];\n    ssize_t bytes_read;\n    while ((bytes_read = read(pipefd[0], buffer, sizeof(buffer))) > 0) {\n        combined_output.append(buffer, bytes_read);\n    }\n    close(pipefd[0]);\n    \n    fclose(strin);\n    \n    while (!combined_output.empty() && \n            (combined_output.back() == \'\\n\' || combined_output.back() == \'\\r\')) {\n        combined_output.pop_back();\n    }\n    \n    return combined_output;\n}\n\nstd::string format(const std::string &str) {\n    size_t end = str.find_last_not_of(" \\t\\n\\r");\n    if (end != std::string::npos) {\n        return str.substr(0, end + 1);\n    } else {\n        return "";\n    }\n}\n\n'
        post_code = '\nint main(int argc, char **argv) {\n    ::testing::InitGoogleTest(&argc, argv);\n    return RUN_ALL_TESTS();\n}\n'

        # 准备进程参数
        process_args = []
        for idx in range(total_tasks):
            data[idx]["code_list"]=[code_list[idx]]
            task_cases = data[idx].get('test_cases', None)
            test_code = 'TEST(MyCodeTest, AllCases) {\n'
            for case_idx, test_case in enumerate(task_cases):
                case_input = test_case['input'].replace('\n','\\n')
                case_output = test_case['output'].replace('\n','\\n')
                test_code += f'    EXPECT_EQ(format(run_mycode("{case_input}")), format("{case_output}"));\n'
            test_code += "}\n"
            cal_cov=False
            process_args.append((
                idx,
                test_code,
                data[idx],
                pre_code,
                post_code,
                print_error,
                cal_cov
            ))

        # 使用多进程池处理
        print(f"CPU核心数:{multiprocessing.cpu_count()}")
        #max_processes = multiprocessing.cpu_count()  # 使用与CPU核心数相同的进程数
        max_processes = 8
        print(f"执行并发数:{max_processes}")
        print(f"开始处理长度为{len(response_list)}的响应列表")
        with multiprocessing.Pool(processes=max_processes) as pool:
            # 处理所有任务
            for result in pool.imap_unordered(self.gtest_action_pair_single, process_args):
                idx, log, res, cov_line, covered_count, total_count = result
                
                # 保存结果
                action_log_list[idx] = log
                action_res_list[idx] = res[0]
                
                # 更新进度
                completed_tasks += 1
                print(f"已执行代码：{completed_tasks}/{total_tasks}")
                    
        
        return action_log_list, action_res_list
