import sys
sys.path.append('./friendly_core')
import numpy as np
from avg_algos import FC_Avg_UnknownDiam
from numpy import matmul, transpose
from sklearn.utils.extmath import randomized_svd
import gc
import random



def generate_random_vector(d):
    v = []
    for i in range(d):
        x = random.random()
        if x < 0.5:
            v.append(-1)
        else:
            v.append(1)
    return np.array(v)



# Computes V V^T x
def compute_projected_vector(Vt: np.ndarray, x: np.ndarray):
    V = transpose(Vt)
    y = matmul(Vt, x)
    z = matmul(V, y)
    return z


def create_projections(dataset: np.ndarray, k: int, num_of_projections: int):
    n = len(dataset)
    pi = [i for i in range(n)]
    np.random.shuffle(pi)

    m = n // num_of_projections
    L = []
    for i in range(num_of_projections):
        X = np.array([dataset[pi[j]] for j in range(m * i, m * (i + 1))])
        _, _, Vt = randomized_svd(X, n_components=k)
        if (i + 1) % 10 == 0:
            print("[FriendlyCore] Created projection %d" % (i + 1))
            #print("Matrix X Dimensions: %d x %d" % (len(X), len(X[0])))
            #print("Matrix VT Dimensions: %d x %d" % (len(Vt), len(Vt[0])))
        L.append(Vt)
        gc.collect()

    gc.collect()
    print("[FriendlyCore] num_of_projections = %d" % len(L))
    return L




'''
EstSubspace: Our main (rho,delta)-zCDP subspace estimator.
Given a database of unit norm points in R^d and a parameter k, the algorithm outputs an k x d  Vt matrix such that 
the d x d matrix Pi = V * Vt, for V = Vt^T, estimates the projection matrix onto the top-k subspace of the database.

Inputs:
database: Database of unit norm points in R^d.
d: Dimension.
k: Low dimension
rho, delta: Privacy parameters.
beta: Confidence parameter (utility parameter of FC_Avg_UnknownDiam).
r_min: Lower bound on the averaging diameter (utility parameter of FC_Avg_UnknownDiam).
r_max: Upper bound on the averaging diameter (utility parameter of FC_Avg_UnknownDiam).
num_of_pieces: the number of pieces that database is splitted into (from each piece, k-rank projection matrix is generated).
'''

def EstSubspace(dataset: np.ndarray, d: int, k: int, rho: float, delta: float, beta: float, r_min: float, r_max: float,
                  t: int):
    Vt_list = create_projections(dataset, k, t)
    q = 20*k
    Ref = np.array([generate_random_vector(d) for i in range(q)])
    Ref = np.array([Ref[j, :, None] for j in range(q)])
    Y = []
    for i in range(t):
        gc.collect()
        Yi = np.array([])
        for j in range(q):
            X = compute_projected_vector(Vt = Vt_list[i], x = Ref[j])
            X = transpose(X)[0]
            Yi = np.concatenate((Yi, X))
        Y.append(Yi)
    Y = np.array(Y)
    print("Y.shape = ")
    print(Y.shape)
    Z = FC_Avg_UnknownDiam(Y, q*d, rho, delta, beta, r_min, r_max)
    Z = np.array(Z)
    tP = np.array([Z[d*i:d*(i+1)] for i in range(q)])
    _, _, Vt = randomized_svd(tP, n_components=k)

    return Vt

