import numpy as np
from subspace_algos import EstSubspace, compute_projected_vector, generate_random_vector
from sklearn.utils.extmath import randomized_svd
from dist_utils import l2_norm_squared
from dwork_subspace import Dwork_Subspace

def L2(est):
    return np.linalg.norm(est)

def gen_samples(n, d, k, gap_factor):
    basis = []
    for i in range(k):
        v = generate_random_vector(d)
        basis.append(v)
    basis = np.array(basis)
    _, _, Vt = randomized_svd(basis, n_components=k)
    dataset = []
    for i in range(n):
        sample = generate_random_vector(d)
        x = compute_projected_vector(Vt, sample)
        x = x/np.sqrt(l2_norm_squared(x))
        noise = generate_random_vector(d)/gap_factor
        x = x + noise
        x = x/np.sqrt(l2_norm_squared(x))
        dataset.append(x)
        if (i + 1) % 200 == 0:
            print("Created %d samples" % (i + 1))
    return np.array(dataset)



if __name__ == '__main__':
    d = 8000
    k = 4
    t = 125
    n = 2*k*t
    gap_factor = 10*d
    rho = 2

    dataset = gen_samples(n, d, k, gap_factor = 10*d)
    _, s, _ = randomized_svd(dataset, n_components=k+1)
    print("Squared Singular Values of the dataset:")
    print(s*s)
    print("Ratio sigma_{k+1}/sigma_k: %f" % float(s[k]/s[k-1]))


    print("Computing the projection (EstSubspace):")
    Vt1 = EstSubspace(dataset, d, k, rho=rho/2, delta=1e-5, beta=0.05, r_min=1e-6, r_max=100, t=t)
    print("Computing the projection (Dwork et al. 2014):")
    Vt2 = Dwork_Subspace(dataset, d, k, rho=rho/2, delta = 1e-5)

    mean = np.mean(dataset, axis=0)
    subspace_mean_tmp = mean + np.random.normal(0, scale=np.sqrt(4 / rho) / n, size=d)
    est_subspace_mean = compute_projected_vector(Vt1, subspace_mean_tmp)
    dwork_mean = compute_projected_vector(Vt2, subspace_mean_tmp)

    print("EstSubspace error: %0.4f" % L2(est_subspace_mean - mean))
    print("Dwork et al. (2014) error: %0.4f" % L2(dwork_mean - mean))





