# Copyright 2024 Bytedance Ltd. and/or its affiliates
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

try:
    from math_verify.errors import TimeoutException
    from math_verify.metric import math_metric
    from math_verify.parser import ExprExtractionConfig, LatexExtractionConfig
except ImportError:
    print("To use Math-Verify, please install it first by running `pip install math-verify`.")
import math
def cosfn(t, T, min_value, max_value):
    return max_value - (max_value - min_value) * (1 - math.cos(t * math.pi / T)) / 2

def get_len_penalty_reward(
    is_correct:bool,
    gen_len:int,
    cosine_min_len_value_wrong: float = -0.2,
    cosine_max_len_value_wrong: float = 0.0,
    cosine_min_len_value_correct: float = 1.2,
    cosine_max_len_value_correct: float = 1.0,
    cosine_max_len: int = 30000,
    cosine_min_len: int = 3000,
):
    if is_correct:
        min_value = cosine_max_len_value_correct
        max_value = cosine_min_len_value_correct
    else:
        min_value = cosine_max_len_value_wrong
        max_value = cosine_min_len_value_wrong
    reward=0
    if gen_len > cosine_max_len:
        reward = min_value
    elif gen_len < cosine_min_len:
        reward=max_value
    else:
        reward = cosfn(gen_len, cosine_max_len, min_value, max_value)
   
    return reward

def compute_score(
                model_output: str,
                ground_truth: str, 
                timeout_score: float = 0,
                ):
    verify_func = math_metric(
        gold_extraction_target=(LatexExtractionConfig(),),
        pred_extraction_target=(ExprExtractionConfig(), LatexExtractionConfig()),
    )
    ret_score = 0.0
    ground_truth_boxed = ground_truth 
    # Wrap the ground truth in \boxed{} format for verification
    if not ground_truth.startswith("\\boxed{"):
        ground_truth_boxed = "\\boxed{" + ground_truth + "}"
    
    print("\n" + "="*80)
    print(" Processing New Sample ".center(80, '='))
    print(f"[Ground Truth]: {ground_truth_boxed}")
    print(f"[Model Response](last 200 char):\n {model_output[-200:]}")
    try:
        ret_score, _ = verify_func([ground_truth_boxed], [model_output])
    except Exception:
        pass
    except TimeoutException:
        print("TimeoutException")
        ret_score = timeout_score
    is_correct=ret_score>=1
    gen_len=len(model_output)
    reward=get_len_penalty_reward(is_correct,gen_len)
    print(f"[Cosine Penalty] is_correct:{is_correct},gen_len:{gen_len},reward {reward}")
    print("="*80 + "\n")
    return reward
