from functools import reduce

import torch
import torch.nn as nn
import torch.utils.checkpoint as checkpoint
from timm.models.layers import DropPath, to_2tuple, trunc_normal_
from .linear_bayesian_layer import BayesianLinear, BayesianModule
from .weight_sampler import TrainableRandomDistribution, PriorWeightDistribution

from einops import rearrange, repeat


class BayesianMlp(nn.Module):
    def __init__(self, in_features, hidden_features=None, out_features=None, act_layer=nn.GELU, drop=0.):
        super().__init__()
        out_features = out_features or in_features
        hidden_features = hidden_features or in_features
        self.fc1 = BayesianLinear(in_features, hidden_features)
        self.act = act_layer()
        self.fc2 = BayesianLinear(hidden_features, out_features)
        self.drop = nn.Dropout(drop)

    def forward(self, x):
        x = self.fc1(x)
        x = self.act(x)
        x = self.drop(x)
        x = self.fc2(x)
        x = self.drop(x)
        return x


def window_partition_4d(x, window_size):
    """
    Args:
        x: (B, H_q, W_q, H_s, W_s, C)
        window_size (int): window size
    Returns:
        x: (num_windows*B, window_size, window_size, window_size, window_size, C)
    """
    kwargs = {f'W_{i}':window_size for i in range(1, 5)}
    x = rearrange(x, 'B (H_q W_1) (W_q W_2) (H_s W_3) (W_s W_4) C -> (B H_q W_q H_s W_s) W_1 W_2 W_3 W_4 C', **kwargs)
    return x


def window_reverse_4d(windows, window_size, H_q, W_q, H_s, W_s):
    """
    Args:
        windows: (num_windows*B, window_size, window_size, window_size, window_size, C)
        window_size (int): size of window
        H_q (int): Height of query image
        W_q (int): Width of query image
        H_s (int): Height of support image
        W_s (int): Width of support image
    Returns:
        x: (B, H_q, W_q, H_s, W_s, C)
    """
    kwargs = {
        'H_q': H_q // window_size,
        'W_q': W_q // window_size,
        'H_s': H_s // window_size,
        'W_s': W_s // window_size
    }
    x = rearrange(windows, '(B H_q W_q H_s W_s) W_1 W_2 W_3 W_4 C -> B (H_q W_1) (W_q W_2) (H_s W_3) (W_s W_4) C', **kwargs)
    return x


class BayesianAttention(BayesianModule):
    def __init__(self, dim, window_size, num_heads, qkv_bias=True, qk_scale=None, attn_drop=0., proj_drop=0.,
                 prior_sigma_1 = 0.1,
                 prior_sigma_2 = 0.4,
                 prior_pi = 1,
                 posterior_mu_init = 0,
                 posterior_rho_init = -7.0,
                 freeze = False,
                 prior_dist = None):

        super().__init__()
        self.dim = dim
        self.window_size = window_size  # Wh, Ww
        self.num_heads = num_heads
        head_dim = dim // num_heads
        self.scale = qk_scale or head_dim ** -0.5

        self.bias = qkv_bias
        self.freeze = freeze

        self.posterior_mu_init = posterior_mu_init
        self.posterior_rho_init = posterior_rho_init

        #parameters for the scale mixture prior
        self.prior_sigma_1 = prior_sigma_1
        self.prior_sigma_2 = prior_sigma_2
        self.prior_pi = prior_pi
        self.prior_dist = prior_dist

        # Variational weight parameters and sample
        sparse = nn.Parameter(torch.eye((2 * window_size[0] - 1) * (2 * window_size[0] - 1) * (2 * window_size[0] - 1) * (2 * window_size[0] - 1), num_heads))
        self.weight_mu = nn.Parameter(torch.Tensor((2 * window_size[0] - 1) * (2 * window_size[0] - 1) * (2 * window_size[0] - 1) * (2 * window_size[0] - 1), num_heads).normal_(posterior_mu_init, 0.1) * sparse) 
        self.weight_rho = nn.Parameter(torch.Tensor((2 * window_size[0] - 1) * (2 * window_size[0] - 1) * (2 * window_size[0] - 1) * (2 * window_size[0] - 1), num_heads).normal_(posterior_rho_init, 0.1) * sparse)
        self.weight_sampler = TrainableRandomDistribution(self.weight_mu, self.weight_rho)

        # Priors (as BBP paper)
        self.weight_prior_dist = PriorWeightDistribution(self.prior_pi, self.prior_sigma_1, self.prior_sigma_2, dist = self.prior_dist)
        self.log_prior = 0
        self.log_variational_posterior = 0

        relative_position_index = self.relative_bias_4d(window_size)
        self.register_buffer("relative_position_index", relative_position_index)

        self.qkv = BayesianLinear(dim, dim * 3, bias=qkv_bias)
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = BayesianLinear(dim, dim)
        self.proj_drop = nn.Dropout(proj_drop)

        self.softmax = nn.Softmax(dim=-1)

    def relative_bias_4d(self, window_size=(5, 5, 5, 5)):
        assert len(window_size) == 4, f'Invalid window size {window_size}'

        coords_H_q = torch.arange(window_size[0])
        coords_W_q = torch.arange(window_size[1])
        coords_H_s = torch.arange(window_size[2])
        coords_W_s = torch.arange(window_size[3])
        coords = torch.stack(torch.meshgrid((coords_H_q, coords_W_q, coords_H_s, coords_W_s))) # 4, H_q, W_q, H_s, W_s

        coords_flatten = rearrange(coords, 'd H_q W_q H_s W_s -> d (H_q W_q H_s W_s)')
        relative_coords = coords_flatten[:, :, None] - coords_flatten[:, None, :]  # 2, H_q*W_q*H_s*W_s, H_q*W_q*H_s*W_s
        relative_coords = rearrange(relative_coords, 'd q k -> q k d')

        relative_coords[..., 0] += window_size[0] - 1
        relative_coords[..., 1] += window_size[1] - 1
        relative_coords[..., 2] += window_size[2] - 1
        relative_coords[..., 3] += window_size[3] - 1

        relative_coords[..., 0] *= (2 * window_size[0] - 1) ** 3
        relative_coords[..., 1] *= (2 * window_size[1] - 1) ** 2
        relative_coords[..., 2] *= 2 * window_size[2] - 1

        relative_position_index = relative_coords.sum(-1)

        return relative_position_index

    def forward(self, x, mask=None):
        """
        Args:
            x: input features with shape of (num_windows*B, N, C)
            mask: (0/-inf) mask with shape of (num_windows, Wh*Ww, Wh*Ww) or None
        """
        #if the model is frozen, return frozen
        if self.freeze:
            return self.forward_frozen(x, mask)

        relative_position_bias_table = self.weight_sampler.sample()

        # Get the complexity cost
        self.log_variational_posterior = self.weight_sampler.log_posterior()
        self.log_prior = self.weight_prior_dist.log_prior(relative_position_bias_table)

        B_, N, C = x.shape
        qkv = self.qkv(x).reshape(B_, N, 3, self.num_heads, C // self.num_heads).permute(2, 0, 3, 1, 4)
        q, k, v = qkv[0], qkv[1], qkv[2]  # make torchscript happy (cannot use tensor as tuple)

        q = q * self.scale
        attn = (q @ k.transpose(-2, -1))

        relative_position_bias = relative_position_bias_table[self.relative_position_index.view(-1)].view(
            reduce(lambda x, y: x * y, self.window_size), reduce(lambda x, y: x * y, self.window_size), -1)  # H_q*W_q*H_s*W_s,H_q*W_q*H_s*W_s,nH
        relative_position_bias = relative_position_bias.permute(2, 0, 1).contiguous()  # nH, H_q*W_q*H_s*W_s, H_q*W_q*H_s*W_s
        attn = attn + relative_position_bias.unsqueeze(0)

        if mask is not None:
            nW = mask.shape[0]
            attn = attn.view(B_ // nW, nW, self.num_heads, N, N) + mask.unsqueeze(1).unsqueeze(0)
            attn = attn.view(-1, self.num_heads, N, N)
            attn = self.softmax(attn)
        else:
            attn = self.softmax(attn)

        attn = self.attn_drop(attn)

        x = (attn @ v).transpose(1, 2).reshape(B_, N, C)
        x = self.proj(x)
        x = self.proj_drop(x)
        return x

    def forward_frozen(self, x, mask=None):
        relative_position_bias_table = self.weight_mu
        B_, N, C = x.shape
        qkv = self.qkv(x).reshape(B_, N, 3, self.num_heads, C // self.num_heads).permute(2, 0, 3, 1, 4)
        q, k, v = qkv[0], qkv[1], qkv[2]  # make torchscript happy (cannot use tensor as tuple)

        q = q * self.scale
        attn = (q @ k.transpose(-2, -1))

        relative_position_bias = relative_position_bias_table[self.relative_position_index.view(-1)].view(
            reduce(lambda x, y: x * y, self.window_size), reduce(lambda x, y: x * y, self.window_size), -1)  # H_q*W_q*H_s*W_s,H_q*W_q*H_s*W_s,nH
        relative_position_bias = relative_position_bias.permute(2, 0, 1).contiguous()  # nH, H_q*W_q*H_s*W_s, H_q*W_q*H_s*W_s
        attn = attn + relative_position_bias.unsqueeze(0)

        if mask is not None:
            nW = mask.shape[0]
            attn = attn.view(B_ // nW, nW, self.num_heads, N, N) + mask.unsqueeze(1).unsqueeze(0)
            attn = attn.view(-1, self.num_heads, N, N)
            attn = self.softmax(attn)
        else:
            attn = self.softmax(attn)

        attn = self.attn_drop(attn)

        x = (attn @ v).transpose(1, 2).reshape(B_, N, C)
        x = self.proj(x)
        x = self.proj_drop(x)
        return x

    def extra_repr(self) -> str:
        return f'dim={self.dim}, window_size={self.window_size}, num_heads={self.num_heads}'


class BayesianTransformerBlock(nn.Module):
    def __init__(self, dim, input_resolution, num_heads, window_size=7, shift_size=0,
                 mlp_ratio=4., qkv_bias=True, qk_scale=None, drop=0., attn_drop=0., drop_path=0.,
                 act_layer=nn.GELU, norm_layer=nn.LayerNorm):
        super().__init__()
        self.dim = dim
        self.input_resolution = input_resolution
        self.num_heads = num_heads
        self.window_size = window_size
        self.shift_size = shift_size
        self.mlp_ratio = mlp_ratio
        if min(self.input_resolution) <= self.window_size:
            # if window size is larger than input resolution, we don't partition windows
            self.shift_size = 0
            self.window_size = min(self.input_resolution)
        assert 0 <= self.shift_size < self.window_size, "shift_size must in 0-window_size"

        self.norm1 = norm_layer(dim)
        self.attn = BayesianAttention(
            dim, window_size=(self.window_size,)*4, num_heads=num_heads,
            qkv_bias=qkv_bias, qk_scale=qk_scale, attn_drop=attn_drop, proj_drop=drop)

        self.drop_path = DropPath(drop_path) if drop_path > 0. else nn.Identity()
        self.norm2 = norm_layer(dim)
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = BayesianMlp(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop)

        if self.shift_size > 0:
            # calculate attention mask for SW-MSA
            H_q, W_q, H_s, W_s = self.input_resolution
            img_mask = torch.zeros((1, H_q, W_q, H_s, W_s, 1))  # 1 H W 1
            slices = (slice(0, -self.window_size),
                        slice(-self.window_size, -self.shift_size),
                        slice(-self.shift_size, None))
            cnt = 0
            for h_q in slices:
                for w_q in slices:
                    for h_s in slices:
                        for w_s in slices:
                            img_mask[:, h_q, w_q, h_s, w_s, :] = cnt
                            cnt += 1

            mask_windows = window_partition_4d(img_mask, self.window_size)
            # nW, window_size, window_size, window_size, window_size, 1
            mask_windows = mask_windows.view(-1, self.window_size ** 4)
            attn_mask = mask_windows.unsqueeze(1) - mask_windows.unsqueeze(2)
            attn_mask = attn_mask.masked_fill(attn_mask != 0, float(-100.0)).masked_fill(attn_mask == 0, float(0.0))
        else:
            attn_mask = None

        self.register_buffer("attn_mask", attn_mask)

    def forward(self, x):
        H_q, W_q, H_s, W_s = self.input_resolution
        B, L, C = x.shape
        assert L == H_q * W_q * H_s * W_s, "input feature has wrong size"

        shortcut = x
        x = self.norm1(x)
        x = rearrange(x, 'B (H_q W_q H_s W_s) C -> B H_q W_q H_s W_s C', H_q=H_q, W_q=W_q, H_s=H_s, W_s=W_s)

        # cyclic shift
        if self.shift_size > 0:
            shifted_x = torch.roll(x, shifts=(-self.shift_size,)*4, dims=(1, 2, 3, 4))
        else:
            shifted_x = x

        # partition windows
        x_windows = window_partition_4d(shifted_x, self.window_size)  # nW*B, window_size, window_size, window_size, window_size, C
        x_windows = rearrange(x_windows, 'B W_1 W_2 W_3 W_4 C -> B (W_1 W_2 W_3 W_4) C')

        # W-MSA/SW-MSA
        attn_windows = self.attn(x_windows, mask=self.attn_mask)  # nW*B, window_size*window_size, C

        # merge windows
        kwargs = {f'W_{i}':self.window_size for i in range(1, 5)}
        attn_windows = rearrange(attn_windows, 'B (W_1 W_2 W_3 W_4) C -> B W_1 W_2 W_3 W_4 C', **kwargs)
        shifted_x = window_reverse_4d(attn_windows, self.window_size, H_q=H_q, W_q=W_q, H_s=H_s, W_s=W_s)  # B H' W' C

        # reverse cyclic shift
        if self.shift_size > 0:
            x = torch.roll(shifted_x, shifts=(self.shift_size,)*4, dims=(1, 2, 3, 4))
        else:
            x = shifted_x
        x = rearrange(x, 'B H_q W_q H_s W_s C -> B (H_q W_q H_s W_s) C')

        # FFN
        x = shortcut + self.drop_path(x)
        x = x + self.drop_path(self.mlp(self.norm2(x)))

        return x

    def extra_repr(self) -> str:
        return f"dim={self.dim}, input_resolution={self.input_resolution}, num_heads={self.num_heads}, " \
               f"window_size={self.window_size}, shift_size={self.shift_size}, mlp_ratio={self.mlp_ratio}"


class BasicBayeisanLayer(nn.Module):
    def __init__(self, dim, input_resolution, depth, num_heads, window_size,
                 mlp_ratio=4., qkv_bias=True, qk_scale=None, drop=0., attn_drop=0.,
                 drop_path=0., norm_layer=nn.LayerNorm, downsample=None, use_checkpoint=False):

        super().__init__()
        self.dim = dim
        self.input_resolution = input_resolution
        self.depth = depth
        self.use_checkpoint = use_checkpoint

        # build blocks
        self.blocks = nn.ModuleList([
            BayesianTransformerBlock(dim=dim, input_resolution=input_resolution,
                                 num_heads=num_heads, window_size=window_size,
                                 shift_size=0 if (i % 2 == 0) else window_size // 2,
                                 mlp_ratio=mlp_ratio,
                                 qkv_bias=qkv_bias, qk_scale=qk_scale,
                                 drop=drop, attn_drop=attn_drop,
                                 drop_path=drop_path[i] if isinstance(drop_path, list) else drop_path,
                                 norm_layer=norm_layer)
            for i in range(depth)])

        # patch merging layer
        if downsample is not None:
            self.downsample = downsample(input_resolution, dim=dim, norm_layer=norm_layer)
        else:
            self.downsample = None

    def forward(self, x):
        for blk in self.blocks:
            if self.use_checkpoint:
                x = checkpoint.checkpoint(blk, x)
            else:
                x = blk(x)
        if self.downsample is not None:
            x = self.downsample(x)
        return x

    def extra_repr(self) -> str:
        return f"dim={self.dim}, input_resolution={self.input_resolution}, depth={self.depth}"


class PatchEmbed(nn.Module):
    def __init__(self, img_size=224, patch_size=4, in_chans=3, embed_dim=96, norm_layer=None):
        super().__init__()
        img_size = to_2tuple(img_size)
        patch_size = to_2tuple(patch_size)
        patches_resolution = [img_size[0] // patch_size[0], img_size[1] // patch_size[1]]
        self.img_size = img_size
        self.patch_size = patch_size
        self.patches_resolution = patches_resolution
        self.num_patches = patches_resolution[0] * patches_resolution[1]

        self.in_chans = in_chans
        self.embed_dim = embed_dim

        self.proj = nn.Conv2d(in_chans, embed_dim, kernel_size=patch_size, stride=patch_size)
        if norm_layer is not None:
            self.norm = norm_layer(embed_dim)
        else:
            self.norm = None

    def forward(self, x):
        B, C, H, W = x.shape
        # FIXME look at relaxing size constraints
        assert H == self.img_size[0] and W == self.img_size[1], \
            f"Input image size ({H}*{W}) doesn't match model ({self.img_size[0]}*{self.img_size[1]})."
        x = self.proj(x).flatten(2).transpose(1, 2)  # B Ph*Pw C
        if self.norm is not None:
            x = self.norm(x)
        return x


class BayesianTransformer4d(nn.Module):
    def __init__(self, corr_size=(16, 16, 16, 16),
                 embed_dim=64, depth=2, num_head=4,
                 window_size=4, mlp_ratio=4., qkv_bias=True, qk_scale=None,
                 drop_rate=0., attn_drop_rate=0., drop_path_rate=0.,
                 norm_layer=nn.LayerNorm,
                 use_checkpoint=False):
        super().__init__()
        self.corr_size = corr_size
        self.embed_dim = embed_dim
        self.mlp_ratio = mlp_ratio
        self.pos_drop = nn.Dropout(p=drop_rate)

        self.query_pos_embed = nn.Parameter(torch.zeros(1, embed_dim//2, *corr_size[:-2], 1, 1))
        self.supp_pos_embed = nn.Parameter(torch.zeros(1, embed_dim//2, 1, 1, *corr_size[-2:]))
        trunc_normal_(self.query_pos_embed, std=.02)
        trunc_normal_(self.supp_pos_embed, std=.02)

        # stochastic depth
        dpr = [x.item() for x in torch.linspace(0, drop_path_rate, depth)]  # stochastic depth decay rule

        # build layers
        self.layer = BasicBayeisanLayer(dim=embed_dim,
                            input_resolution=corr_size,
                            depth=depth,
                            num_heads=num_head,
                            window_size=window_size,
                            mlp_ratio=self.mlp_ratio,
                            qkv_bias=qkv_bias, qk_scale=qk_scale,
                            drop=drop_rate, attn_drop=attn_drop_rate,
                            drop_path=dpr,
                            norm_layer=norm_layer,
                            downsample=None,
                            use_checkpoint=use_checkpoint)

        self.norm = norm_layer(self.embed_dim)

        self.apply(self._init_weights)

    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            trunc_normal_(m.weight, std=.02)
            if isinstance(m, nn.Linear) and m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, nn.LayerNorm):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)

    @torch.jit.ignore
    def no_weight_decay_keywords(self):
        return {'relative_position_bias_table'}

    def forward(self, x):
        query_pos_embed = repeat(self.query_pos_embed, 'B L H_q W_q a b -> B L H_q W_q (H_s a) (W_s b)', H_s=self.corr_size[2], W_s=self.corr_size[3])
        supp_pos_embed = repeat(self.supp_pos_embed, 'B L a b H_s W_s -> B L (H_q a) (W_q b) H_s W_s', H_q=self.corr_size[0], W_q=self.corr_size[1])
        pos_embed = torch.cat((query_pos_embed, supp_pos_embed), dim=1)
        x += rearrange(pos_embed, 'B L H_q W_q H_s W_s -> B (H_q W_q H_s W_s) L')
        x = self.pos_drop(x)
        return self.layer(x)


class TransformerWarper4d(nn.Module):
    def __init__(self, transformer):
        super().__init__()
        self.transformer = transformer

    def forward(self, x):
        _, _, H_q, W_q, H_s, W_s = x.size()
        x = rearrange(x, 'B L H_q W_q H_s W_s -> B (H_q W_q H_s W_s) L')
        x = self.transformer(x)
        x = rearrange(x, 'B (H_q W_q H_s W_s) L -> B L H_q W_q H_s W_s', H_q=H_q, W_q=W_q, H_s=H_s, W_s=W_s)
        return x