"""
Prompt management system with YAML templates and experiment tracking.

This module provides a simple but robust system for managing prompts across
reasoning scaffolds, with version control and experiment tracking capabilities.
"""

import os
import yaml
import json
from pathlib import Path
from datetime import datetime
from typing import Dict, Any, Optional, List
from dataclasses import dataclass, asdict
from jinja2 import Template, Environment, FileSystemLoader
import subprocess


@dataclass
class PromptTemplate:
    """
    A prompt template with metadata and versioning.
    """
    # Template content
    system_prompt: str
    user_prompt_template: str
    
    # Metadata
    version: str
    created: str
    description: str
    scaffold_type: str = "general"
    
    # Tracking information
    git_hash: Optional[str] = None
    experiment_id: Optional[str] = None
    author: Optional[str] = None
    
    # Template variables and examples
    template_variables: Optional[Dict[str, str]] = None
    example_usage: Optional[Dict[str, Any]] = None
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary for serialization."""
        return asdict(self)
    
    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'PromptTemplate':
        """Create from dictionary."""
        return cls(**data)
    
    def render(self, **kwargs) -> str:
        """
        Render the user prompt template with provided variables.
        
        Args:
            **kwargs: Template variables
            
        Returns:
            Rendered prompt string
        """
        template = Template(self.user_prompt_template)
        return template.render(**kwargs)


class PromptManager:
    """
    Manager for prompt templates with YAML-based storage and version control.
    
    This class provides a simple but robust system for managing prompts
    across different reasoning scaffolds while maintaining experiment tracking.
    """
    
    def __init__(self, templates_dir: Optional[str] = None):
        """
        Initialize prompt manager.
        
        Args:
            templates_dir: Directory containing prompt template YAML files.
                          If None, will automatically find templates relative to this module.
        """
        if templates_dir is None:
            # Automatically find templates directory relative to this file
            current_file = Path(__file__).resolve()
            # This file is in src/reasoning_frameworks/prompts/manager.py
            # Templates are in src/reasoning_frameworks/prompts/templates/
            templates_dir = current_file.parent / "templates"
        
        self.templates_dir = Path(templates_dir)
        self.templates_dir.mkdir(parents=True, exist_ok=True)
        
        # Set up Jinja2 environment for template rendering
        self.env = Environment(
            loader=FileSystemLoader(str(self.templates_dir)),
            trim_blocks=True,
            lstrip_blocks=True
        )
        
        # Cache for loaded templates
        self._template_cache: Dict[str, PromptTemplate] = {}
    
    def load_template(
        self,
        template_name: str,
        version: str = "latest"
    ) -> PromptTemplate:
        """
        Load a prompt template from YAML file.
        
        Args:
            template_name: Name of the template (without extension)
            version: Version to load ("latest" or specific version)
            
        Returns:
            Loaded prompt template
        """
        cache_key = f"{template_name}_{version}"
        
        # Check cache first
        if cache_key in self._template_cache:
            return self._template_cache[cache_key]
        
        # Determine file path
        if version == "latest":
            # Find the latest version
            template_files = list(self.templates_dir.glob(f"{template_name}_v*.yaml"))
            if not template_files:
                # Try without version
                template_file = self.templates_dir / f"{template_name}.yaml"
                if not template_file.exists():
                    raise FileNotFoundError(f"Template {template_name} not found")
            else:
                # Sort by version and get latest
                template_files.sort(key=lambda x: x.stem.split('_v')[-1])
                template_file = template_files[-1]
        else:
            template_file = self.templates_dir / f"{template_name}_v{version}.yaml"
            if not template_file.exists():
                raise FileNotFoundError(f"Template {template_name} version {version} not found")
        
        # Load YAML file
        with open(template_file, 'r') as f:
            template_data = yaml.safe_load(f)
        
        # Add metadata
        template_data['metadata'] = template_data.get('metadata', {})
        template_data['metadata']['loaded_at'] = datetime.now().isoformat()
        template_data['metadata']['template_file'] = str(template_file)
        
        # Create template object
        template = PromptTemplate.from_dict(template_data)
        
        # Cache and return
        self._template_cache[cache_key] = template
        return template
    
    def load_prompt_collection(
        self,
        collection_name: str,
        version: str = "latest"
    ) -> Dict[str, Any]:
        """
        Load a collection of prompts from YAML file.
        
        This method loads YAML files that contain multiple prompts and metadata,
        returning the raw dictionary structure for access like prompts['vlm_description_prompt'].
        
        Args:
            collection_name: Name of the prompt collection (without extension)
            version: Version to load ("latest" or specific version)
            
        Returns:
            Dictionary containing all prompts and metadata
        """
        cache_key = f"collection_{collection_name}_{version}"
        
        # Check cache first
        if cache_key in self._template_cache:
            return self._template_cache[cache_key]
        
        # Determine file path
        if version == "latest":
            # Find the latest version
            template_files = list(self.templates_dir.glob(f"{collection_name}_v*.yaml"))
            if not template_files:
                # Try without version
                template_file = self.templates_dir / f"{collection_name}.yaml"
                if not template_file.exists():
                    raise FileNotFoundError(f"Prompt collection {collection_name} not found")
            else:
                # Sort by version and get latest
                template_files.sort(key=lambda x: x.stem.split('_v')[-1])
                template_file = template_files[-1]
        else:
            template_file = self.templates_dir / f"{collection_name}_v{version}.yaml"
            if not template_file.exists():
                raise FileNotFoundError(f"Prompt collection {collection_name} version {version} not found")
        
        # Load YAML file
        with open(template_file, 'r') as f:
            collection_data = yaml.safe_load(f)
        
        # Add metadata
        collection_data['metadata'] = collection_data.get('metadata', {})
        collection_data['metadata']['loaded_at'] = datetime.now().isoformat()
        collection_data['metadata']['template_file'] = str(template_file)
        
        # Cache and return
        self._template_cache[cache_key] = collection_data
        return collection_data
    
    def save_template(
        self,
        template: PromptTemplate,
        template_name: str,
        auto_version: bool = True
    ) -> str:
        """
        Save a prompt template to YAML file.
        
        Args:
            template: Template to save
            template_name: Name for the template file
            auto_version: Automatically increment version number
            
        Returns:
            Path to saved template file
        """
        # Auto-increment version if requested
        if auto_version:
            existing_versions = self._get_existing_versions(template_name)
            if existing_versions:
                latest_version = max(existing_versions)
                new_version = f"v{int(latest_version[1:]) + 1}"
            else:
                new_version = "v1"
            template.version = new_version
            filename = f"{template_name}_{new_version}.yaml"
        else:
            filename = f"{template_name}.yaml"
        
        template_file = self.templates_dir / filename
        
        # Add git information if available
        template.git_hash = self._get_git_hash()
        template.created = datetime.now().isoformat()
        
        # Save to YAML
        template_data = template.to_dict()
        with open(template_file, 'w') as f:
            yaml.dump(template_data, f, default_flow_style=False, indent=2)
        
        return str(template_file)
    
    def create_template(
        self,
        template_name: str,
        system_prompt: str,
        user_prompt_template: str,
        description: str,
        scaffold_type: str = "general",
        template_variables: Optional[Dict[str, str]] = None,
        example_usage: Optional[Dict[str, Any]] = None
    ) -> PromptTemplate:
        """
        Create a new prompt template.
        
        Args:
            template_name: Name for the template
            system_prompt: System prompt text
            user_prompt_template: User prompt template with Jinja2 variables
            description: Description of the template
            scaffold_type: Type of scaffold this template is for
            template_variables: Documentation of template variables
            example_usage: Example of how to use the template
            
        Returns:
            Created template
        """
        template = PromptTemplate(
            system_prompt=system_prompt,
            user_prompt_template=user_prompt_template,
            version="v1",
            created=datetime.now().isoformat(),
            description=description,
            scaffold_type=scaffold_type,
            git_hash=self._get_git_hash(),
            template_variables=template_variables,
            example_usage=example_usage
        )
        
        # Save template
        self.save_template(template, template_name, auto_version=False)
        
        return template
    
    def list_templates(self) -> Dict[str, List[str]]:
        """
        List all available templates and their versions.
        
        Returns:
            Dictionary mapping template names to list of versions
        """
        templates = {}
        
        for yaml_file in self.templates_dir.glob("*.yaml"):
            name_parts = yaml_file.stem.split('_v')
            if len(name_parts) == 2:
                template_name = name_parts[0]
                version = f"v{name_parts[1]}"
            else:
                template_name = yaml_file.stem
                version = "v1"
            
            if template_name not in templates:
                templates[template_name] = []
            templates[template_name].append(version)
        
        # Sort versions
        for name in templates:
            templates[name].sort(key=lambda x: int(x[1:]))
        
        return templates
    
    def get_template_info(self, template_name: str, version: str = "latest") -> Dict[str, Any]:
        """
        Get information about a template without loading it fully.
        
        Args:
            template_name: Name of the template
            version: Version to get info for
            
        Returns:
            Template metadata and information
        """
        template = self.load_template(template_name, version)
        
        return {
            'name': template_name,
            'version': template.version,
            'description': template.description,
            'scaffold_type': template.scaffold_type,
            'created': template.created,
            'git_hash': template.git_hash,
            'template_variables': template.template_variables,
            'example_usage': template.example_usage,
        }
    
    def render_template(
        self,
        template_name: str,
        version: str = "latest",
        **kwargs
    ) -> Dict[str, str]:
        """
        Render a template with provided variables.
        
        Args:
            template_name: Name of template to render
            version: Version to use
            **kwargs: Template variables
            
        Returns:
            Dictionary with 'system_prompt' and 'user_prompt' keys
        """
        template = self.load_template(template_name, version)
        
        user_prompt = template.render(**kwargs)
        
        return {
            'system_prompt': template.system_prompt,
            'user_prompt': user_prompt,
            'metadata': {
                'template_name': template_name,
                'version': template.version,
                'git_hash': template.git_hash,
                'rendered_at': datetime.now().isoformat(),
                'variables_used': kwargs
            }
        }
    
    def track_template_usage(
        self,
        template_name: str,
        version: str,
        experiment_id: str,
        variables_used: Dict[str, Any],
        result_quality: Optional[float] = None
    ):
        """
        Track template usage for experiment analysis.
        
        Args:
            template_name: Name of template used
            version: Version used
            experiment_id: ID of experiment
            variables_used: Variables used in rendering
            result_quality: Optional quality score for the result
        """
        usage_log = {
            'timestamp': datetime.now().isoformat(),
            'template_name': template_name,
            'version': version,
            'experiment_id': experiment_id,
            'variables_used': variables_used,
            'result_quality': result_quality,
            'git_hash': self._get_git_hash()
        }
        
        # Save to usage log file
        usage_log_file = self.templates_dir / "usage_log.jsonl"
        with open(usage_log_file, 'a') as f:
            f.write(json.dumps(usage_log) + '\n')
    
    def _get_existing_versions(self, template_name: str) -> List[str]:
        """Get list of existing versions for a template."""
        versions = []
        for yaml_file in self.templates_dir.glob(f"{template_name}_v*.yaml"):
            version_str = yaml_file.stem.split('_v')[-1]
            versions.append(f"v{version_str}")
        return versions
    
    def _get_git_hash(self) -> Optional[str]:
        """Get current git commit hash."""
        try:
            result = subprocess.run(
                ['git', 'rev-parse', 'HEAD'],
                capture_output=True,
                text=True,
                cwd=self.templates_dir
            )
            if result.returncode == 0:
                return result.stdout.strip()
        except (subprocess.SubprocessError, FileNotFoundError):
            pass
        return None
    
    def export_templates(self, output_file: str):
        """
        Export all templates to a single file for backup or sharing.
        
        Args:
            output_file: Path to output file
        """
        all_templates = {}
        
        for template_name, versions in self.list_templates().items():
            all_templates[template_name] = {}
            for version in versions:
                template = self.load_template(template_name, version)
                all_templates[template_name][version] = template.to_dict()
        
        with open(output_file, 'w') as f:
            yaml.dump(all_templates, f, default_flow_style=False, indent=2)
    
    def import_templates(self, input_file: str):
        """
        Import templates from a backup file.
        
        Args:
            input_file: Path to input file
        """
        with open(input_file, 'r') as f:
            all_templates = yaml.safe_load(f)
        
        for template_name, versions in all_templates.items():
            for version, template_data in versions.items():
                template = PromptTemplate.from_dict(template_data)
                filename = f"{template_name}_{version}.yaml"
                template_file = self.templates_dir / filename
                
                with open(template_file, 'w') as f:
                    yaml.dump(template_data, f, default_flow_style=False, indent=2) 