"""
Adaptive reasoning scaffold implementation.

This scaffold implements adaptive reasoning with iterative refinement:
1. VLM provides initial description
2. Adaptive reasoner decides if more information is needed
3. If needed, VLM provides focused descriptions
4. Process repeats until confident answer is reached
"""

import time
from pathlib import Path
from typing import Dict, Any, Optional, List, Union

from ..core.base_scaffold import BaseReasoningScaffold, ReasoningResult
from ..core.vlm_interface import VLMInterface
from ..core.reasoner_interface import ReasonerInterface
from ..prompts.manager import PromptManager
from ..utils.logging import get_logger
from ..utils.unified_logger import UnifiedReasoningLogger


class AdaptiveScaffold(BaseReasoningScaffold):
    """Adaptive reasoning scaffold with iterative refinement."""
    
    def __init__(
        self,
        vlm: VLMInterface,
        reasoner: ReasonerInterface,
        max_iterations: int = 7,
        confidence_threshold: float = 0.8,
        logger: Optional[UnifiedReasoningLogger] = None,
        prompt_template_name: str = "adaptive_v1",
        enable_vlm_confidence: bool = False,
        use_confidence_in_reasoner: bool = False,
        **kwargs
    ):
        """
        Initialize adaptive scaffold.
        
        Args:
            vlm: VLM interface for image understanding
            reasoner: Reasoner interface for adaptive reasoning
            max_iterations: Maximum number of iterations
            confidence_threshold: Confidence threshold to stop iterations
            logger: Optional unified logger instance
            prompt_template_name: Name of prompt template to load (default: "adaptive_v1")
            enable_vlm_confidence: Enable VLM confidence estimation (experimental)
            use_confidence_in_reasoner: Use VLM confidence in reasoner prompts (experimental)
            **kwargs: Additional configuration
        """
        super().__init__(vlm, reasoner, "adaptive_scaffold")
        
        self.vlm_interface = vlm
        self.reasoner_interface = reasoner
        self.max_iterations = max_iterations
        self.confidence_threshold = confidence_threshold
        self.prompt_template_name = prompt_template_name
        
        # Confidence estimation experiment flags
        self.enable_vlm_confidence = enable_vlm_confidence
        self.use_confidence_in_reasoner = use_confidence_in_reasoner
        
        # Set up unified logging
        self.unified_logger = logger
        
        # Basic logging
        self.logger = get_logger(self.__class__.__name__)
        
        # Load prompt templates
        self.prompt_manager = PromptManager()
        try:
            self.prompts = self.prompt_manager.load_prompt_collection(self.prompt_template_name)
            self.logger.info(f"Loaded {self.prompt_template_name} prompt collection")
        except Exception as e:
            self.logger.warning(f"Could not load prompt templates: {e}")
            self.prompts = None

    def reason(
        self,
        image: Union[str, Any],
        question: str,
        **kwargs
    ) -> ReasoningResult:
        """
        Perform reasoning on the given image and question (BaseReasoningScaffold interface).
        
        Args:
            image: Input image (path or PIL Image) 
            question: Question to answer about the image
            **kwargs: Additional parameters (dataset, sample_id, etc.)
            
        Returns:
            ReasoningResult containing the complete reasoning trace
        """
        start_time = time.time()
        
        # Extract parameters from kwargs
        dataset = kwargs.get('dataset', 'Unknown')
        sample_id = kwargs.get('sample_id')
        original_index = kwargs.get('original_index')
        ground_truth = kwargs.get('ground_truth')
        generation_kwargs = kwargs.get('generation_kwargs')
        
        # Convert image to path string if needed
        image_path = str(image) if image else ""
        
        try:
            # Call the existing solve method
            solve_result = self.solve(
                image_path=image_path,
                question=question,
                dataset=dataset,
                sample_id=sample_id,
                original_index=original_index,
                ground_truth=ground_truth,
                generation_kwargs=generation_kwargs,
                **kwargs
            )
            
            # Convert solve result to ReasoningResult
            processing_time = time.time() - start_time
            
            reasoning_steps = []
            if 'reasoning' in solve_result:
                reasoning_steps = [solve_result['reasoning']]
            
            return ReasoningResult(
                final_answer=solve_result.get('answer', ''),
                reasoning_steps=reasoning_steps,
                success=solve_result.get('success', False),
                vlm_initial_response=None,  # Could be extracted from solve_result if available
                vlm_intermediate_responses=None,
                reasoner_analysis=solve_result.get('reasoning', ''),
                reasoner_steps=reasoning_steps,
                scaffold_type="adaptive",
                total_iterations=solve_result.get('iterations', 0),
                processing_time=processing_time,
                debug_info={
                    'confidence': solve_result.get('confidence', 0.0),
                    'termination_reason': solve_result.get('termination_reason', 'unknown'),
                    'total_time': solve_result.get('total_time', processing_time)
                },
                error_message=solve_result.get('error') if not solve_result.get('success', False) else None
            )
            
        except Exception as e:
            processing_time = time.time() - start_time
            error_msg = f"Error in adaptive reasoning: {str(e)}"
            self.logger.error(error_msg, exc_info=True)
            
            return ReasoningResult(
                final_answer="",
                reasoning_steps=[],
                success=False,
                scaffold_type="adaptive",
                total_iterations=0,
                processing_time=processing_time,
                error_message=error_msg
            )

    def solve(
        self,
        image_path: str,
        question: str,
        dataset: str = "Unknown",
        sample_id: Optional[str] = None,
        original_index: Optional[str] = None,
        ground_truth: Optional[str] = None,
        generation_kwargs: Optional[Dict[str, Any]] = None,
        vlm_generation_kwargs: Optional[Dict[str, Any]] = None,
        reasoner_generation_kwargs: Optional[Dict[str, Any]] = None,
        **kwargs
    ) -> Dict[str, Any]:
        """Solve a visual reasoning problem using the adaptive scaffold."""
        try:
            start_time = time.time()
            session_id = f"adaptive_{sample_id or 'unknown'}_{int(start_time)}"
            
            # Start logging session if logger is available
            if self.unified_logger:
                session = self.unified_logger.start_session(
                    session_id=session_id,
                    sample_id=sample_id or "unknown",
                    question=question,
                    image_path=image_path,
                    reasoning_approach="adaptive",
                    original_index=original_index,
                    dataset=dataset,
                    ground_truth=ground_truth
                )
            
            # Normalise generation kwargs
            if vlm_generation_kwargs is None and reasoner_generation_kwargs is None:
                vlm_generation_kwargs = generation_kwargs or {}
                reasoner_generation_kwargs = generation_kwargs or {}
            else:
                vlm_generation_kwargs = vlm_generation_kwargs or generation_kwargs or {}
                reasoner_generation_kwargs = reasoner_generation_kwargs or generation_kwargs or {}
            
            # Check if prompts are loaded
            if not self.prompts:
                error_msg = "Prompt templates not loaded - cannot proceed with adaptive reasoning"
                self.logger.error(error_msg)
                if self.unified_logger:
                    self.unified_logger.finish_session(
                        final_answer="Error: Prompt templates not loaded",
                        success=False,
                        termination_reason="prompt_error",
                        error=error_msg,
                        session_id=session_id
                    )
                return {
                    'answer': "Error: Prompt templates not loaded",
                    'success': False,
                    'error': error_msg,
                    'total_time': time.time() - start_time,
                    'iterations': 0
                }
            
            # Initialize adaptive reasoning state  
            cumulative_description = ""
            interaction_history = ""  # Will be populated with prior iterations
            current_iteration = 1
            
            # Initial VLM description
            initial_desc_start = time.time()
            initial_desc_result = self._get_initial_description(
                image_path,
                question,
                vlm_generation_kwargs
            )
            initial_desc_time = time.time() - initial_desc_start
            
            if self.unified_logger:
                self.unified_logger.log_step(
                    step_name="Initial VLM Description",
                    step_type="vlm",
                    input_data={"image_path": image_path, "question": question},
                    output_data={
                        "description": initial_desc_result.get("description", ""),
                        "confidence": initial_desc_result.get("confidence"),
                        "used_confidence_prompt": initial_desc_result.get("used_confidence_prompt", False)
                    },
                    runtime=initial_desc_time,
                    success=initial_desc_result.get("success", False),
                    error=initial_desc_result.get("error"),
                    metadata={
                        "iteration": 0, 
                        "prompt": initial_desc_result.get("prompt", ""),
                        "confidence_experiment": {
                            "enable_vlm_confidence": self.enable_vlm_confidence,
                            "use_confidence_in_reasoner": self.use_confidence_in_reasoner
                        }
                    },
                    session_id=session_id
                )
            
            if not initial_desc_result['success']:
                if self.unified_logger:
                    self.unified_logger.finish_session(
                        final_answer=f"Error in initial VLM description: {initial_desc_result.get('error', 'Unknown error')}",
                        success=False,
                        termination_reason="initial_vlm_failed",
                        error=initial_desc_result.get('error'),
                        session_id=session_id
                    )
                return {
                    'answer': f"Error in initial VLM description: {initial_desc_result.get('error', 'Unknown error')}",
                    'success': False,
                    'error': initial_desc_result.get('error'),
                    'total_time': time.time() - start_time,
                    'iterations': 0
                }
            
            cumulative_description = initial_desc_result['description']
            
            # Stage 1.5: VLM Verification (matching old adaptive system)
            verification_start = time.time()
            verification_result = self._verify_initial_description(
                image_path,
                question,
                cumulative_description,
                vlm_generation_kwargs
            )
            verification_time = time.time() - verification_start
            
            if self.unified_logger:
                self.unified_logger.log_step(
                    step_name="VLM Verification",
                    step_type="vlm",
                    input_data={"image_path": image_path, "question": question, "initial_description": cumulative_description},
                    output_data={"verified_description": verification_result.get("verified_description", "")},
                    runtime=verification_time,
                    success=verification_result.get("success", False),
                    error=verification_result.get("error"),
                    metadata={"iteration": 0, "prompt": verification_result.get("prompt", "")},
                    session_id=session_id
                )
            
            if verification_result['success']:
                # Use verified description with === VERIFIED === prefix (legacy format)
                cumulative_description = f"=== VERIFIED ===\n{verification_result['verified_description']}"
            else:
                # Fallback to original description with warning
                cumulative_description = f"=== VERIFIED (VERIFICATION FAILED) ===\n{cumulative_description}"
            
            self._stage_log("[Stage] Initial description acquired")
            
            # Adaptive reasoning loop
            while current_iteration <= self.max_iterations:
                iteration_start = time.time()
                
                # Adaptive reasoning step
                reasoning_result = self._adaptive_reasoning_step(
                    question,
                    cumulative_description,
                    interaction_history,
                    current_iteration,
                    reasoner_generation_kwargs
                )
                
                iteration_time = time.time() - iteration_start
                
                if self.unified_logger:
                    self.unified_logger.log_step(
                        step_name=f"Adaptive Reasoning (Iteration {current_iteration})",
                        step_type="reasoner",
                        input_data={
                            "question": question,
                            "cumulative_description": cumulative_description,
                            "interaction_history": interaction_history,
                            "iteration": current_iteration
                        },
                        output_data={
                            "reasoning": reasoning_result.get("reasoning", ""),
                            "confidence": reasoning_result.get("confidence", 0),
                            "status": reasoning_result.get("status", ""),
                            "answer": reasoning_result.get("answer", ""),
                            "request": reasoning_result.get("request", "")
                        },
                        runtime=iteration_time,
                        success=reasoning_result.get("success", False),
                        error=reasoning_result.get("error"),
                        metadata={"iteration": current_iteration, "prompt": reasoning_result.get("prompt", "")},
                        session_id=session_id
                    )
                
                # Record this reasoning attempt in interaction history  
                interaction_history += f"\n--- Iteration {current_iteration} ---\n"
                interaction_history += f"Reasoning: {reasoning_result.get('reasoning', '')}\n"
                interaction_history += f"Status: {reasoning_result.get('status', 'UNKNOWN')}\n"
                if reasoning_result.get('answer'):
                    interaction_history += f"Answer: {reasoning_result.get('answer')}\n"
                if reasoning_result.get('request'):
                    interaction_history += f"Request: {reasoning_result.get('request')}\n"
                
                if not reasoning_result['success']:
                    if self.unified_logger:
                        self.unified_logger.finish_session(
                            final_answer=f"Error in adaptive reasoning iteration {current_iteration}: {reasoning_result.get('error', 'Unknown error')}",
                            success=False,
                            termination_reason="adaptive_reasoning_failed",
                            error=reasoning_result.get('error'),
                            session_id=session_id
                        )
                    return {
                        'answer': f"Error in adaptive reasoning: {reasoning_result.get('error', 'Unknown error')}",
                        'success': False,
                        'error': reasoning_result.get('error'),
                        'total_time': time.time() - start_time,
                        'iterations': current_iteration
                    }
                
                # Check if we have a final answer
                if reasoning_result['status'] == 'SOLVED':
                    confidence = reasoning_result.get('confidence', 1.0)
                    final_answer = reasoning_result.get('answer', '')
                    
                    # Add debugging for the reasoning result
                    self.logger.debug("Reasoning result status: SOLVED")
                    self.logger.debug(f"Raw reasoning answer: {repr(final_answer)}")
                    self.logger.debug(f"Reasoning answer type: {type(final_answer)}")
                    
                    # Legacy systems perform a dedicated LLM-based extraction step
                    extracted_answer = self._extract_final_answer(
                        question,
                        cumulative_description,
                        interaction_history,
                        reasoning_result,
                        reasoner_generation_kwargs
                    )
                    
                    self.logger.debug(f"🔍 DEBUG: Extracted answer: {repr(extracted_answer)}")
                    self.logger.debug(f"🔍 DEBUG: Extracted answer type: {type(extracted_answer)}")

                    if self.unified_logger:
                        self.unified_logger.finish_session(
                            final_answer=extracted_answer,
                            success=True,
                            termination_reason="solved_confidently",
                            session_id=session_id
                        )
                    
                    self._stage_log(f"[Stage] Iteration {current_iteration} Reasoner step completed | status={reasoning_result['status']}")
                    
                    return {
                        'answer': extracted_answer,
                        'success': True,
                        'confidence': confidence,
                        'reasoning': reasoning_result.get('reasoning', ''),
                        'total_time': time.time() - start_time,
                        'iterations': current_iteration,
                        'termination_reason': 'solved_confidently'
                    }
                
                # Check if we need more information
                elif reasoning_result['status'] == 'NEED_MORE_INFO':
                    if current_iteration >= self.max_iterations:
                        # Max iterations reached
                        final_answer = reasoning_result.get('answer', 'Unable to solve within maximum iterations')
                        
                        if self.unified_logger:
                            self.unified_logger.finish_session(
                                final_answer=final_answer,
                                success=False,
                                termination_reason="max_iterations_reached",
                                session_id=session_id
                            )
                        
                        return {
                            'answer': final_answer,
                            'success': False,
                            'confidence': reasoning_result.get('confidence', 0),
                            'reasoning': reasoning_result.get('reasoning', ''),
                            'total_time': time.time() - start_time,
                            'iterations': current_iteration,
                            'termination_reason': 'max_iterations_reached'
                        }
                    
                    # Get more specific information from VLM
                    info_request = reasoning_result.get('request', '')
                    
                    # Handle CONTINUE_REASONING case (legacy system logic)
                    if info_request == 'CONTINUE_REASONING':
                        print(f"   Detected abrupt reasoning cutoff - requesting continuation...")
                        
                        # Continue reasoning from where it was cut off
                        continuation_start = time.time()
                        
                        # Create continuation prompt using legacy format
                        continuation_prompt = self._create_continuation_prompt(
                            question=question,
                            cumulative_description=cumulative_description,
                            interaction_history=interaction_history,
                            incomplete_reasoning=reasoning_result.get('reasoning', '')
                        )
                        
                        # Get continuation from reasoner
                        continued_result = self.reasoner_interface.reason(
                            context=continuation_prompt,
                            **(generation_kwargs or {})
                        )
                        
                        # Parse the continuation response
                        continued_parsed = self._parse_generic_reasoning_output(continued_result)
                        
                        continuation_end = time.time()
                        
                        # Add continuation attempt to state
                        if self.unified_logger:
                            self.unified_logger.log_step(
                                step_name=f"Reasoning Continuation (Iteration {current_iteration})",
                                step_type="reasoner",
                                input_data={
                                    "continuation_prompt": continuation_prompt[:500] + "...",
                                    "original_reasoning": reasoning_result.get('reasoning', '')[:200] + "..."
                                },
                                output_data={
                                    "continued_reasoning": continued_parsed.get("reasoning", ""),
                                    "continued_status": continued_parsed.get("status", ""),
                                    "continued_answer": continued_parsed.get("answer", "")
                                },
                                runtime=continuation_end - continuation_start,
                                success=continued_parsed.get("success", True),
                                error=continued_parsed.get("error"),
                                metadata={"iteration": current_iteration, "type": "continuation"},
                                session_id=session_id
                            )
                        
                        print(f"   Continued reasoning: {continued_parsed.get('reasoning', '')[:100]}...")
                        
                        # Replace the original reasoning result with the continued one
                        reasoning_result = continued_parsed
                        
                        # Re-check the status after continuation
                        if reasoning_result['status'] == 'SOLVED':
                            print(f"✅ Solved after continuation!")
                            print(f"   Answer: {reasoning_result['answer']}")
                            
                            # Add debugging for the continuation result
                            self.logger.debug("Reasoning result status: SOLVED")
                            self.logger.debug(f"Continuation answer: {repr(reasoning_result.get('answer'))}")
                            
                            # Extract final answer from reasoning trajectory 
                            print(f"\n[Answer Extraction] Extracting clean final answer from reasoning trajectory...")
                            extracted_answer = self._extract_final_answer(
                                question,
                                cumulative_description,
                                interaction_history,
                                reasoning_result,
                                generation_kwargs
                            )
                            
                            if self.unified_logger:
                                self.unified_logger.finish_session(
                                    final_answer=extracted_answer,
                                    success=True,
                                    termination_reason="solved_after_continuation",
                                    session_id=session_id
                                )
                            
                            return {
                                'answer': extracted_answer,
                                'success': True,
                                'confidence': reasoning_result.get('confidence', 1.0),
                                'reasoning': reasoning_result.get('reasoning', ''),
                                'total_time': time.time() - start_time,
                                'iterations': current_iteration,
                                'termination_reason': 'solved_after_continuation'
                            }
                        # If still NEED_MORE_INFO, fall through to normal handling
                        
                    # Handle normal VLM information request
                    if info_request and info_request != 'CONTINUE_REASONING':
                        # print(f"   Request: {info_request[:100]}...")
                        
                        # Get more specific information from VLM
                        vlm_info_start = time.time()
                        vlm_info_result = self._get_specific_information(
                            image_path, question, info_request, cumulative_description, generation_kwargs
                        )
                        vlm_info_time = time.time() - vlm_info_start
                        
                        if self.unified_logger:
                            self.unified_logger.log_step(
                                step_name=f"VLM Specific Information (Iteration {current_iteration})",
                                step_type="vlm",
                                input_data={
                                    "image_path": image_path,
                                    "question": question,
                                    "info_request": info_request
                                },
                                output_data={"additional_info": vlm_info_result.get("additional_info", "")},
                                runtime=vlm_info_time,
                                success=vlm_info_result.get("success", False),
                                error=vlm_info_result.get("error"),
                                metadata={"iteration": current_iteration, "prompt": vlm_info_result.get("prompt", "")},
                                session_id=session_id
                            )
                        
                        if vlm_info_result['success']:
                            # Update cumulative description
                            additional_info = vlm_info_result['additional_info']
                            cumulative_description += f"\n\nAdditional Information (Iteration {current_iteration}):\n{additional_info}"
                            
                            # Update interaction history (matches legacy format)
                            interaction_history += f"\nIteration {current_iteration}:\n"
                            interaction_history += f"Request: {info_request}\n"
                            interaction_history += f"Response: {additional_info}\n"
                            interaction_history += f"Status: {reasoning_result.get('status', 'NEED_MORE_INFO')}\n"
                            
                            # print(f"   Focused description: {additional_info[:100]}...")
                    else:
                        self.logger.warning(f"Iteration {current_iteration}: Reasoner said NEED_MORE_INFO but provided no valid request!")
                
                current_iteration += 1
            
            # If we get here, we've exhausted iterations without a confident answer
            final_answer = "Unable to solve the problem within the maximum number of iterations. Try to make the best educated guess."
            
            if self.unified_logger:
                self.unified_logger.finish_session(
                    final_answer=final_answer,
                    success=False,
                    termination_reason="max_iterations_reached",
                    session_id=session_id
                )
            
            return {
                'answer': final_answer,
                'success': False,
                'confidence': 0.0,
                'total_time': time.time() - start_time,
                'iterations': self.max_iterations,
                'termination_reason': 'max_iterations_reached'
            }
            
        except Exception as e:
            error_msg = f"Unexpected error in adaptive reasoning: {str(e)}"
            self.logger.error(error_msg, exc_info=True)
            
            if self.unified_logger:
                self.unified_logger.finish_session(
                    final_answer=f"Error: {error_msg}",
                    success=False,
                    termination_reason="unexpected_error",
                    error=error_msg,
                    session_id=session_id
                )
            
            return {
                'answer': f"Error: {error_msg}",
                'success': False,
                'error': error_msg,
                'total_time': time.time() - start_time,
                'iterations': 0
            }

    def _get_initial_description(
        self,
        image_path: str,
        question: str,
        generation_kwargs: Optional[Dict[str, Any]] = None
    ) -> Dict[str, Any]:
        """Get initial description from VLM."""
        try:
            # Choose prompt based on confidence experiment flags
            if self.enable_vlm_confidence:
                # Try to use confidence prompt from two-stage template
                confidence_prompt_template = None
                try:
                    # Load two-stage prompts for confidence prompts
                    two_stage_prompts = self.prompt_manager.load_prompt_collection("two_stage_math_v1")
                    confidence_prompt_template = two_stage_prompts.get('vlm_confidence_prompt')
                except Exception as e:
                    self.logger.warning(f"Could not load two-stage confidence prompts: {e}")
                
                if confidence_prompt_template:
                    prompt = confidence_prompt_template
                    use_confidence_prompt = True
                else:
                    self.logger.warning("VLM confidence prompt not found, falling back to regular prompt")
                    if not self.prompts or 'vlm_initial_description_prompt' not in self.prompts:
                        raise ValueError(
                            "Initial description prompt template not loaded. "
                            "Cannot proceed without proper prompt template. "
                            "Check that 'adaptive_v1.yaml' template file exists and contains 'vlm_initial_description_prompt'."
                        )
                    prompt = self.prompts['vlm_initial_description_prompt']
                    use_confidence_prompt = False
            else:
                if not self.prompts or 'vlm_initial_description_prompt' not in self.prompts:
                    raise ValueError(
                        "Initial description prompt template not loaded. "
                        "Cannot proceed without proper prompt template. "
                        "Check that 'adaptive_v1.yaml' template file exists and contains 'vlm_initial_description_prompt'."
                    )
                prompt = self.prompts['vlm_initial_description_prompt']
                use_confidence_prompt = False
            
            formatted_prompt = prompt.format(question=question)
            
            gen_kwargs = (generation_kwargs or {}).copy()
            
            result = self.vlm_interface.generate(
                image=image_path,
                prompt=formatted_prompt,
                **gen_kwargs
            )
            
            # Parse confidence if using confidence prompt
            confidence_score = None
            description = result
            
            if self.enable_vlm_confidence and use_confidence_prompt:
                try:
                    description, confidence_score = self._parse_vlm_confidence_response(result)
                except Exception as e:
                    self.logger.warning(f"Error parsing confidence from VLM response: {e}")
                    # Safe fallback: use entire response as description
                    description = result
                    confidence_score = None
            
            return {
                'success': True,
                'description': description,
                'confidence': confidence_score,
                'raw_vlm_response': result,
                'prompt': formatted_prompt,
                'used_confidence_prompt': use_confidence_prompt
            }
            
        except Exception as e:
            self.logger.error(f"Error in initial description: {e}", exc_info=True)
            return {
                'success': False,
                'error': str(e),
                'description': '',
                'confidence': None,
                'raw_vlm_response': '',
                'prompt': '',
                'used_confidence_prompt': False
            }
    
    def _adaptive_reasoning_step(
        self, 
        question: str, 
        cumulative_description: str,
        interaction_history: str,
        current_iteration: int,
        generation_kwargs: Optional[Dict[str, Any]] = None
    ) -> Dict[str, Any]:
        """Perform one step of adaptive reasoning."""
        try:
            if not self.prompts or 'reasoner_adaptive_prompt' not in self.prompts:
                raise ValueError(
                    "Adaptive reasoning prompt template not loaded. "
                    "Cannot proceed without proper prompt template. "
                    "Check that 'adaptive_v1.yaml' template file exists and contains 'reasoner_adaptive_prompt'."
                )
            
            prompt = self.prompts['reasoner_adaptive_prompt']

            # Create urgency note based on iteration count (EXACT legacy format)
            if current_iteration >= self.max_iterations - 1:  # Last iteration
                urgency_note = f"""
⚠️  FINAL ITERATION WARNING: This is iteration {current_iteration} of {self.max_iterations}. You MUST provide a final answer now.
Do not request more information - provide your best answer based on available information."""
            elif current_iteration >= self.max_iterations - 2:  # Second to last iteration
                urgency_note = f"""
⚠️  APPROACHING LIMIT: This is iteration {current_iteration} of {self.max_iterations}. Only 1-2 iterations remain.
Be more decisive - only request additional information if absolutely critical."""
            else:
                urgency_note = f"""Current iteration: {current_iteration} of {self.max_iterations} iterations available."""

            # Fill the adaptive prompt
            formatted_prompt = prompt.format(
                question=question,
                cumulative_description=cumulative_description,
                interaction_history=interaction_history,
                urgency_note=urgency_note,
                answer='{answer}'
            )

            # Call the frozen reasoner
            result = self.reasoner_interface.reason(
                context=formatted_prompt,
                **(generation_kwargs or {})
            )

            # Check for incomplete response (missing Status field) and try to continue.
            # This mimics the legacy system's robustness to truncated outputs.
            if "status:" not in result.lower():
                self.logger.warning(
                    "Reasoner response may be incomplete (missing 'Status:' field). "
                    "Attempting to continue."
                )
                continuation_prompt_template = self.prompts.get('reasoner_continuation_prompt')

                if not continuation_prompt_template:
                    self.logger.error("`reasoner_continuation_prompt` not found in templates. Cannot continue.")
                else:
                    continuation_prompt = continuation_prompt_template.format(
                        question=question,
                        cumulative_description=cumulative_description,
                        interaction_history=interaction_history,
                        incomplete_reasoning=result,
                        answer='{answer}'
                    )
                    
                    continued_result = self.reasoner_interface.reason(
                        context=continuation_prompt,
                        **(generation_kwargs or {})
                    )
                    
                    self.logger.info("Received continuation from reasoner.")
                    # Prepend the original incomplete reasoning to the new result.
                    result = result + "\\n\\n--- CONTINUATION ---\\n" + continued_result
            
            # Parse the final result using EXACT legacy parsing logic
            parsed_result = self._parse_generic_reasoning_output(result)
            
            # Return with prompt for debugging
            parsed_result['prompt'] = formatted_prompt
            parsed_result['success'] = True
            
            return parsed_result
            
        except Exception as e:
            self.logger.error(f"Error in adaptive reasoning step: {e}", exc_info=True)
            return {
                'success': False,
                'error': str(e),
                'status': 'ERROR',
                'confidence': 0.0,
                'answer': '',
                'request': '',
                'reasoning': '',
                'prompt': ''
            }

    def _parse_generic_reasoning_output(self, result_or_string: Any) -> Dict[str, Any]:
        """Parse the output using EXACT legacy system logic.
        This method handles both dspy.Prediction objects and raw string outputs.
        """
        import re
        
        raw_text = ""
        if isinstance(result_or_string, str):
            raw_text = result_or_string
        elif hasattr(result_or_string, 'reasoning'): # It's likely a dspy.Prediction
            # If it's a dspy.Prediction, try to get all fields
            # This assumes the signature fields are present on the result object
            return {
                'reasoning': getattr(result_or_string, 'reasoning', ""),
                'status': self._parse_status(getattr(result_or_string, 'status', 'NEED_MORE_INFO')),
                'answer': getattr(result_or_string, 'answer', "N/A") if getattr(result_or_string, 'status', 'NEED_MORE_INFO') == 'SOLVED' else None,
                'request': getattr(result_or_string, 'request', "N/A") if getattr(result_or_string, 'status', 'NEED_MORE_INFO') == 'NEED_MORE_INFO' else None,
            }
        else: # Fallback for unexpected types
            raw_text = str(result_or_string)

        # Common parsing logic for raw text based on the expected output format
        # Handle both plain text and markdown formatting (with **asterisks**)
        reasoning = re.search(r"Reasoning:\s*(.*?)(?=\n(?:\*\*)?Status:|$)", raw_text, re.DOTALL | re.IGNORECASE)
        status_val = re.search(r"(?:\*\*)?Status(?:\*\*)?:\s*(SOLVED|NEED_MORE_INFO)", raw_text, re.IGNORECASE)
        answer_val = re.search(r"(?:\*\*)?Answer(?:\*\*)?:\s*(.*?)(?=\n(?:\*\*)?Request:|$)", raw_text, re.DOTALL | re.IGNORECASE)
        request_val = re.search(r"(?:\*\*)?Request(?:\*\*)?:\s*(.*)", raw_text, re.DOTALL | re.IGNORECASE)

        parsed_reasoning = reasoning.group(1).strip() if reasoning else raw_text # fallback to raw_text
        
        # Determine status – if the model forgot the explicit field but we have a
        # non-empty answer, we treat it as SOLVED to avoid unnecessary NEED_MORE_INFO loops.
        if status_val:
            parsed_status = self._parse_status(status_val.group(1))
        else:
            parsed_status = 'SOLVED' if (answer_val and answer_val.group(1).strip()) else 'NEED_MORE_INFO'
        
        def _clean_field(s: str) -> str:
            return s.strip().strip('*').strip().upper()

        parsed_answer_str = answer_val.group(1).strip() if answer_val else "N/A"
        if _clean_field(parsed_answer_str) == "N/A":
            parsed_answer_str = "N/A"
        parsed_answer = parsed_answer_str if parsed_status == 'SOLVED' and parsed_answer_str != "N/A" else None

        parsed_request_str = request_val.group(1).strip() if request_val else "N/A"
        if _clean_field(parsed_request_str) == "N/A":
            parsed_request_str = "N/A"
        parsed_request = parsed_request_str if parsed_status == 'NEED_MORE_INFO' and parsed_request_str != "N/A" else None

        # If status is SOLVED but answer is still None (e.g. parsed as "N/A"), try to extract from reasoning
        if parsed_status == 'SOLVED' and not parsed_answer:
            extracted_answer = self._extract_answer_from_reasoning(parsed_reasoning)
            if extracted_answer.upper() != "N/A" and extracted_answer != "Unable to determine answer":
                parsed_answer = extracted_answer

        # If status is NEED_MORE_INFO but request is None, check if this is an abrupt cutoff
        if parsed_status == 'NEED_MORE_INFO' and not parsed_request:
            # Check if this appears to be an abrupt cutoff rather than a genuine info request
            if self._is_abrupt_cutoff(raw_text):
                # Mark this as needing continuation rather than more visual info
                parsed_request = "CONTINUE_REASONING"
            else:
                parsed_request = "Please provide more specific details about the visual elements in the image."
            
        return {
            'reasoning': parsed_reasoning,
            'status': parsed_status,
            'answer': parsed_answer,
            'request': parsed_request,
            'confidence': 0.0  # Default confidence
        }
    
    def _is_abrupt_cutoff(self, raw_text: str) -> bool:
        """Detect if the reasoning response was cut off abruptly (e.g., due to token limits).
        
        Simple logic: If there's no clear status, it means abrupt cutoff.
        """
        if not raw_text:
            return False
            
        text = raw_text.strip().lower()
        
        # Check if there's a clear status indicator (handle markdown formatting)
        has_clear_status = (
            'status:' in text or
            'status ' in text or  # Handle cases like "Status SOLVED" without colon
            '**status:**' in text or  # Handle markdown formatting
            ('solved' in text and ('status' in text or text.endswith('solved'))) or
            ('need_more_info' in text and ('status' in text or text.endswith('need_more_info'))) or
            ('need more info' in text and ('status' in text or text.endswith('need more info')))
        )
        
        # If no clear status found, it's likely an abrupt cutoff
        return not has_clear_status
    
    def _parse_status(self, status_str: str) -> str:
        """Parse and validate status string."""
        status = str(status_str).upper().strip()
        if 'SOLVED' in status:
            return 'SOLVED'
        elif 'NEED_MORE_INFO' in status or 'NEED MORE INFO' in status:
            return 'NEED_MORE_INFO'
        else:
            # Default to NEED_MORE_INFO if unclear
            return 'NEED_MORE_INFO'
    
    def _extract_answer_from_reasoning(self, reasoning: str) -> str:
        """Extract answer from reasoning text if not explicitly provided."""
        import re
        
        # Look for common answer patterns
        patterns = [
            r'(?:answer|solution|result)(?:\s*is)?:?\s*(.+?)(?:\n|$)',
            r'(?:therefore|thus|so),?\s*(.+?)(?:\n|$)',
            r'(?:final answer|conclusion):?\s*(.+?)(?:\n|$)'
        ]
        
        for pattern in patterns:
            match = re.search(pattern, reasoning, re.IGNORECASE)
            if match:
                extracted = match.group(1).strip()
                # Never return "unable to determine" - provide a substantive guess
                if "unable to determine" not in extracted.lower() and "cannot determine" not in extracted.lower():
                    return extracted
        
        # If no pattern matches, extract the last meaningful sentence
        sentences = [s.strip() for s in reasoning.split('.') if s.strip()]
        if sentences:
            last_sentence = sentences[-1]
            # Avoid "unable to determine" responses
            if "unable to determine" not in last_sentence.lower() and "cannot determine" not in last_sentence.lower():
                return last_sentence
        
        # Final fallback: encourage providing any guess rather than giving up
        return "Based on the available information, my best assessment is that more specific details are needed to provide a definitive answer"

    def _get_specific_information(
        self, 
        image_path: str,
        question: str, 
        info_request: str,
        cumulative_description: str,
        generation_kwargs: Optional[Dict[str, Any]] = None
    ) -> Dict[str, Any]:
        """Get specific information from VLM based on adaptive reasoning request."""
        try:
            if not self.prompts or 'vlm_focused_description_prompt' not in self.prompts:
                raise ValueError(
                    "Specific information prompt template not loaded. "
                    "Cannot proceed without proper prompt template. "
                    "Check that 'adaptive_v1.yaml' template file exists and contains 'vlm_focused_description_prompt'."
                )
            
            prompt = self.prompts['vlm_focused_description_prompt']
            
            formatted_prompt = prompt.format(
                question=question,
                focus_request=info_request,
                previous_descriptions=cumulative_description
            )
            
            gen_kwargs = (generation_kwargs or {}).copy()
            
            result = self.vlm_interface.generate(
                image=image_path,
                prompt=formatted_prompt,
                **gen_kwargs
            )
        
            return {
                'success': True,
                'additional_info': result,
                'prompt': formatted_prompt
            }
            
        except Exception as e:
            self.logger.error(f"Error getting specific information: {e}", exc_info=True)
            return {
                'success': False,
                'error': str(e),
                'additional_info': '',
                'prompt': ''
            }
    
    def _verify_initial_description(
        self,
        image_path: str,
        question: str,
        initial_description: str,
        generation_kwargs: Optional[Dict[str, Any]] = None
    ) -> Dict[str, Any]:
        """Verify initial description using VLM (matching old adaptive system exactly)."""
        try:
            if not self.prompts or 'vlm_verification_prompt' not in self.prompts:
                raise ValueError(
                    "VLM verification prompt template not loaded. "
                    "Cannot proceed without proper prompt template. "
                    "Check that 'adaptive_v1.yaml' template file exists and contains 'vlm_verification_prompt'."
                )
            
            prompt = self.prompts['vlm_verification_prompt']
            formatted_prompt = prompt.format(question=question, description=initial_description)
            
            gen_kwargs = (generation_kwargs or {}).copy()
            
            verification_response = self.vlm_interface.generate(
                image=image_path,
                prompt=formatted_prompt,
                **gen_kwargs
            )
            
            # Process verification response (same logic as old system)
            if "The description is accurate and complete" not in verification_response:
                if "Improved description:" in verification_response:
                    corrected_parts = verification_response.split("Improved description:", 1)
                    if len(corrected_parts) > 1:
                        verified_description = corrected_parts[1].strip()
                    else:
                        verified_description = verification_response.strip()
                elif "accurate and complete" not in verification_response.lower():
                    # Use the whole response as improved description
                    verified_description = verification_response.strip()
                else:
                    verified_description = initial_description
            else:
                # Description was deemed accurate
                verified_description = initial_description
            
            return {
                'success': True,
                'verified_description': verified_description,
                'prompt': formatted_prompt,
                'verification_response': verification_response
            }
            
        except Exception as e:
            self.logger.error(f"Error in verification step: {e}", exc_info=True)
            return {
                'success': False,
                'error': str(e),
                'verified_description': initial_description,  # Fallback to original
                'prompt': '',
                'verification_response': ''
            }

    # ------------------------------------------------------------------
    # 🧹  Answer extraction helper (LLM-based, mirrors legacy AdaptiveVLMSystem)
    # ------------------------------------------------------------------
    def _extract_final_answer(
        self,
        question: str,
        cumulative_description: str,
        interaction_history: str,
        reasoning_result: dict,
        generation_kwargs: Optional[Dict[str, Any]] = None
    ) -> str:
        """Use the frozen reasoner itself to extract/clean the final answer string.

        This faithfully reproduces the legacy behaviour where a *second* LLM call
        post-processes the answer instead of relying on deterministic regexes.
        """
        try:
            # ------------------------------------------------------------------
            # Load extraction prompt from template (robust & configurable)
            # ------------------------------------------------------------------
            if 'answer_extraction_prompt' not in self.prompts:
                raise ValueError("answer_extraction_prompt missing in prompt template. Ensure adaptive_v1.yaml is up to date and on the prompt path.")

            extraction_prompt_tmpl = self.prompts['answer_extraction_prompt'].replace("boxed{}", "boxed{{}}")
            extraction_prompt = extraction_prompt_tmpl.format(
                question=question,
                cumulative_description=cumulative_description,
                interaction_history=interaction_history,
                final_reasoning=reasoning_result.get('reasoning', ''),
                current_answer=reasoning_result.get('answer', '')
            )

            self.logger.info("[Stage] Running answer extraction step via reasoner…")
            print("[Stage] Answer Extraction – calling reasoner")

            extraction_response = self.reasoner_interface.reason(
                context=extraction_prompt,
                **(generation_kwargs or {})
            )

            # Add debugging information
            self.logger.debug(f"Answer extraction response type: {type(extraction_response)}")
            self.logger.debug(f"Answer extraction response value: {repr(extraction_response)}")
            self.logger.debug(f"Original reasoning result answer: {repr(reasoning_result.get('answer', ''))}")
            
            self.logger.debug(f"Extraction response: {extraction_response[:200]}…")

            # ------------------------------------------------------------------
            # Try several patterns in order of preference
            # ------------------------------------------------------------------
            import re  # Needed for regex extraction below

            final_answer = ""

            patterns = [
                # Pattern 1: Most specific. Handles LaTeX \boxed{} format, which is common.
                # This was the main issue in your examples.
                r"\\{1,4}boxed{([^}]+)}",

                # Pattern 2: Handles "Final Answer: <text>" on a single line, preventing greedy matches.
                # The [^\n]+ captures everything until the end of the line.
                r"\*\*?Final Answer\*\*?[:\s]+([^\n]+)",

                # Pattern 3: Handles conversational phrases like "The final answer is <text>".
                # Captures until a period or a new line.
                r"The final answer is\s+([^.\n]+)",

                # Pattern 4: A more robust version for "Answer: <text>".
                # Uses a word boundary (\b) and is case-insensitive for 'A'.
                r"\b[Aa]nswer\s*:\s*([^\n]+)",
            ]
            for p in patterns:
                m = re.search(p, extraction_response, re.IGNORECASE)
                if m:
                    candidate = m.group(1).strip()
                    if candidate:
                        self.logger.debug(f"Extracted final answer via pattern '{p}': {repr(candidate)}")
                        final_answer = candidate
                        break

            # If regex did not succeed, fall back to last non-empty line
            if not final_answer:
                cleaned = extraction_response.strip().split("\n")[-1].strip()
                final_answer = cleaned
                self.logger.debug(f"Cleaned extraction response fallback: {repr(final_answer)}")

            # ------------------------------------------------------------------
            # Guard: if we ended up with something clearly invalid (e.g. only '**')
            # or not alphanumeric, revert to the original answer that the reasoner
            # had already provided earlier.
            # ------------------------------------------------------------------
            if not any(ch.isalnum() for ch in final_answer):
                original_answer = reasoning_result.get('answer', '').strip()
                if original_answer:
                    self.logger.debug("Extracted answer looked invalid – falling back to original answer")
                    final_answer = original_answer

            return final_answer

        except Exception as e:
            # In case extraction fails, just return original candidate
            self.logger.warning(f"Answer extraction failed: {e}")
            original_answer = reasoning_result.get('answer', '')
            self.logger.debug(f"Using original answer due to extraction failure: {repr(original_answer)}")
            return original_answer

    # ------------------------------------------------------------------
    # Add informative prints / logs at each critical stage
    # ------------------------------------------------------------------
    def _stage_log(self, msg: str):
        """Utility to log and print stage messages uniformly."""
        print(msg)
        self.logger.info(msg)

    def _create_continuation_prompt(self, question: str, cumulative_description: str, 
                                  interaction_history: str, incomplete_reasoning: str) -> str:
        """Create a prompt to continue reasoning from where it was cut off (using prompt manager)."""
        if not self.prompts or 'reasoner_continuation_prompt' not in self.prompts:
            raise ValueError(
                "Continuation prompt template not loaded. "
                "Cannot proceed without proper prompt template. "
                "Check that 'adaptive_v1.yaml' template file exists and contains 'reasoner_continuation_prompt'."
            )
        
        continuation_prompt_template = self.prompts['reasoner_continuation_prompt']
        
        return continuation_prompt_template.format(
            question=question,
            cumulative_description=cumulative_description,
            interaction_history=interaction_history,
            incomplete_reasoning=incomplete_reasoning
        )

    # ------------------------------------------------------------------
    # 🔄 Split scaffold methods for GRPO training
    # ------------------------------------------------------------------
    def get_captioner_description(
        self,
        image_path: str,
        question: str,
        generation_kwargs: Optional[Dict[str, Any]] = None
    ) -> str:
        """
        Get verified image description from captioner (Part 1 for GRPO training).
        
        This method runs only the captioner part of the adaptive scaffold:
        1. Initial VLM description
        2. VLM verification 
        
        Returns the verified description that will be used as input to the reasoning loop.
        
        Args:
            image_path: Path to the image
            question: Question about the image
            generation_kwargs: Generation parameters for VLM
            
        Returns:
            Verified description string
        """
        try:
            # Get initial description
            initial_desc_result = self._get_initial_description(
                image_path, question, generation_kwargs
            )
            
            if not initial_desc_result['success']:
                # Return error description that reasoning loop can handle
                return f"Error in initial description: {initial_desc_result.get('error', 'Unknown error')}"
            
            initial_description = initial_desc_result['description']
            
            # Verify the description
            verification_result = self._verify_initial_description(
                image_path, question, initial_description, generation_kwargs
            )
            
            if verification_result['success']:
                # Use verified description with === VERIFIED === prefix (legacy format)
                verified_description = f"=== VERIFIED ===\n{verification_result['verified_description']}"
            else:
                # Fallback to original description with warning
                verified_description = f"=== VERIFIED (VERIFICATION FAILED) ===\n{initial_description}"
            
            return verified_description
            
        except Exception as e:
            self.logger.error(f"Error in captioner description: {e}", exc_info=True)
            return f"Error: Failed to generate description - {str(e)}"

    def reason_from_description(
        self,
        description: str,
        question: str,
        generation_kwargs: Optional[Dict[str, Any]] = None
    ) -> Dict[str, Any]:
        """
        Run adaptive reasoning from a given description (Part 2 for reward function).
        
        This method runs the reasoning loop part of the adaptive scaffold:
        1. Takes a verified description as input
        2. Runs the adaptive reasoning loop with the reasoner
        3. Returns the final answer
        
        Args:
            description: Verified image description (from get_captioner_description)
            question: Question about the image  
            generation_kwargs: Generation parameters for reasoner
            
        Returns:
            Dict with 'answer', 'success', 'reasoning', etc.
        """
        try:
            start_time = time.time()
            
            # Initialize adaptive reasoning state
            cumulative_description = description
            interaction_history = ""
            current_iteration = 1
            
            # Adaptive reasoning loop (same as original solve method)
            while current_iteration <= self.max_iterations:
                iteration_start = time.time()
                
                # Adaptive reasoning step
                reasoning_result = self._adaptive_reasoning_step(
                    question,
                    cumulative_description,
                    interaction_history,
                    current_iteration,
                    generation_kwargs
                )
                
                # Record this reasoning attempt in interaction history  
                interaction_history += f"\n--- Iteration {current_iteration} ---\n"
                interaction_history += f"Reasoning: {reasoning_result.get('reasoning', '')}\n"
                interaction_history += f"Status: {reasoning_result.get('status', 'UNKNOWN')}\n"
                if reasoning_result.get('answer'):
                    interaction_history += f"Answer: {reasoning_result.get('answer')}\n"
                if reasoning_result.get('request'):
                    interaction_history += f"Request: {reasoning_result.get('request')}\n"
                
                if not reasoning_result['success']:
                    return {
                        'answer': f"Error in adaptive reasoning iteration {current_iteration}: {reasoning_result.get('error', 'Unknown error')}",
                        'success': False,
                        'error': reasoning_result.get('error'),
                        'total_time': time.time() - start_time,
                        'iterations': current_iteration,
                        'reasoning': reasoning_result.get('reasoning', '')
                    }
                
                # Check if we have a final answer
                if reasoning_result['status'] == 'SOLVED':
                    confidence = reasoning_result.get('confidence', 1.0)
                    final_answer = reasoning_result.get('answer', '')
                    
                    # Extract final answer using LLM-based extraction
                    extracted_answer = self._extract_final_answer(
                        question,
                        cumulative_description,
                        interaction_history,
                        reasoning_result,
                        generation_kwargs
                    )
                    
                    return {
                        'answer': extracted_answer,
                        'success': True,
                        'confidence': confidence,
                        'reasoning': reasoning_result.get('reasoning', ''),
                        'total_time': time.time() - start_time,
                        'iterations': current_iteration,
                        'termination_reason': 'solved_confidently'
                    }
                
                # Check if we need more information (but we can't get it from VLM in reward function)
                elif reasoning_result['status'] == 'NEED_MORE_INFO':
                    if current_iteration >= self.max_iterations:
                        # Max iterations reached
                        final_answer = reasoning_result.get('answer', 'Unable to solve within maximum iterations')
                        
                        return {
                            'answer': final_answer,
                            'success': False,
                            'confidence': reasoning_result.get('confidence', 0),
                            'reasoning': reasoning_result.get('reasoning', ''),
                            'total_time': time.time() - start_time,
                            'iterations': current_iteration,
                            'termination_reason': 'max_iterations_reached'
                        }
                    
                    # Handle CONTINUE_REASONING case 
                    info_request = reasoning_result.get('request', '')
                    if info_request == 'CONTINUE_REASONING':
                        # Continue reasoning from where it was cut off
                        continuation_prompt = self._create_continuation_prompt(
                            question=question,
                            cumulative_description=cumulative_description,
                            interaction_history=interaction_history,
                            incomplete_reasoning=reasoning_result.get('reasoning', '')
                        )
                        
                        # Get continuation from reasoner
                        continued_result = self.reasoner_interface.reason(
                            context=continuation_prompt,
                            **(generation_kwargs or {})
                        )
                        
                        # Parse the continuation response
                        continued_parsed = self._parse_generic_reasoning_output(continued_result)
                        
                        # Replace the original reasoning result with the continued one
                        reasoning_result = continued_parsed
                        
                        # Re-check the status after continuation
                        if reasoning_result['status'] == 'SOLVED':
                            # Extract final answer from reasoning trajectory 
                            extracted_answer = self._extract_final_answer(
                                question,
                                cumulative_description,
                                interaction_history,
                                reasoning_result,
                                generation_kwargs
                            )
                            
                            return {
                                'answer': extracted_answer,
                                'success': True,
                                'confidence': reasoning_result.get('confidence', 1.0),
                                'reasoning': reasoning_result.get('reasoning', ''),
                                'total_time': time.time() - start_time,
                                'iterations': current_iteration,
                                'termination_reason': 'solved_after_continuation'
                            }
                    
                    # For other info requests, we can't get more VLM info in reward function
                    # So we tell the reasoner to work with what it has
                    cumulative_description += f"\n\nNote: Cannot get additional visual information. Please work with the provided description."
                
                current_iteration += 1
            
            # If we get here, we've exhausted iterations
            final_answer = "Unable to solve the problem within the maximum number of iterations."
            
            return {
                'answer': final_answer,
                'success': False,
                'confidence': 0.0,
                'total_time': time.time() - start_time,
                'iterations': self.max_iterations,
                'termination_reason': 'max_iterations_reached'
            }
            
        except Exception as e:
            error_msg = f"Unexpected error in reasoning from description: {str(e)}"
            self.logger.error(error_msg, exc_info=True)
            
            return {
                'answer': f"Error: {error_msg}",
                'success': False,
                'error': error_msg,
                'total_time': 0.0,
                'iterations': 0
            }

    def _parse_vlm_confidence_response(self, vlm_response: str) -> tuple[str, Optional[int]]:
        """
        Parse VLM response to extract description and confidence score.
        
        Args:
            vlm_response: Raw VLM response potentially containing confidence
            
        Returns:
            Tuple of (description, confidence_score)
            confidence_score is None if parsing fails or confidence not found
        """
        import re
        
        try:
            # Look for the structured format: DESCRIPTION: ... CONFIDENCE: ...
            desc_match = re.search(r"DESCRIPTION:\s*(.*?)(?=CONFIDENCE:|$)", vlm_response, re.DOTALL | re.IGNORECASE)
            conf_match = re.search(r"CONFIDENCE:\s*(\d+)", vlm_response, re.IGNORECASE)
            
            if desc_match and conf_match:
                description = desc_match.group(1).strip()
                confidence = int(conf_match.group(1))
                
                # Validate confidence range
                if 0 <= confidence <= 100:
                    self.logger.debug(f"Parsed VLM confidence: {confidence}/100")
                    return description, confidence
                else:
                    self.logger.warning(f"Confidence score out of range (0-100): {confidence}")
                    return description, None
            else:
                # Fallback: if structured format not found, use entire response as description
                self.logger.debug("Structured confidence format not found, using entire response as description")
                return vlm_response.strip(), None
                
        except Exception as e:
            self.logger.warning(f"Error parsing VLM confidence response: {e}")
            # Safe fallback: return entire response as description with no confidence
            return vlm_response.strip(), None