"""
Logging utilities for the reasoning frameworks package.

This module provides consistent logging configuration and utilities
for tracking training, evaluation, and debugging information.
"""

import logging
import sys
import os
from datetime import datetime
from pathlib import Path
from typing import Optional, Dict, Any


def setup_logging(
    level: str = "INFO",
    log_file: Optional[str] = None,
    log_dir: Optional[str] = None,
    format_style: str = "detailed"
) -> logging.Logger:
    """
    Set up logging configuration for the framework.
    
    Args:
        level: Logging level (DEBUG, INFO, WARNING, ERROR)
        log_file: Specific log file path
        log_dir: Directory for log files (creates timestamped file)
        format_style: Logging format style ("simple" or "detailed")
        
    Returns:
        Configured logger
    """
    # Convert string level to logging level
    numeric_level = getattr(logging, level.upper(), logging.INFO)
    
    # Define format styles
    if format_style == "simple":
        formatter = logging.Formatter(
            '%(levelname)s - %(message)s'
        )
    else:  # detailed
        formatter = logging.Formatter(
            '%(asctime)s - %(name)s - %(levelname)s - %(filename)s:%(lineno)d - %(message)s'
        )
    
    # Get or create logger
    logger = logging.getLogger("reasoning_frameworks")
    logger.setLevel(numeric_level)
    
    # Clear existing handlers to avoid duplicates
    logger.handlers.clear()
    
    # Console handler
    console_handler = logging.StreamHandler(sys.stdout)
    console_handler.setLevel(numeric_level)
    console_handler.setFormatter(formatter)
    logger.addHandler(console_handler)
    
    # File handler (if requested)
    if log_file or log_dir:
        if log_dir:
            # Create timestamped log file in directory
            os.makedirs(log_dir, exist_ok=True)
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            log_file = os.path.join(log_dir, f"reasoning_frameworks_{timestamp}.log")
        
        file_handler = logging.FileHandler(log_file)
        file_handler.setLevel(numeric_level)
        file_handler.setFormatter(formatter)
        logger.addHandler(file_handler)
        
        logger.info(f"Logging to file: {log_file}")
    
    return logger


def get_logger(name: Optional[str] = None) -> logging.Logger:
    """
    Get a logger instance.
    
    Args:
        name: Logger name (defaults to calling module)
        
    Returns:
        Logger instance
    """
    if name is None:
        # Get the calling module name
        frame = sys._getframe(1)
        name = frame.f_globals.get('__name__', 'reasoning_frameworks')
    
    return logging.getLogger(name)


class ExperimentLogger:
    """
    Logger for experiment tracking and results.
    
    This class provides structured logging for experiments,
    including hyperparameters, metrics, and artifacts.
    """
    
    def __init__(
        self,
        experiment_name: str,
        log_dir: str = "logs",
        enable_wandb: bool = False,
        enable_mlflow: bool = False
    ):
        """
        Initialize experiment logger.
        
        Args:
            experiment_name: Name of the experiment
            log_dir: Directory for log files
            enable_wandb: Enable Weights & Biases logging
            enable_mlflow: Enable MLflow logging
        """
        self.experiment_name = experiment_name
        self.log_dir = Path(log_dir)
        self.log_dir.mkdir(exist_ok=True)
        
        # Set up basic logger
        self.logger = get_logger(f"experiment.{experiment_name}")
        
        # External logging tools
        self.wandb_enabled = enable_wandb
        self.mlflow_enabled = enable_mlflow
        
        if enable_wandb:
            try:
                import wandb
                self.wandb = wandb
                self.logger.info("Weights & Biases logging enabled")
            except ImportError:
                self.logger.warning("wandb not available, disabling W&B logging")
                self.wandb_enabled = False
        
        if enable_mlflow:
            try:
                import mlflow
                self.mlflow = mlflow
                self.logger.info("MLflow logging enabled")
            except ImportError:
                self.logger.warning("mlflow not available, disabling MLflow logging")
                self.mlflow_enabled = False
        
        # Experiment metadata
        self.start_time = datetime.now()
        self.metrics = {}
        self.hyperparameters = {}
        
    def log_hyperparameters(self, params: Dict[str, Any]):
        """
        Log hyperparameters for the experiment.
        
        Args:
            params: Dictionary of hyperparameters
        """
        self.hyperparameters.update(params)
        self.logger.info(f"Hyperparameters: {params}")
        
        if self.wandb_enabled:
            self.wandb.config.update(params)
        
        if self.mlflow_enabled:
            self.mlflow.log_params(params)
    
    def log_metric(self, name: str, value: float, step: Optional[int] = None):
        """
        Log a metric value.
        
        Args:
            name: Metric name
            value: Metric value
            step: Optional step number
        """
        if name not in self.metrics:
            self.metrics[name] = []
        
        metric_entry = {'value': value, 'timestamp': datetime.now()}
        if step is not None:
            metric_entry['step'] = step
        
        self.metrics[name].append(metric_entry)
        
        log_msg = f"Metric {name}: {value}"
        if step is not None:
            log_msg += f" (step {step})"
        self.logger.info(log_msg)
        
        if self.wandb_enabled:
            self.wandb.log({name: value}, step=step)
        
        if self.mlflow_enabled:
            self.mlflow.log_metric(name, value, step=step)
    
    def log_metrics(self, metrics: Dict[str, float], step: Optional[int] = None):
        """
        Log multiple metrics at once.
        
        Args:
            metrics: Dictionary of metric name -> value
            step: Optional step number
        """
        for name, value in metrics.items():
            self.log_metric(name, value, step)
    
    def log_artifact(self, file_path: str, artifact_type: str = "file"):
        """
        Log an artifact (file, model, etc.).
        
        Args:
            file_path: Path to the artifact
            artifact_type: Type of artifact
        """
        self.logger.info(f"Logging artifact: {file_path} (type: {artifact_type})")
        
        if self.wandb_enabled:
            self.wandb.log_artifact(file_path)
        
        if self.mlflow_enabled:
            self.mlflow.log_artifact(file_path)
    
    def log_text(self, text: str, name: str = "output"):
        """
        Log text output.
        
        Args:
            text: Text to log
            name: Name for the text log
        """
        self.logger.info(f"{name}: {text}")
        
        if self.wandb_enabled:
            self.wandb.log({name: wandb.Html(f"<pre>{text}</pre>")})
    
    def finish(self):
        """Finish the experiment and clean up."""
        duration = datetime.now() - self.start_time
        self.logger.info(f"Experiment {self.experiment_name} completed in {duration}")
        
        # Save final experiment summary
        summary = {
            'experiment_name': self.experiment_name,
            'start_time': self.start_time.isoformat(),
            'duration': str(duration),
            'hyperparameters': self.hyperparameters,
            'final_metrics': {
                name: values[-1]['value'] if values else None
                for name, values in self.metrics.items()
            }
        }
        
        summary_file = self.log_dir / f"{self.experiment_name}_summary.json"
        import json
        with open(summary_file, 'w') as f:
            json.dump(summary, f, indent=2)
        
        if self.wandb_enabled:
            self.wandb.finish()
        
        if self.mlflow_enabled:
            self.mlflow.end_run()


def log_function_call(func):
    """
    Decorator to log function calls and execution time.
    
    Args:
        func: Function to decorate
        
    Returns:
        Decorated function
    """
    def wrapper(*args, **kwargs):
        logger = get_logger(func.__module__)
        logger.debug(f"Calling {func.__name__} with args={args}, kwargs={kwargs}")
        
        start_time = datetime.now()
        try:
            result = func(*args, **kwargs)
            duration = datetime.now() - start_time
            logger.debug(f"{func.__name__} completed in {duration}")
            return result
        except Exception as e:
            duration = datetime.now() - start_time
            logger.error(f"{func.__name__} failed after {duration}: {e}")
            raise
    
    return wrapper 