"""
Reproducibility utilities for consistent training and evaluation.

This module provides functions to ensure reproducible results across different
runs and environments, which is critical for scientific research.
"""

import os
import random
import numpy as np
import torch
from typing import Optional


def set_seed(seed: int = 42) -> None:
    """
    Set random seeds for reproducibility.
    
    Args:
        seed: Random seed value
    """
    # Set Python random seed
    random.seed(seed)
    
    # Set NumPy random seed
    np.random.seed(seed)
    
    # Set PyTorch random seeds
    torch.manual_seed(seed)
    torch.cuda.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)  # For multi-GPU
    
    # Set environment variable for additional reproducibility
    os.environ['PYTHONHASHSEED'] = str(seed)


def configure_deterministic() -> None:
    """
    Configure deterministic operations for maximum reproducibility.
    
    This function enables deterministic operations which may be slower
    but ensures reproducible results.
    """
    # Enable deterministic operations
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False
    
    # Set additional environment variables for determinism
    os.environ['CUDA_LAUNCH_BLOCKING'] = '1'
    os.environ['CUBLAS_WORKSPACE_CONFIG'] = ':4096:8'
    
    # Enable deterministic algorithms in PyTorch (if available)
    try:
        torch.use_deterministic_algorithms(True)
    except AttributeError:
        # Older PyTorch versions don't have this function
        pass


def ensure_reproducibility(
    seed: int = 42,
    deterministic: bool = True,
    benchmark: bool = False
) -> None:
    """
    Ensure maximum reproducibility for training and evaluation.
    
    Args:
        seed: Random seed value
        deterministic: Enable deterministic operations (may be slower)
        benchmark: Enable cuDNN benchmark (faster but less deterministic)
    """
    # Set all random seeds
    set_seed(seed)
    
    if deterministic:
        configure_deterministic()
    else:
        # Disable deterministic mode for better performance
        torch.backends.cudnn.deterministic = False
        torch.backends.cudnn.benchmark = True


def get_device(device: Optional[str] = None) -> torch.device:
    """
    Get the appropriate device for computation.
    
    Args:
        device: Specific device string or None for auto-detection
        
    Returns:
        PyTorch device object
    """
    if device is None:
        device = "cuda" if torch.cuda.is_available() else "cpu"
    elif device == "auto":
        device = "cuda" if torch.cuda.is_available() else "cpu"
    
    return torch.device(device)


def print_environment_info() -> None:
    """Print environment information for reproducibility."""
    print("🔧 Environment Information:")
    print(f"   Python: {os.sys.version}")
    print(f"   PyTorch: {torch.__version__}")
    print(f"   CUDA Available: {torch.cuda.is_available()}")
    
    if torch.cuda.is_available():
        print(f"   CUDA Version: {torch.version.cuda}")
        print(f"   GPU Count: {torch.cuda.device_count()}")
        print(f"   GPU Name: {torch.cuda.get_device_name(0)}")
    
    print(f"   Random Seed: {os.environ.get('PYTHONHASHSEED', 'Not set')}")
    print(f"   Deterministic: {torch.backends.cudnn.deterministic}")
    print(f"   Benchmark: {torch.backends.cudnn.benchmark}")


def create_run_config(
    seed: int = 42,
    deterministic: bool = True,
    device: Optional[str] = None
) -> dict:
    """
    Create a run configuration for reproducibility tracking.
    
    Args:
        seed: Random seed
        deterministic: Whether deterministic mode is enabled
        device: Compute device
        
    Returns:
        Dictionary with run configuration
    """
    device_obj = get_device(device)
    
    config = {
        'seed': seed,
        'deterministic': deterministic,
        'device': str(device_obj),
        'python_version': os.sys.version,
        'pytorch_version': torch.__version__,
        'cuda_available': torch.cuda.is_available(),
        'cuda_version': torch.version.cuda if torch.cuda.is_available() else None,
        'gpu_count': torch.cuda.device_count() if torch.cuda.is_available() else 0,
        'gpu_name': torch.cuda.get_device_name(0) if torch.cuda.is_available() else None,
        'environment_variables': {
            'PYTHONHASHSEED': os.environ.get('PYTHONHASHSEED'),
            'CUDA_LAUNCH_BLOCKING': os.environ.get('CUDA_LAUNCH_BLOCKING'),
            'CUBLAS_WORKSPACE_CONFIG': os.environ.get('CUBLAS_WORKSPACE_CONFIG'),
        }
    }
    
    return config 