"""
Unified logging system for reasoning frameworks.

This module provides a comprehensive logging system that supports:
- MLflow experiment tracking
- Weights & Biases integration
- Unified HTML report generation
- Structured logging for both two-stage and adaptive reasoning
"""

import os
import json
import time
import base64
import html
from datetime import datetime
from pathlib import Path
from typing import Dict, Any, List, Optional, Union
from dataclasses import dataclass, asdict

try:
    import mlflow
    MLFLOW_AVAILABLE = True
except ImportError:
    MLFLOW_AVAILABLE = False

try:
    import wandb
    WANDB_AVAILABLE = True
except ImportError:
    WANDB_AVAILABLE = False

from .logging import get_logger


@dataclass
class ReasoningStep:
    """Represents a single step in the reasoning process."""
    step_name: str
    step_type: str  # "vlm", "reasoner", "verification", etc.
    input_data: Dict[str, Any]
    output_data: Dict[str, Any]
    runtime: float
    success: bool
    error: Optional[str] = None
    metadata: Optional[Dict[str, Any]] = None


@dataclass
class ReasoningSession:
    """Represents a complete reasoning session for one sample."""
    # Basic identification
    session_id: str
    sample_id: str
    original_index: Optional[Union[str, int]] = None
    dataset: str = "Unknown"
    
    # Input data
    question: str = ""
    image_path: str = ""
    ground_truth: Optional[str] = None
    
    # Results
    final_answer: str = ""
    predicted_answer: str = ""
    success: bool = False
    
    # Process information
    reasoning_approach: str = "unknown"  # "two_stage", "adaptive"
    steps: List[ReasoningStep] = None
    total_runtime: float = 0.0
    iterations: int = 1
    termination_reason: str = "completed"
    
    # Metadata
    timestamp: str = ""
    error: Optional[str] = None
    debug_info: Optional[Dict[str, Any]] = None
    
    def __post_init__(self):
        if self.steps is None:
            self.steps = []
        if not self.timestamp:
            self.timestamp = datetime.now().isoformat()
    
    def add_step(self, step: ReasoningStep):
        """Add a reasoning step to this session."""
        self.steps.append(step)
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary for serialization."""
        return asdict(self)


class UnifiedReasoningLogger:
    """
    Unified logging system for reasoning frameworks.
    
    Supports MLflow, wandb, HTML reports, and structured logging.
    """
    
    def __init__(
        self,
        experiment_name: str,
        output_dir: str = "./outputs",
        enable_mlflow: bool = False,
        enable_wandb: bool = False,
        enable_html: bool = True,
        mlflow_tracking_uri: Optional[str] = None,
        wandb_project: Optional[str] = None,
        wandb_entity: Optional[str] = None,
        **kwargs
    ):
        """
        Initialize the unified reasoning logger.
        
        Args:
            experiment_name: Name of the experiment
            output_dir: Directory for outputs
            enable_mlflow: Enable MLflow tracking
            enable_wandb: Enable Weights & Biases tracking
            enable_html: Enable HTML report generation
            mlflow_tracking_uri: MLflow tracking URI
            wandb_project: Wandb project name
            wandb_entity: Wandb entity name
            **kwargs: Additional configuration
        """
        self.experiment_name = experiment_name
        self.output_dir = Path(output_dir)
        self.output_dir.mkdir(parents=True, exist_ok=True)
        
        # Basic logging
        self.logger = get_logger(f"reasoning.{experiment_name}")
        
        # Configuration
        self.enable_mlflow = enable_mlflow and MLFLOW_AVAILABLE
        self.enable_wandb = enable_wandb and WANDB_AVAILABLE
        self.enable_html = enable_html
        
        # HTML output directory
        self.html_dir = self.output_dir / "html_reports"
        if self.enable_html:
            self.html_dir.mkdir(parents=True, exist_ok=True)
        
        # ------------------------------------------------------------------
        # 🔹  Per-session raw logging directory (for concurrency-safe debugging)
        # ------------------------------------------------------------------
        #  Each reasoning session will have its own <session_id>.jsonl file that
        #  captures the *raw* input/output payloads for every step, making it
        #  trivial to trace a single example even under high concurrency.
        # ------------------------------------------------------------------
        self.raw_log_dir = self.output_dir / "raw_session_logs"
        self.raw_log_dir.mkdir(parents=True, exist_ok=True)
        # Mapping: session_id -> Path
        self._session_log_files: Dict[str, Path] = {}
        
        # Session tracking - use mapping for parallel process safety
        self.sessions: List[ReasoningSession] = []
        self.active_sessions: Dict[str, ReasoningSession] = {}  # session_id -> session
        
        # MLflow setup
        if self.enable_mlflow:
            try:
                if mlflow_tracking_uri:
                    mlflow.set_tracking_uri(mlflow_tracking_uri)
                mlflow.set_experiment(experiment_name)
                self._mlflow_run = mlflow.start_run(run_name=f"{experiment_name}_{int(time.time())}")
                self.logger.info("MLflow tracking enabled")
            except Exception as e:
                self.logger.warning(f"Failed to initialize MLflow: {e}")
                self.enable_mlflow = False
        
        # Wandb setup
        if self.enable_wandb:
            try:
                wandb.init(
                    project=wandb_project or experiment_name,
                    entity=wandb_entity,
                    name=f"{experiment_name}_{int(time.time())}",
                    config=kwargs
                )
                self.logger.info("Wandb tracking enabled")
            except Exception as e:
                self.logger.warning(f"Failed to initialize wandb: {e}")
                self.enable_wandb = False
    
    def start_session(
        self,
        session_id: str,
        sample_id: str,
        question: str,
        image_path: str,
        reasoning_approach: str,
        original_index: Optional[Union[str, int]] = None,
        dataset: str = "Unknown",
        ground_truth: Optional[str] = None
    ) -> ReasoningSession:
        """
        Start a new reasoning session.
        
        Args:
            session_id: Unique session identifier
            sample_id: Sample identifier
            question: Question being answered
            image_path: Path to the image
            reasoning_approach: Type of reasoning ("two_stage", "adaptive")
            original_index: Original index from dataset
            dataset: Dataset name
            ground_truth: Ground truth answer if available
            
        Returns:
            Created ReasoningSession
        """
        session = ReasoningSession(
            session_id=session_id,
            sample_id=sample_id,
            original_index=original_index,
            dataset=dataset,
            question=question,
            image_path=image_path,
            ground_truth=ground_truth,
            reasoning_approach=reasoning_approach
        )
        
        # Store in both collections for parallel process safety
        self.active_sessions[session_id] = session
        self.sessions.append(session)
        
        self.logger.info(f"Started session {session_id} for sample {sample_id}")
        
        # ------------------------------------------------------------------
        # Register a dedicated raw-log file for this session
        # ------------------------------------------------------------------
        safe_session = "".join(c if c.isalnum() or c in ["-","_", "."] else "_" for c in session_id)
        raw_path = self.raw_log_dir / f"{safe_session}.jsonl"
        self._session_log_files[session_id] = raw_path
        # Write initial metadata line
        self._write_raw(session_id, {
            "event": "start_session",
            "timestamp": time.time(),
            "sample_id": sample_id,
            "dataset": dataset,
            "question": question,
            "image_path": image_path
        })
        
        return session
    
    def log_step(
        self,
        step_name: str,
        step_type: str,
        input_data: Dict[str, Any],
        output_data: Dict[str, Any],
        runtime: float,
        success: bool = True,
        error: Optional[str] = None,
        metadata: Optional[Dict[str, Any]] = None,
        session_id: Optional[str] = None
    ):
        """
        Log a reasoning step.
        
        Args:
            step_name: Name of the step
            step_type: Type of step ("vlm", "reasoner", "verification")
            input_data: Input data for the step
            output_data: Output data from the step
            runtime: Runtime in seconds
            success: Whether the step succeeded
            error: Error message if failed
            metadata: Additional metadata
            session_id: Session ID to log to (if None, tries to find active session)
        """
        # Find the active session for this step
        current_session = None
        if session_id and session_id in self.active_sessions:
            current_session = self.active_sessions[session_id]
        elif len(self.active_sessions) == 1:
            # If only one active session, use it
            current_session = next(iter(self.active_sessions.values()))
        
        if not current_session:
            self.logger.warning("No active session for logging step")
            return
        
        step = ReasoningStep(
            step_name=step_name,
            step_type=step_type,
            input_data=input_data,
            output_data=output_data,
            runtime=runtime,
            success=success,
            error=error,
            metadata=metadata
        )
        
        current_session.add_step(step)
        
        # Log to MLflow if enabled
        if self.enable_mlflow:
            try:
                mlflow.log_metric(f"{step_type}_runtime", runtime)
                mlflow.log_metric(f"{step_type}_success", 1 if success else 0)
            except Exception as e:
                self.logger.debug(f"MLflow step logging failed: {e}")
        
        # Log to wandb if enabled
        if self.enable_wandb:
            try:
                wandb.log({
                    f"{step_type}_runtime": runtime,
                    f"{step_type}_success": success,
                    "step": len(current_session.steps)
                })
            except Exception as e:
                self.logger.debug(f"Wandb step logging failed: {e}")
        
        self.logger.debug(f"Logged step {step_name} (type: {step_type}, runtime: {runtime:.2f}s)")
        
        # ------------------------------------------------------------------
        # EXTRA VERBOSITY FOR DEBUGGING
        # ------------------------------------------------------------------
        #  The project maintainers requested that *every* prompt fed to the
        #  captioner (VLM) and the reasoning LLM, together with their raw
        #  outputs, is emitted to the standard logger so that discrepancies
        #  between the "legacy" and the new implementation can be diagnosed
        #  quickly. We therefore emit an INFO-level log for both the input
        #  and the output payloads whenever the step_type is either
        #  "vlm" (captioner) or "reasoner".
        #
        #  The payloads can occasionally contain non-serialisable objects
        #  (e.g. NumPy arrays, tensors).  We defensively fall back to a
        #  string representation in that case so that logging never throws.
        # ------------------------------------------------------------------
        
        if step_type in {"vlm", "reasoner"}:
            try:
                input_serialised = json.dumps(input_data, ensure_ascii=False, indent=2, default=str)
            except (TypeError, ValueError):
                input_serialised = str(input_data)

            try:
                output_serialised = json.dumps(output_data, ensure_ascii=False, indent=2, default=str)
            except (TypeError, ValueError):
                output_serialised = str(output_data)

            prefix = step_type.upper()
            self.logger.debug(f"[{prefix}] {step_name} INPUT:\n{input_serialised}")
            self.logger.debug(f"[{prefix}] {step_name} OUTPUT:\n{output_serialised}")
        
        # Raw JSONL write
        self._write_raw(current_session.session_id, {
            "event": "step",
            "timestamp": time.time(),
            "step_name": step_name,
            "step_type": step_type,
            "runtime": runtime,
            "success": success,
            "input": input_data,
            "output": output_data,
            "error": error,
            "metadata": metadata
        })
    
    def finish_session(
        self,
        final_answer: str,
        predicted_answer: Optional[str] = None,
        success: bool = True,
        termination_reason: str = "completed",
        error: Optional[str] = None,
        debug_info: Optional[Dict[str, Any]] = None,
        session_id: Optional[str] = None
    ):
        """
        Finish a reasoning session.
        
        Args:
            final_answer: Final answer from reasoning
            predicted_answer: Predicted answer (if different from final_answer)
            success: Whether the session succeeded
            termination_reason: Reason for termination
            error: Error message if failed
            debug_info: Additional debug information
            session_id: Session ID to finish (if None, tries to find active session)
        """
        # Find the session to finish
        current_session = None
        if session_id and session_id in self.active_sessions:
            current_session = self.active_sessions[session_id]
        elif len(self.active_sessions) == 1:
            # If only one active session, use it
            current_session = next(iter(self.active_sessions.values()))
        
        if not current_session:
            self.logger.warning("No active session to finish")
            return
        
        # Calculate total runtime
        total_runtime = sum(step.runtime for step in current_session.steps)
        
        # Update session
        current_session.final_answer = final_answer
        current_session.predicted_answer = predicted_answer or final_answer
        current_session.success = success
        current_session.total_runtime = total_runtime
        current_session.iterations = len([s for s in current_session.steps if s.step_type == "reasoner"])
        current_session.termination_reason = termination_reason
        current_session.error = error
        current_session.debug_info = debug_info
        
        # Log to MLflow
        if self.enable_mlflow:
            try:
                mlflow.log_metric("total_runtime", total_runtime)
                mlflow.log_metric("success", 1 if success else 0)
                mlflow.log_metric("iterations", current_session.iterations)
                mlflow.log_param("termination_reason", termination_reason)
                mlflow.log_param("reasoning_approach", current_session.reasoning_approach)
            except Exception as e:
                self.logger.debug(f"MLflow session logging failed: {e}")
        
        # Log to wandb
        if self.enable_wandb:
            try:
                wandb.log({
                    "total_runtime": total_runtime,
                    "success": success,
                    "iterations": current_session.iterations,
                    "session_finished": 1
                })
            except Exception as e:
                self.logger.debug(f"Wandb session logging failed: {e}")
        
        # Generate HTML report
        if self.enable_html:
            try:
                self._generate_html_report(current_session)
            except Exception as e:
                self.logger.warning(f"HTML report generation failed: {e}")
        
        # Log session completion
        session_id_str = current_session.session_id
        self.logger.info(f"Finished session {session_id_str} "
                        f"(success: {success}, runtime: {total_runtime:.2f}s)")
        
        # Raw JSONL write – session end summary
        self._write_raw(current_session.session_id, {
            "event": "finish_session",
            "timestamp": time.time(),
            "success": success,
            "iterations": current_session.iterations,
            "total_runtime": total_runtime,
            "termination_reason": termination_reason,
            "final_answer": final_answer,
            "error": error
        })
        
        # Remove from active sessions (but keep in sessions list)
        if session_id:
            self.active_sessions.pop(session_id, None)
        else:
            # Remove by session_id if we found it
            self.active_sessions.pop(current_session.session_id, None)
    
    def log_experiment_params(self, params: Dict[str, Any]):
        """Log experiment parameters."""
        if self.enable_mlflow:
            try:
                mlflow.log_params(params)
            except Exception as e:
                self.logger.debug(f"MLflow param logging failed: {e}")
        
        if self.enable_wandb:
            try:
                wandb.config.update(params)
            except Exception as e:
                self.logger.debug(f"Wandb config update failed: {e}")
    
    def log_metrics(self, metrics: Dict[str, float]):
        """Log experiment metrics."""
        if self.enable_mlflow:
            try:
                mlflow.log_metrics(metrics)
            except Exception as e:
                self.logger.debug(f"MLflow metrics logging failed: {e}")
        
        if self.enable_wandb:
            try:
                wandb.log(metrics)
            except Exception as e:
                self.logger.debug(f"Wandb metrics logging failed: {e}")
    
    def _generate_html_report(self, session: ReasoningSession):
        """Generate HTML report for a session."""
        # Read image for base64 encoding
        base64_image = None
        if session.image_path and os.path.exists(session.image_path):
            try:
                with open(session.image_path, 'rb') as f:
                    image_data = base64.b64encode(f.read()).decode('utf-8')
                base64_image = image_data
            except Exception as e:
                self.logger.debug(f"Failed to encode image: {e}")
        
        # Generate HTML content
        html_content = self._create_unified_html(session, base64_image)
        
        # Save HTML file
        filename = f"{session.reasoning_approach}_{session.dataset}_sample_{session.original_index or session.sample_id}.html"
        safe_filename = "".join(c if c.isalnum() or c in ['-', '_', '.'] else "_" for c in filename)
        
        html_path = self.html_dir / safe_filename
        with open(html_path, 'w', encoding='utf-8') as f:
            f.write(html_content)
        
        self.logger.debug(f"Generated HTML report: {html_path}")
    
    def _create_unified_html(self, session: ReasoningSession, base64_image: Optional[str]) -> str:
        """Create unified HTML content that adapts to different reasoning approaches."""
        
        def escape(text):
            return html.escape(str(text)) if text is not None else "N/A"
        
        # Basic information
        title = f"{session.reasoning_approach.title()} Reasoning Report"
        
        # Image section
        img_tag = "[Image Not Available]"
        if base64_image:
            img_tag = f'<img src="data:image/jpeg;base64,{base64_image}" alt="Sample Image" style="max-width: 500px; height: auto; border: 1px solid #ccc; margin-bottom: 15px;">'
        
        # Steps section - adapt based on reasoning approach
        steps_html = self._create_steps_html(session)
        
        # Performance metrics
        metrics_html = self._create_metrics_html(session)
        
        html_content = f"""
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{escape(title)} - Sample {escape(session.sample_id)}</title>
    <style>
        body {{ font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6; padding: 20px; max-width: 1200px; margin: auto; background-color: #f5f5f5; }}
        .container {{ background-color: white; padding: 30px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }}
        h1 {{ color: #2c3e50; border-bottom: 3px solid #3498db; padding-bottom: 10px; }}
        h2 {{ color: #34495e; border-bottom: 2px solid #ecf0f1; padding-bottom: 8px; margin-top: 30px; }}
        h3 {{ color: #7f8c8d; margin-top: 20px; }}
        .section {{ background-color: #fafafa; border: 1px solid #e1e8ed; padding: 20px; margin-bottom: 20px; border-radius: 8px; }}
        .metadata {{ display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 15px; }}
        .metadata-item {{ background: white; padding: 15px; border-radius: 5px; border-left: 4px solid #3498db; }}
        .label {{ font-weight: bold; color: #2c3e50; display: inline-block; min-width: 150px; }}
        .value {{ color: #34495e; }}
        .success {{ color: #27ae60; font-weight: bold; }}
        .error {{ color: #e74c3c; font-weight: bold; }}
        .step {{ background: white; border: 1px solid #ddd; margin: 10px 0; border-radius: 5px; overflow: hidden; }}
        .step-header {{ background: #ecf0f1; padding: 15px; cursor: pointer; font-weight: bold; }}
        .step-content {{ padding: 15px; display: none; }}
        .step-content.active {{ display: block; }}
        .step.success .step-header {{ background: #d5f4e6; }}
        .step.error .step-header {{ background: #fadbd8; }}
        pre {{ background-color: #f8f9fa; padding: 15px; border-radius: 4px; overflow-x: auto; border: 1px solid #e9ecef; }}
        .runtime {{ font-size: 0.9em; color: #7f8c8d; float: right; }}
        .metrics {{ display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; }}
        .metric {{ background: white; padding: 15px; border-radius: 5px; text-align: center; border: 1px solid #ddd; }}
        .metric-value {{ font-size: 1.5em; font-weight: bold; color: #3498db; }}
        .metric-label {{ color: #7f8c8d; font-size: 0.9em; }}
        .reasoning-approach {{ display: inline-block; background: #3498db; color: white; padding: 5px 15px; border-radius: 20px; font-size: 0.9em; }}
    </style>
    <script>
        function toggleStep(stepId) {{
            const content = document.getElementById(stepId);
            content.classList.toggle('active');
        }}
    </script>
</head>
<body>
    <div class="container">
        <h1>{escape(title)}</h1>
        <span class="reasoning-approach">{escape(session.reasoning_approach.replace('_', ' ').title())}</span>
        
        <div class="section">
            <h2>Metadata</h2>
            <div class="metadata">
                <div class="metadata-item">
                    <div><span class="label">Dataset:</span> <span class="value">{escape(session.dataset)}</span></div>
                </div>
                <div class="metadata-item">
                    <div><span class="label">Sample ID:</span> <span class="value">{escape(session.sample_id)}</span></div>
                </div>
                <div class="metadata-item">
                    <div><span class="label">Original Index:</span> <span class="value">{escape(session.original_index)}</span></div>
                </div>
                <div class="metadata-item">
                    <div><span class="label">Success:</span> <span class="{'success' if session.success else 'error'}">{escape('Yes' if session.success else 'No')}</span></div>
                </div>
            </div>
        </div>

        <div class="section">
            <h2>Input Data</h2>
            <h3>Image</h3>
            {img_tag}
            <h3>Question</h3>
            <p>{escape(session.question)}</p>
            {f'<h3>Ground Truth</h3><pre>{escape(session.ground_truth)}</pre>' if session.ground_truth else ''}
        </div>

        <div class="section">
            <h2>Results</h2>
            <h3>Final Answer</h3>
            <pre>{escape(session.final_answer)}</pre>
            {f'<h3>Error</h3><pre class="error">{escape(session.error)}</pre>' if session.error else ''}
        </div>

        {metrics_html}
        {steps_html}
    </div>
</body>
</html>
        """
        
        return html_content
    
    def _create_steps_html(self, session: ReasoningSession) -> str:
        """Create HTML for reasoning steps."""
        if not session.steps:
            return ""
        
        steps_html = '<div class="section"><h2>Reasoning Steps</h2>'
        
        for i, step in enumerate(session.steps):
            step_id = f"step_{i}"
            status_class = "success" if step.success else "error"
            
            steps_html += f'''
            <div class="step {status_class}">
                <div class="step-header" onclick="toggleStep('{step_id}')">
                    {html.escape(step.step_name)} ({html.escape(step.step_type)})
                    <span class="runtime">{step.runtime:.2f}s</span>
                </div>
                <div class="step-content" id="{step_id}">
                    <h4>Input Data</h4>
                    <pre>{html.escape(json.dumps(step.input_data, indent=2))}</pre>
                    <h4>Output Data</h4>
                    <pre>{html.escape(json.dumps(step.output_data, indent=2))}</pre>
                    {f'<h4>Error</h4><pre class="error">{html.escape(step.error)}</pre>' if step.error else ''}
                    {f'<h4>Metadata</h4><pre>{html.escape(json.dumps(step.metadata, indent=2))}</pre>' if step.metadata else ''}
                </div>
            </div>
            '''
        
        steps_html += '</div>'
        return steps_html
    
    def _create_metrics_html(self, session: ReasoningSession) -> str:
        """Create HTML for performance metrics."""
        vlm_steps = [s for s in session.steps if s.step_type == "vlm"]
        reasoner_steps = [s for s in session.steps if s.step_type == "reasoner"]
        
        vlm_runtime = sum(s.runtime for s in vlm_steps)
        reasoner_runtime = sum(s.runtime for s in reasoner_steps)
        
        metrics_html = f'''
        <div class="section">
            <h2>Performance Metrics</h2>
            <div class="metrics">
                <div class="metric">
                    <div class="metric-value">{session.total_runtime:.2f}s</div>
                    <div class="metric-label">Total Runtime</div>
                </div>
                <div class="metric">
                    <div class="metric-value">{len(session.steps)}</div>
                    <div class="metric-label">Total Steps</div>
                </div>
                <div class="metric">
                    <div class="metric-value">{session.iterations}</div>
                    <div class="metric-label">Iterations</div>
                </div>
                <div class="metric">
                    <div class="metric-value">{vlm_runtime:.2f}s</div>
                    <div class="metric-label">VLM Runtime</div>
                </div>
                <div class="metric">
                    <div class="metric-value">{reasoner_runtime:.2f}s</div>
                    <div class="metric-label">Reasoner Runtime</div>
                </div>
                <div class="metric">
                    <div class="metric-value">{html.escape(session.termination_reason)}</div>
                    <div class="metric-label">Termination Reason</div>
                </div>
            </div>
        </div>
        '''
        return metrics_html
    
    def export_sessions(self, output_path: Optional[str] = None) -> str:
        """Export all sessions to JSON file."""
        if not output_path:
            output_path = self.output_dir / f"sessions_{int(time.time())}.json"
        
        sessions_data = [session.to_dict() for session in self.sessions]
        
        with open(output_path, 'w') as f:
            json.dump(sessions_data, f, indent=2)
        
        self.logger.info(f"Exported {len(self.sessions)} sessions to {output_path}")
        return str(output_path)
    
    def get_summary_metrics(self) -> Dict[str, Any]:
        """Get summary metrics across all sessions."""
        if not self.sessions:
            return {}
        
        total_sessions = len(self.sessions)
        successful_sessions = sum(1 for s in self.sessions if s.success)
        
        avg_runtime = sum(s.total_runtime for s in self.sessions) / total_sessions
        avg_iterations = sum(s.iterations for s in self.sessions) / total_sessions
        
        approach_counts = {}
        for session in self.sessions:
            approach = session.reasoning_approach
            approach_counts[approach] = approach_counts.get(approach, 0) + 1
        
        return {
            'total_sessions': total_sessions,
            'success_rate': successful_sessions / total_sessions,
            'avg_runtime': avg_runtime,
            'avg_iterations': avg_iterations,
            'approach_distribution': approach_counts,
            'total_runtime': sum(s.total_runtime for s in self.sessions)
        }
    
    def finish(self):
        """Finish the logging session and clean up."""
        # Log summary metrics
        summary = self.get_summary_metrics()
        if summary:
            self.log_metrics(summary)
            self.logger.info(f"Session summary: {summary}")
        
        # Export sessions
        self.export_sessions()
        
        # Clean up external trackers
        if self.enable_mlflow:
            try:
                mlflow.end_run()
            except Exception as e:
                self.logger.debug(f"MLflow cleanup failed: {e}")
        
        if self.enable_wandb:
            try:
                wandb.finish()
            except Exception as e:
                self.logger.debug(f"Wandb cleanup failed: {e}")
        
        self.logger.info(f"Finished logging session with {len(self.sessions)} total sessions")
    
    # ------------------------------------------------------------------
    # Internal helper: append a JSONL record for the given session
    # ------------------------------------------------------------------
    def _write_raw(self, session_id: str, record: Dict[str, Any]):
        """Append a JSON-serialisable record to this session's raw log file."""
        path = self._session_log_files.get(session_id)
        if not path:
            return  # Should not happen but guard anyway
        try:
            with open(path, "a", encoding="utf-8") as fh:
                json.dump(record, fh, ensure_ascii=False)
                fh.write("\n")
        except Exception as e:
            # Do not crash if file write fails; just log debug
            self.logger.debug(f"Failed to write raw log for session {session_id}: {e}") 