
from git import Repo
import os

class InstanceRepo: 

    def __init__(self, repo_folder, repo_name):
        self.repo = Repo(repo_folder+"/"+repo_name)

    def checkout(self, commit):
        """ Discards uncommitted changes and checks out the commit """
        self.repo.git.restore('.')
        self.repo.git.clean('-f','-d')
        self.repo.git.checkout(commit, '-f')  # careful! ignores untracked files (including temp.patch)

    def apply_patchstr(self, patch_str):
        """ Applies a patch string by writing to a temporary patch file
            Returns a list of files modified by the patch """
        with open(self.repo.working_dir+"/temp.patch", "w", encoding='utf-8') as f_patch:
            f_patch.write(patch_str)
        self.repo.git.apply("temp.patch")
        modified_files = [self.repo.working_dir+"/"+f.a_path for f in self.repo.index.diff(None)]
        return modified_files  # full path of each modified file
    
    def get_modified_lines(self, source_file):
        """ Lists all uncommitted, modified line numbers for a given source file """
        modified_lines = []
        for line_info in self.repo.git.blame(source_file,"--porcelain").splitlines():
            if line_info[:40] == "0000000000000000000000000000000000000000":  # indicates uncommitted change
                modified_lines.append(int(line_info.split()[2]))  # line number (following modification)
        return modified_lines

def clone_from_github(repo_folder, repo_name):
    """ Clone a specified repository from github to local """

    gitpath_remote = "https://github.com/"+repo_name+".git"
    gitpath_local = repo_folder+"/"+repo_name

    if os.path.isdir(gitpath_local):
        print(gitpath_local + " already exists")
    else:
        print("cloning " + gitpath_remote + " to " + gitpath_local + " ...")
        Repo.clone_from(gitpath_remote, gitpath_local)
