
import ast
import builtins

def overlaps(node, line_numbers):
    """
    Returns True if the start line of the AST node is contained within the specified line numbers.
    If the node has no line number information, it is assumed to be contained.
    Input:
    - node: An AST node.
    - line_numbers: A list or set of line numbers to check against.
    Output:
    - True if the node overlaps with any of the line numbers, False otherwise.
    """
    if node.lineno is None:
      return False
    else:
      # could consider or node.end_lineno in line_numbers
      return node.lineno in line_numbers or len(line_numbers) == 0

def declared_identifiers(source_file, modified_lines):
    """
    Returns all identifiers (names) declared in a given Python file on the specified lines.
    This function uses the Abstract Syntax Tree (AST) module to parse the program
    and identify declared identifiers. Declared identifiers are considered to be the names introduced by
    function definitions, function parameters, class definitions, and variable assignments.

    Only nodes reflecting tokens that start or end on the specified source file
    line numbers will be included (unless line_numbers is empty).
    Input:
    - source_file: Name of Python file to analyze.
    - line_number: Source file line numbers to analyze (or [] for all lines)
    Output:
    - A set of declared variable names (strings) found in the program.
    """
    def internal(ast_node, identifiers=[], in_class=False, in_function=False):

        # handle a non-nested function or class definition
        if isinstance(ast_node, (ast.FunctionDef, ast.ClassDef)) and not in_function:
            # record whether this is a function or class, and get the name
            is_func = isinstance(ast_node, ast.FunctionDef)
            is_class = isinstance(ast_node, ast.ClassDef)
            name = ast_node.name
            # check if the start line of the node is in the modified lines
            # if so, add the function/class name to the set of identifiers
            if overlaps(ast_node, modified_lines):
                identifiers.append(name)
            if is_func:  # if this is a function def, add all keyword argument names as well
                for arg in (ast_node.args.args
                              + ast_node.args.kwonlyargs
                              + [ast_node.args.kwarg]):
                    if arg is not None and overlaps(arg, modified_lines):
                        identifiers.append(arg.arg)
            # recursively process all child nodes
            for child in ast.iter_child_nodes(ast_node):
                internal(child, identifiers, in_class=is_class or in_class, in_function=is_func or in_function)

        # handle an assignment
        elif isinstance(ast_node, ast.Assign):
            # check if the assignment is on a modified line
            if overlaps(ast_node, modified_lines):
                for target in ast_node.targets:
                    # add assignments to variables in classes of the form `self.name = `
                    if in_class and isinstance(target, ast.Attribute) and target.value.id == "self":
                        identifiers.append(target.attr)
                    # check for top-level names (not in a class or function)
                    elif (not in_class and not in_function):
                        # usual bindings
                        if isinstance(target, ast.Name):
                            identifiers.append(target.id)
                        # tuple unpacking
                        if isinstance(target, ast.Tuple):
                            for elt in target.elts:
                                if isinstance(elt, ast.Name):
                                    identifiers.append(elt.id)

        # not a function, class or assignment
        else:
            # process children
            for child in ast.iter_child_nodes(ast_node):
                internal(child, identifiers, in_class=in_class, in_function=in_function)

        return identifiers

    # convert source file to string
    with open(source_file, "r", encoding='utf-8') as f_prog:
        prog = f_prog.read()

    # process ast recursively
    identifiers = internal(ast.parse(prog), identifiers=[])
    return set(identifiers)


def used_identifiers(source_file, modified_lines):
    """
    Returns all identifiers (names) used in a given Python file on the specified lines.
    This function uses the Abstract Syntax Tree (AST) module to parse the program
    and identify used identifiers. Used identifiers include function names,
    class names, variable names, and attribute names.

    Only nodes reflecting tokens that start or end on the specified source file
    line numbers will be included (unless line_numbers is empty).
    Input:
    - source_file: Name of Python file to analyze.
    - line_number: Source file line numbers to analyze (or [] for all lines)
    Output:
    - A set of used variable names (strings) found in the program.
    """
    def internal(node, identifiers=[], binder_stack=[]):

        # check if the node is a variable and bound in the current context
        # if it is a bound variable, add it to the set
        if (isinstance(node, ast.Name)
            and (not (any(map(lambda binders: node.id in binders, binder_stack))))
            and overlaps(node, modified_lines)):
            # check that the name is not something in the standard library
            # (this is a simple heuristic and may not cover all cases)
            if node.id not in dir(builtins):
                identifiers.add(node.id)

        # handle an attribute
        elif (isinstance(node, ast.Attribute)
            and overlaps(node, modified_lines)):
            identifiers.add(node.attr)  # attribute name

        # handle an assignment
        elif isinstance(node, ast.Assign):
            # add targets to the binding stack as they count as binders
            for target in node.targets:
                if isinstance(target, ast.Name):
                    binder_stack[-1].add(target.id)
                else:
                    # handle tuple unpacking
                    if isinstance(target, ast.Tuple):
                        for elt in target.elts:
                            if isinstance(elt, ast.Name):
                                binder_stack[-1].add(elt.id)
            # process the value (right-hand side) of the assignment
            if overlaps(node, modified_lines):
                internal(node.value, identifiers, binder_stack)

        # handle a loop
        elif isinstance(node, ast.For):
            # add the loop variable(s) to the binding stack
            if isinstance(node.target, ast.Name):
                binder_stack[-1].add(node.target.id)
            elif isinstance(node.target, ast.Tuple):
                for elt in node.target.elts:
                    if isinstance(elt, ast.Name):
                        binder_stack[-1].add(elt.id)
            # process the iterable and body of the loop
            if overlaps(node, modified_lines):
                internal(node.iter, identifiers, binder_stack)
                internal(node.body, identifiers, binder_stack)
                internal(node.orelse, identifiers, binder_stack)

        # handle a function definition
        elif isinstance(node, ast.FunctionDef):
            binder_stack.append(set())                      # start a new binder scope for the function
            for arg in node.args.args:                      # add the function's arguments to the current binder scope
                binder_stack[-1].add(arg.arg)
            internal(node.body, identifiers, binder_stack)  # recursively process the function body
            binder_stack.pop()                              # end the current binder scope

        # handle a list of nodes
        elif isinstance(node, list):
            for item in node:
                internal(item, identifiers, binder_stack)

        # handle other ast nodes
        else:
            for child in ast.iter_child_nodes(node):
                internal(child, identifiers, binder_stack)
        return identifiers

    # convert source file to string
    with open(source_file, "r", encoding='utf-8') as f_prog:
        prog = f_prog.read()

    # process ast recursively
    identifiers = internal(ast.parse(prog), identifiers=set(), binder_stack=[set()])
    return set(identifiers)
