from enum import Enum
from typing import Optional
from typing import List
from metagpt.schema import Message
from metagpt.actions import Action

from di_project.actions.code_reflection import CodeReflection
from di_project.actions.code_self_verification import CodeSelfVerification
from di_project.actions.execute_nb_code import ExecuteNbCode

OUTPUT_ANSWER_CONTEXT = """
## Context
{content}
-----
"""


class EvaluationResult(Enum):
    """Evaluation result states"""
    TRUE = 'true'
    FALSE = 'false'
    LIKELY = 'likely'


class ResultFormatter:
    def format_evaluation_result(self, csv_code_result: Optional[str]) -> str:
        """
        Format evaluation result string to standardized output.

        Args:
            csv_code_result: Raw evaluation result string

        Returns:
            Standardized result string: 'true', 'false', or 'likely'

        Examples:
            >>> formatter = ResultFormatter()
            >>> formatter.format_evaluation_result("The result is TRUE")
            'true'
            >>> formatter.format_evaluation_result("The answer is False")
            'false'
            >>> formatter.format_evaluation_result("Maybe correct")
            'likely'
        """
        if not csv_code_result:
            return EvaluationResult.LIKELY.value
        
        result_lower = csv_code_result.lower()
        
        # Use dictionary for mapping results
        result_mapping = {
            'true': EvaluationResult.TRUE.value,
            'false': EvaluationResult.FALSE.value
        }
        
        # Get result based on substring presence
        for key, value in result_mapping.items():
            if key in result_lower:
                return value
        
        return EvaluationResult.LIKELY.value

class MathOutputAnswer(Action):
    async def run(self, context: List[Message], ):
        content = context[0].content
        prompt = OUTPUT_ANSWER_CONTEXT.format(content=content, )
        prompt += "\nAccording to the above information and runtime answer, put the answer in \\boxed{}, in LaTex format."
        rsp = await self._aask(prompt)
        return rsp
    
    
    async def post_process(self, plan, answer, code, result, execute_nb_code: ExecuteNbCode):
        csv_code, csv_code_result, csv_success = await CodeSelfVerification().run(
            plan=plan, answer=answer, execute_code=execute_nb_code
        )
        formatter = ResultFormatter()
        csv_result_is_true = formatter.format_evaluation_result(csv_code_result)
       
        if not csv_success or not csv_result_is_true:
            code_summary, code_reflection_confirmed, suggestion = \
                await CodeReflection().run(
                    plan=plan,
                    code=code,
                    code_result=result,
                )
            
            # self.planner.working_memory.add(
            #     Message(
            #         content=result + '\n' + '--------' + '\n' + 'suggestion : ' + suggestion,
            #         role="user",
            #         cause_by=ExecuteNbCode
            #     )
            # )
            
            success = False
        else:
            suggestion = ""
            success = True
        
        return success, answer, csv_result_is_true, suggestion
