import torch
import torch.nn as nn
import torch.utils.checkpoint as checkpoint
from torch.nn.functional import silu
from timm.models.layers import DropPath, to_2tuple, trunc_normal_
from utils.graph_utils import mask_nodes, mask_adjs


class Mlp(nn.Module):
    def __init__(self, in_features, hidden_features=None, out_features=None, act_layer=nn.GELU, drop=0.):
        super().__init__()
        out_features = out_features or in_features
        hidden_features = hidden_features or in_features
        self.fc1 = nn.Linear(in_features, hidden_features)
        self.act = act_layer()
        self.fc2 = nn.Linear(hidden_features, out_features)
        self.drop = nn.Dropout(drop)

    def forward(self, x):
        x = self.fc1(x)
        x = self.act(x)
        x = self.drop(x)
        x = self.fc2(x)
        x = self.drop(x)
        return x


def window_partition(x, window_size):
    """
    Args:
        x: (B, H, W, C)
        window_size (int): window size

    Returns:
        windows: (num_windows*B, window_size, window_size, C)
    """
    B, H, W, C = x.shape
    x = x.view(B, H // window_size, window_size, W // window_size, window_size, C)
    windows = x.permute(0, 1, 3, 2, 4, 5).contiguous().view(-1, window_size, window_size, C)
    return windows


def window_reverse(windows, window_size, H, W):
    """
    Args:
        windows: (num_windows*B, window_size, window_size, C)
        window_size (int): Window size
        H (int): Height of image
        W (int): Width of image

    Returns:
        x: (B, H, W, C)
    """
    B = int(windows.shape[0] / (H * W / window_size / window_size))
    x = windows.view(B, H // window_size, W // window_size, window_size, window_size, -1)
    x = x.permute(0, 1, 3, 2, 4, 5).contiguous().view(B, H, W, -1)
    return x


class WindowAttention(nn.Module):
    r""" Window based multi-head self attention (W-MSA) module with relative position bias.
    It supports both of shifted and non-shifted window.

    Args:
        dim (int): Number of input channels.
        window_size (tuple[int]): The height and width of the window.
        num_heads (int): Number of attention heads.
        qkv_bias (bool, optional):  If True, add a learnable bias to query, key, value. Default: True
        qk_scale (float | None, optional): Override default qk scale of head_dim ** -0.5 if set
        attn_drop (float, optional): Dropout ratio of attention weight. Default: 0.0
        proj_drop (float, optional): Dropout ratio of output. Default: 0.0
    """

    def __init__(self, dim, window_size, num_heads, qkv_bias=True, qk_scale=None, attn_drop=0., proj_drop=0.):

        super().__init__()
        self.dim = dim
        self.window_size = window_size  # Wh, Ww
        self.num_heads = num_heads
        head_dim = dim // num_heads
        self.scale = qk_scale or head_dim ** -0.5

        # define a parameter table of relative position bias
        self.relative_position_bias_table = nn.Parameter(
            torch.zeros((2 * window_size[0] - 1) * (2 * window_size[1] - 1), num_heads))  # 2*Wh-1 * 2*Ww-1, nH

        # get pair-wise relative position index for each token inside the window
        coords_h = torch.arange(self.window_size[0])
        coords_w = torch.arange(self.window_size[1])
        coords = torch.stack(torch.meshgrid([coords_h, coords_w], indexing='ij'))  # 2, Wh, Ww
        coords_flatten = torch.flatten(coords, 1)  # 2, Wh*Ww
        relative_coords = coords_flatten[:, :, None] - coords_flatten[:, None, :]  # 2, Wh*Ww, Wh*Ww
        relative_coords = relative_coords.permute(1, 2, 0).contiguous()  # Wh*Ww, Wh*Ww, 2
        relative_coords[:, :, 0] += self.window_size[0] - 1  # shift to start from 0
        relative_coords[:, :, 1] += self.window_size[1] - 1
        relative_coords[:, :, 0] *= 2 * self.window_size[1] - 1
        relative_position_index = relative_coords.sum(-1)  # Wh*Ww, Wh*Ww
        self.register_buffer("relative_position_index", relative_position_index)

        self.qkv = nn.Linear(dim, dim * 3, bias=qkv_bias)
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = nn.Linear(dim, dim)
        self.proj_drop = nn.Dropout(proj_drop)

        trunc_normal_(self.relative_position_bias_table, std=.02)
        self.softmax = nn.Softmax(dim=-1)

    def forward(self, x, mask=None):
        """
        Args:
            x: input features with shape of (num_windows*B, N, C)
            mask: (0/-inf) mask with shape of (num_windows, Wh*Ww, Wh*Ww) or None
        """
        B_, N, C = x.shape
        qkv = self.qkv(x).reshape(B_, N, 3, self.num_heads, C // self.num_heads).permute(2, 0, 3, 1, 4)
        q, k, v = qkv[0], qkv[1], qkv[2]  # make torchscript happy (cannot use tensor as tuple)

        q = q * self.scale
        attn = (q @ k.transpose(-2, -1))

        relative_position_bias = self.relative_position_bias_table[self.relative_position_index.view(-1)].view(
            self.window_size[0] * self.window_size[1], self.window_size[0] * self.window_size[1], -1)  # Wh*Ww,Wh*Ww,nH
        relative_position_bias = relative_position_bias.permute(2, 0, 1).contiguous()  # nH, Wh*Ww, Wh*Ww
        attn = attn + relative_position_bias.unsqueeze(0)

        if mask is not None:
            nW = mask.shape[0]
            attn = attn.view(B_ // nW, nW, self.num_heads, N, N) + mask.unsqueeze(1).unsqueeze(0)
            attn = attn.view(-1, self.num_heads, N, N)
            attn = self.softmax(attn)
        else:
            attn = self.softmax(attn)

        attn = self.attn_drop(attn)

        x = (attn @ v).transpose(1, 2).reshape(B_, N, C)
        x = self.proj(x)
        x = self.proj_drop(x)
        return x

    def extra_repr(self) -> str:
        return f'dim={self.dim}, window_size={self.window_size}, num_heads={self.num_heads}'

    def flops(self, N):
        # calculate flops for 1 window with token length of N
        flops = 0
        # qkv = self.qkv(x)
        flops += N * self.dim * 3 * self.dim
        # attn = (q @ k.transpose(-2, -1))
        flops += self.num_heads * N * (self.dim // self.num_heads) * N
        #  x = (attn @ v)
        flops += self.num_heads * N * N * (self.dim // self.num_heads)
        # x = self.proj(x)
        flops += N * self.dim * self.dim
        return flops


class SwinTransformerBlock(nn.Module):
    r""" Swin Transformer Block.

    Args:
        dim (int): Number of input channels.
        noise_emb_channels (int): Number of noise embedding channels.
        input_resolution (tuple[int]): Input resulotion.
        num_heads (int): Number of attention heads.
        window_size (int): Window size.
        shift_size (int): Shift size for SW-MSA.
        mlp_ratio (float): Ratio of mlp hidden dim to embedding dim.
        qkv_bias (bool, optional): If True, add a learnable bias to query, key, value. Default: True
        qk_scale (float | None, optional): Override default qk scale of head_dim ** -0.5 if set.
        drop (float, optional): Dropout rate. Default: 0.0
        attn_drop (float, optional): Attention dropout rate. Default: 0.0
        drop_path (float, optional): Stochastic depth rate. Default: 0.0
        act_layer (nn.Module, optional): Activation layer. Default: nn.GELU
        norm_layer (nn.Module, optional): Normalization layer.  Default: nn.LayerNorm
    """

    def __init__(self, dim, noise_emb_channels, input_resolution, num_heads, window_size=7, shift_size=0,
                 mlp_ratio=4., qkv_bias=True, qk_scale=None, drop=0., attn_drop=0., drop_path=0.,
                 act_layer=nn.GELU, norm_layer=nn.LayerNorm,):
        super().__init__()
        self.dim = dim
        self.noise_emb_channels = noise_emb_channels
        self.input_resolution = input_resolution
        self.num_heads = num_heads
        self.window_size = window_size
        self.shift_size = shift_size
        self.mlp_ratio = mlp_ratio
        if min(self.input_resolution) <= self.window_size:
            # if window size is larger than input resolution, we don't partition windows
            self.shift_size = 0
            self.window_size = min(self.input_resolution)
        assert 0 <= self.shift_size < self.window_size, "shift_size must in 0-window_size"

        self.affine = nn.Linear(in_features=noise_emb_channels, out_features=dim * 2)

        self.norm1 = norm_layer(dim)
        self.attn = WindowAttention(
            dim, window_size=to_2tuple(self.window_size), num_heads=num_heads,
            qkv_bias=qkv_bias, qk_scale=qk_scale, attn_drop=attn_drop, proj_drop=drop)

        self.drop_path = DropPath(drop_path) if drop_path > 0. else nn.Identity()
        self.norm2 = norm_layer(dim)
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = Mlp(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop)

        if self.shift_size > 0:
            # calculate attention mask for SW-MSA
            H, W = self.input_resolution
            img_mask = torch.zeros((1, H, W, 1))  # 1 H W 1
            h_slices = (slice(0, -self.window_size),
                        slice(-self.window_size, -self.shift_size),
                        slice(-self.shift_size, None))
            w_slices = (slice(0, -self.window_size),
                        slice(-self.window_size, -self.shift_size),
                        slice(-self.shift_size, None))
            cnt = 0
            for h in h_slices:
                for w in w_slices:
                    img_mask[:, h, w, :] = cnt
                    cnt += 1

            mask_windows = window_partition(img_mask, self.window_size)  # nW, window_size, window_size, 1
            mask_windows = mask_windows.view(-1, self.window_size * self.window_size)
            attn_mask = mask_windows.unsqueeze(1) - mask_windows.unsqueeze(2)
            attn_mask = attn_mask.masked_fill(attn_mask != 0, float(-100.0)).masked_fill(attn_mask == 0, float(0.0))
        else:
            attn_mask = None

        self.register_buffer("attn_mask", attn_mask)

    def forward(self, x, emb):
        H, W = self.input_resolution
        B, L, C = x.shape
        assert L == H * W, "input feature has wrong size"

        # noise conditioning
        params = self.affine(emb).unsqueeze(1).to(x.dtype)
        scale, shift = params.chunk(chunks=2, dim=-1)
        x = silu(torch.addcmul(shift, x, scale + 1))

        shortcut = x
        x = self.norm1(x)
        x = x.view(B, H, W, C)

        # cyclic shift
        if self.shift_size > 0:
            shifted_x = torch.roll(x, shifts=(-self.shift_size, -self.shift_size), dims=(1, 2))
            # partition windows
            x_windows = window_partition(shifted_x, self.window_size)  # nW*B, window_size, window_size, C
        else:
            shifted_x = x
            # partition windows
            x_windows = window_partition(shifted_x, self.window_size)  # nW*B, window_size, window_size, C

        x_windows = x_windows.view(-1, self.window_size * self.window_size, C)  # nW*B, window_size*window_size, C

        # W-MSA/SW-MSA
        attn_windows = self.attn(x_windows, mask=self.attn_mask)  # nW*B, window_size*window_size, C

        # merge windows
        attn_windows = attn_windows.view(-1, self.window_size, self.window_size, C)

        # reverse cyclic shift
        if self.shift_size > 0:
            shifted_x = window_reverse(attn_windows, self.window_size, H, W)  # B H' W' C
            x = torch.roll(shifted_x, shifts=(self.shift_size, self.shift_size), dims=(1, 2))
        else:
            shifted_x = window_reverse(attn_windows, self.window_size, H, W)  # B H' W' C
            x = shifted_x
        x = x.view(B, H * W, C)
        x = shortcut + self.drop_path(x)

        # FFN
        x = x + self.drop_path(self.mlp(self.norm2(x)))

        return x

    def extra_repr(self) -> str:
        return f"dim={self.dim}, input_resolution={self.input_resolution}, num_heads={self.num_heads}, " \
               f"window_size={self.window_size}, shift_size={self.shift_size}, mlp_ratio={self.mlp_ratio}"

    def flops(self):
        flops = 0
        H, W = self.input_resolution
        # norm1
        flops += self.dim * H * W
        # W-MSA/SW-MSA
        nW = H * W / self.window_size / self.window_size
        flops += nW * self.attn.flops(self.window_size * self.window_size)
        # mlp
        flops += 2 * H * W * self.dim * self.dim * self.mlp_ratio
        # norm2
        flops += self.dim * H * W
        return flops


class PatchMerging(nn.Module):
    r""" Patch Merging Layer.

    Args:
        input_resolution (tuple[int]): Resolution of input feature.
        dim (int): Number of input channels.
        norm_layer (nn.Module, optional): Normalization layer.  Default: nn.LayerNorm
    """

    def __init__(self, input_resolution, dim, norm_layer=nn.LayerNorm):
        super().__init__()
        self.input_resolution = input_resolution
        self.dim = dim
        self.reduction = nn.Linear(4 * dim, 2 * dim, bias=False)
        self.norm = norm_layer(4 * dim)

    def forward(self, x):
        """
        x: B, H*W, C
        """
        H, W = self.input_resolution
        B, L, C = x.shape
        assert L == H * W, "input feature has wrong size"
        assert H % 2 == 0 and W % 2 == 0, f"x size ({H}*{W}) are not even."

        x = x.view(B, H, W, C)

        x0 = x[:, 0::2, 0::2, :]  # B H/2 W/2 C
        x1 = x[:, 1::2, 0::2, :]  # B H/2 W/2 C
        x2 = x[:, 0::2, 1::2, :]  # B H/2 W/2 C
        x3 = x[:, 1::2, 1::2, :]  # B H/2 W/2 C
        x = torch.cat([x0, x1, x2, x3], -1)  # B H/2 W/2 4*C
        x = x.view(B, -1, 4 * C)  # B H/2*W/2 4*C

        x = self.norm(x)
        x = self.reduction(x)

        return x

    def extra_repr(self) -> str:
        return f"input_resolution={self.input_resolution}, dim={self.dim}"

    def flops(self):
        H, W = self.input_resolution
        flops = H * W * self.dim
        flops += (H // 2) * (W // 2) * 4 * self.dim * 2 * self.dim
        return flops


class PatchBreakup(nn.Module):
    r""" Patch Breakup Layer.

    Args:
        input_resolution (tuple[int]): Resolution of input feature.
        dim (int): Number of input channels.
        norm_layer (nn.Module, optional): Normalization layer.  Default: nn.LayerNorm
        skip_connection: bool
    """

    def __init__(self, input_resolution, dim, norm_layer=nn.LayerNorm, skip_connection=True):
        super().__init__()
        self.input_resolution = input_resolution
        self.dim = dim

        if skip_connection:
            self.pre_linear = nn.Linear(dim, dim, bias=False)
            self.norm = norm_layer(dim)
            self.true_dim = dim // 2
        else:
            self.pre_linear = nn.Linear(dim, 2 * dim, bias=False)
            self.norm = norm_layer(2 * dim)
            self.true_dim = dim

        self.post_linear = nn.Linear(self.true_dim // 2, self.true_dim // 2, bias=False)
        self.post_norm = norm_layer(self.true_dim // 2)

    def forward(self, x):
        """
        x: B, H*W, C
        """
        H, W = self.input_resolution
        B, L, C = x.shape
        assert L == H * W, "input feature has wrong size"
        assert H % 2 == 0 and W % 2 == 0, f"x size ({H}*{W}) are not even."
        assert C == self.dim

        # symmetric inverse of patch merging process
        x = self.pre_linear(x)  # [B, L, 2C_t]
        x = self.norm(x)  # [B, L, 2C_t]

        C_out = self.true_dim // 2
        x = x.view(B, H, W, 4, C_out)  # [B, H, W, 4, C_t/2]
        x0, x1, x2, x3 = x.chunk(4, dim=3)  # [B, H, W, 1, C_t/2] * 4

        x_out = torch.zeros([B, 2 * H, 2 * W, 1, C_out]).to(x)  # [B, 2H, 2W, 1, C_t/2], upsample tensor shape

        x_out[:, 0::2, 0::2, :] = x0  # B H W 1 C_t/2
        x_out[:, 1::2, 0::2, :] = x1  # B H W 1 C_t/2
        x_out[:, 0::2, 1::2, :] = x2  # B H W 1 C_t/2
        x_out[:, 1::2, 1::2, :] = x3  # B H W 1 C_t/2
        x_out = x_out.view(B, -1, C_out)  # B H*W*4 C_t/2

        x_out = self.post_norm(x_out)
        x_out = self.post_linear(x_out)  # B H*W*4 C_t/2

        return x_out

    def extra_repr(self) -> str:
        return f"input_resolution={self.input_resolution}, dim={self.dim}"

    def flops(self):
        H, W = self.input_resolution
        flops = H * W * self.dim
        flops += (H // 2) * (W // 2) * 4 * self.dim * 2 * self.dim
        return flops


class BasicLayer(nn.Module):
    """ A basic Swin Transformer layer for one stage.

    Args:
        dim (int): Number of input channels.
        noise_emb_channels (int): Number of noise embedding channels.
        input_resolution (tuple[int]): Input resolution.
        depth (int): Number of blocks.
        num_heads (int): Number of attention heads.
        window_size (int): Local window size.
        mlp_ratio (float): Ratio of mlp hidden dim to embedding dim.
        qkv_bias (bool, optional): If True, add a learnable bias to query, key, value. Default: True
        qk_scale (float | None, optional): Override default qk scale of head_dim ** -0.5 if set.
        drop (float, optional): Dropout rate. Default: 0.0
        attn_drop (float, optional): Attention dropout rate. Default: 0.0
        drop_path (float | tuple[float], optional): Stochastic depth rate. Default: 0.0
        norm_layer (nn.Module, optional): Normalization layer. Default: nn.LayerNorm
        downsample (nn.Module | None, optional): Downsample layer at the end of the layer. Default: None
        upsample (nn.Module | None, optional): Upsample layer at the end of the layer. Default: None
        use_checkpoint (bool): Whether to use checkpointing to save memory. Default: False.
    """

    def __init__(self, dim, noise_emb_channels, input_resolution, depth, num_heads, window_size,
                 mlp_ratio=4., qkv_bias=True, qk_scale=None, drop=0., attn_drop=0.,
                 drop_path=0., norm_layer=nn.LayerNorm, downsample=None, upsample=None, use_checkpoint=False):

        super().__init__()
        self.dim = dim
        self.input_resolution = input_resolution
        self.depth = depth
        self.use_checkpoint = use_checkpoint

        # patch breakup layer
        if upsample is not None:
            # we always have skip connection when upsampling
            self.upsample = upsample(input_resolution, dim=dim * 4, norm_layer=norm_layer)
            input_resolution = tuple(item * 2 for item in input_resolution)
        else:
            self.upsample = None

        # build blocks
        self.blocks = nn.ModuleList([
            SwinTransformerBlock(dim=dim, noise_emb_channels=noise_emb_channels, input_resolution=input_resolution,
                                 num_heads=num_heads, window_size=window_size,
                                 shift_size=0 if (i % 2 == 0) else window_size // 2,
                                 mlp_ratio=mlp_ratio,
                                 qkv_bias=qkv_bias, qk_scale=qk_scale,
                                 drop=drop, attn_drop=attn_drop,
                                 drop_path=drop_path[i] if isinstance(drop_path, list) else drop_path,
                                 norm_layer=norm_layer)
            for i in range(depth)])

        # patch merging layer
        if downsample is not None:
            self.downsample = downsample(input_resolution, dim=dim, norm_layer=norm_layer)
        else:
            self.downsample = None

    def forward(self, x, emb):
        if self.upsample is not None:
            x = self.upsample(x)
        for blk in self.blocks:
            if self.use_checkpoint:
                x = checkpoint.checkpoint(blk, x)
            else:
                x = blk(x, emb)
        if self.downsample is not None:
            x = self.downsample(x)
        return x

    def extra_repr(self) -> str:
        return f"dim={self.dim}, input_resolution={self.input_resolution}, depth={self.depth}"

    def flops(self):
        flops = 0
        for blk in self.blocks:
            flops += blk.flops()
        if self.downsample is not None:
            flops += self.downsample.flops()
        return flops


#----------------------------------------------------------------------------
# Timestep embedding used in the DDPM++ and ADM architectures.

class PositionalEmbedding(torch.nn.Module):
    def __init__(self, num_channels, max_positions=10000, endpoint=False):
        super().__init__()
        self.num_channels = num_channels
        self.max_positions = max_positions
        self.endpoint = endpoint

    def forward(self, x):
        freqs = torch.arange(start=0, end=self.num_channels//2, dtype=torch.float32, device=x.device)
        freqs = freqs / (self.num_channels // 2 - (1 if self.endpoint else 0))
        freqs = (1 / self.max_positions) ** freqs
        x = x.ger(freqs.to(x.dtype))
        x = torch.cat([x.cos(), x.sin()], dim=1)
        return x


class PatchEmbed(nn.Module):
    r""" Image to Patch Embedding

    Args:
        img_size (int): Image size.  Default: 224.
        patch_size (int): Patch token size. Default: 4.
        in_chans (int): Number of input image channels. Default: 3.
        embed_dim (int): Number of linear projection output channels. Default: 96.
        noise_emb_channels(int): Number of noise embedding vector channels. Default: 512.
        norm_layer (nn.Module, optional): Normalization layer. Default: None
        node_pe (bool, optional): Whether to use node positional embedding. Default: False
    """

    def __init__(self, img_size=224, patch_size=4, in_chans=3, embed_dim=96, noise_emb_channels=512, norm_layer=None,
                 node_pe=False):
        super().__init__()
        img_size = to_2tuple(img_size)
        patch_size = to_2tuple(patch_size)
        patches_resolution = [img_size[0] // patch_size[0], img_size[1] // patch_size[1]]
        self.img_size = img_size
        self.patch_size = patch_size
        self.patches_resolution = patches_resolution
        self.num_patches = patches_resolution[0] * patches_resolution[1]

        self.in_chans = in_chans
        self.embed_dim = embed_dim

        # add node order-based PE
        if node_pe:
            self.node_feat_pe_mlp = nn.Sequential(
                PositionalEmbedding(embed_dim),
                nn.Linear(embed_dim, embed_dim),
                nn.GELU(),
                nn.Linear(embed_dim, embed_dim),
            )
        else:
            self.node_feat_pe_mlp = None

        self.affine = nn.Linear(in_features=noise_emb_channels, out_features=embed_dim * 2)

        self.proj = nn.Conv2d(in_chans, embed_dim, kernel_size=patch_size, stride=patch_size)
        if norm_layer is not None:
            self.norm = norm_layer(embed_dim)
        else:
            self.norm = None

    def forward(self, x, emb, node_flags):
        B, _, H, W = x.shape
        assert H == self.img_size[0] and W == self.img_size[1], \
            f"Input image size ({H}*{W}) doesn't match model ({self.img_size[0]}*{self.img_size[1]})."

        assert len(x.shape) == 4

        x = self.proj(x).flatten(2).transpose(1, 2)  # B Ph*Pw C
        if self.norm is not None:
            x = self.norm(x)

        # noise conditioning
        params = self.affine(emb).unsqueeze(1).to(x.dtype)
        scale, shift = params.chunk(chunks=2, dim=-1)
        x = silu(torch.addcmul(shift, x, scale + 1))
        return x

    def flops(self):
        Ho, Wo = self.patches_resolution
        flops = Ho * Wo * self.embed_dim * self.in_chans * (self.patch_size[0] * self.patch_size[1])
        if self.norm is not None:
            flops += Ho * Wo * self.embed_dim
        return flops


class SwinGNN(nn.Module):
    """
    Swin GNN
    Args:
        img_size (int | tuple(int)): Input image size. Default 224
        patch_size (int | tuple(int)): Patch size. Default: 4
        in_chans (int): Number of input image channels. Default: 3
        embed_dim (int): Patch embedding dimension. Default: 96
        depths (tuple(int)): Depth of each Swin Transformer layer.
        num_heads (tuple(int)): Number of attention heads in different layers.
        window_size (int): Window size. Default: 7
        mlp_ratio (float): Ratio of mlp hidden dim to embedding dim. Default: 4
        qkv_bias (bool): If True, add a learnable bias to query, key, value. Default: True
        qk_scale (float): Override default qk scale of head_dim ** -0.5 if set. Default: None
        drop_rate (float): Dropout rate. Default: 0
        attn_drop_rate (float): Attention dropout rate. Default: 0
        drop_path_rate (float): Stochastic depth rate. Default: 0.1
        norm_layer (nn.Module): Normalization layer. Default: nn.LayerNorm.
        patch_norm (bool): If True, add normalization after patch embedding. Default: True
        use_checkpoint (bool): Whether to use checkpointing to save memory. Default: False
        self_condition (bool): Whether to use self conditioning. Default: False.
    """

    def __init__(self, img_size=224, patch_size=4, in_chans=3,
                 embed_dim=96, depths=[2, 2, 6, 2], num_heads=[3, 6, 12, 24],
                 window_size=7, mlp_ratio=4., qkv_bias=True, qk_scale=None,
                 drop_rate=0., attn_drop_rate=0., drop_path_rate=0.1,
                 norm_layer=nn.LayerNorm, patch_norm=True,
                 use_checkpoint=False,
                 self_condition=False,
                 **kwargs):
        super().__init__()

        self.num_layers = len(depths)
        self.embed_dim = embed_dim
        self.patch_norm = patch_norm
        self.num_features = int(embed_dim * 2 ** (self.num_layers - 1))
        self.mlp_ratio = mlp_ratio

        # self conditioning
        self.self_condition = self_condition
        if self_condition:
            in_chans = in_chans * 2
        else:
            in_chans = in_chans

        # split image into non-overlapping patches
        noise_emb_channels = 512
        self.patch_embed = PatchEmbed(
            img_size=img_size, patch_size=patch_size, in_chans=in_chans, embed_dim=embed_dim,
            noise_emb_channels=noise_emb_channels,
            norm_layer=norm_layer if self.patch_norm else None)
        num_patches = self.patch_embed.num_patches
        patches_resolution = self.patch_embed.patches_resolution
        self.patches_resolution = patches_resolution

        self.pos_drop = nn.Dropout(p=drop_rate)

        # stochastic depth
        dpr = [x.item() for x in torch.linspace(0, drop_path_rate, sum(depths))]  # stochastic depth decay rule

        # build layers
        self.down_layers = nn.ModuleList()
        for i_layer in range(self.num_layers):
            layer = BasicLayer(dim=int(embed_dim * 2 ** i_layer),
                               noise_emb_channels=noise_emb_channels,
                               input_resolution=(patches_resolution[0] // (2 ** i_layer),
                                                 patches_resolution[1] // (2 ** i_layer)),
                               depth=depths[i_layer],
                               num_heads=num_heads[i_layer],
                               window_size=window_size,
                               mlp_ratio=self.mlp_ratio,
                               qkv_bias=qkv_bias, qk_scale=qk_scale,
                               drop=drop_rate, attn_drop=attn_drop_rate,
                               drop_path=dpr[sum(depths[:i_layer]):sum(depths[:i_layer + 1])],
                               norm_layer=norm_layer,
                               downsample=PatchMerging if (i_layer < self.num_layers - 1) else None,
                               upsample=None,
                               use_checkpoint=use_checkpoint)
            self.down_layers.append(layer)

        self.up_layers = nn.ModuleList()

        for i_layer in range(self.num_layers):
            rest_layer = self.num_layers - i_layer - 1
            if i_layer == 0:
                # no skip connection
                input_resolution = (patches_resolution[0] // (2 ** rest_layer),
                                    patches_resolution[1] // (2 ** rest_layer))
            else:
                # with skip connection
                input_resolution = (patches_resolution[0] // (2 ** (rest_layer+1)),
                                    patches_resolution[1] // (2 ** (rest_layer+1)))
            layer = BasicLayer(dim=int(embed_dim * 2 ** rest_layer),
                               noise_emb_channels=noise_emb_channels,
                               input_resolution=input_resolution,
                               depth=depths[rest_layer],
                               num_heads=num_heads[rest_layer],
                               window_size=window_size,
                               mlp_ratio=self.mlp_ratio,
                               qkv_bias=qkv_bias, qk_scale=qk_scale,
                               drop=drop_rate, attn_drop=attn_drop_rate,
                               drop_path=dpr[sum(depths[:rest_layer]):sum(depths[:rest_layer + 1])],
                               norm_layer=norm_layer,
                               downsample=None,
                               upsample=PatchBreakup if (i_layer > 0) else None,
                               use_checkpoint=use_checkpoint)
            self.up_layers.append(layer)

        # [B, D, pH, pW] -> [B, 1, H, W]
        self.read_out = nn.Sequential(
            nn.ConvTranspose2d(embed_dim, embed_dim, kernel_size=patch_size, stride=patch_size),
            nn.Conv2d(embed_dim, embed_dim, kernel_size=1, stride=1, padding=0),
            nn.Conv2d(embed_dim, 1, kernel_size=1, stride=1, padding=0)
        )

        # Mapping. Following EDM network.
        self.map_noise = PositionalEmbedding(num_channels=embed_dim)
        self.map_layer0 = nn.Linear(in_features=embed_dim, out_features=noise_emb_channels)
        self.map_layer1 = nn.Linear(in_features=noise_emb_channels, out_features=noise_emb_channels)

        self.norm = norm_layer(self.embed_dim)

        self.apply(self._init_weights)

    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            trunc_normal_(m.weight, std=.02)
            if isinstance(m, nn.Linear) and m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, nn.LayerNorm):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)

    @torch.jit.ignore
    def no_weight_decay(self):
        return {'absolute_pos_embed'}

    @torch.jit.ignore
    def no_weight_decay_keywords(self):
        return {'relative_position_bias_table'}

    def forward_features(self, x, emb, node_flags, noise_labels):
        B, _, H, W = x.shape
        x = self.patch_embed(x, emb, node_flags)  # B L C
        x = self.pos_drop(x)

        # Encoder.
        skips = []
        for layer in self.down_layers:
            x = layer(x, emb)
            skips.append(x)

        # Decoder.
        for layer in self.up_layers:
            if layer.upsample is not None:
                x = torch.cat([x, skips.pop()], dim=-1)
            else:
                skips.pop()
            x = layer(x, emb)

        x = self.norm(x)  # B L C

        x = x.view(B, *self.patches_resolution, -1).permute(0, 3, 1, 2).contiguous()
        x = self.read_out(x).squeeze(1)  # B H W

        return x

    def forward(self, x, node_flags, noise_labels, self_cond=None):
        # Mapping. Noise conditioning.
        emb = self.map_noise(noise_labels)
        emb = silu(self.map_layer0(emb))
        emb = self.map_layer1(emb)
        emb = silu(emb)
        def _shape_trimming(in_tensor):
            if len(in_tensor.shape) == 3:
                in_tensor = in_tensor.unsqueeze(1)  # [B, 1, N, N] <- [B, N, N]
            return in_tensor

        x = _shape_trimming(x)

        # self_conditioning
        if self.self_condition:
            x_self_cond = torch.zeros_like(x) if self_cond is None else _shape_trimming(self_cond)
            x = torch.cat([x_self_cond, x], dim=1)  # [B, 2, N, N]

        x = self.forward_features(x, emb, node_flags, noise_labels)

        # Enforce matrix symmetry
        x = x.squeeze(1)  # [B, N, N]
        x = mask_adjs(x, node_flags)
        x = 0.5 * (x + x.transpose(-1, -2))
        return x


class NodeAdjSwinGNN(nn.Module):
    """ Swin Transformer for node and adjacency matrix embeddings.

    Args:
        img_size (int | tuple(int)): Input image size. Default 224
        patch_size (int | tuple(int)): Patch size. Default: 4
        in_chans (int): Number of input image channels. Default: 3
        embed_dim (int): Patch embedding dimension. Default: 96
        depths (tuple(int)): Depth of each Swin Transformer layer.
        num_heads (tuple(int)): Number of attention heads in different layers.
        window_size (int): Window size. Default: 7
        mlp_ratio (float): Ratio of mlp hidden dim to embedding dim. Default: 4
        qkv_bias (bool): If True, add a learnable bias to query, key, value. Default: True
        qk_scale (float): Override default qk scale of head_dim ** -0.5 if set. Default: None
        drop_rate (float): Dropout rate. Default: 0
        attn_drop_rate (float): Attention dropout rate. Default: 0
        drop_path_rate (float): Stochastic depth rate. Default: 0.1
        norm_layer (nn.Module): Normalization layer. Default: nn.LayerNorm.
        patch_norm (bool): If True, add normalization after patch embedding. Default: True
        use_checkpoint (bool): Whether to use checkpointing to save memory. Default: False
        self_condition (bool): Whether to use self conditioning. Default: False.
    """

    def __init__(self, img_size=224, patch_size=4, in_chans=3,
                 embed_dim=96, depths=[2, 2, 6, 2], num_heads=[3, 6, 12, 24],
                 window_size=7, mlp_ratio=4., qkv_bias=True, qk_scale=None,
                 drop_rate=0., attn_drop_rate=0., drop_path_rate=0.1,
                 out_chans_adj=1, out_chans_node=1,
                 norm_layer=nn.LayerNorm, patch_norm=True,
                 use_checkpoint=False,
                 self_condition=False,
                 symmetric_noise=True,
                 **kwargs):
        super().__init__()

        self.num_layers = len(depths)
        self.embed_dim = embed_dim
        self.patch_norm = patch_norm
        self.num_features = int(embed_dim * 2 ** (self.num_layers - 1))
        self.mlp_ratio = mlp_ratio

        # self conditioning
        self.self_condition = self_condition
        if self_condition:
            in_chans = in_chans * 2
            # feat_dim = 8
        else:
            in_chans = in_chans
            # feat_dim = 4

        self.symmetric_noise = symmetric_noise
        self.out_chans_adj = out_chans_adj  # adj output dim
        self.out_chans_node = out_chans_node  # node output dim

        # split image into non-overlapping patches
        noise_emb_channels = 512
        self.patch_embed = PatchEmbed(
            img_size=img_size, patch_size=patch_size, in_chans=in_chans, embed_dim=embed_dim,
            noise_emb_channels=noise_emb_channels,
            norm_layer=norm_layer if self.patch_norm else None)
        num_patches = self.patch_embed.num_patches
        patches_resolution = self.patch_embed.patches_resolution
        self.patches_resolution = patches_resolution

        self.pos_drop = nn.Dropout(p=drop_rate)

        # stochastic depth
        dpr = [x.item() for x in torch.linspace(0, drop_path_rate, sum(depths))]  # stochastic depth decay rule

        # build layers
        self.down_layers = nn.ModuleList()
        for i_layer in range(self.num_layers):
            layer = BasicLayer(dim=int(embed_dim * 2 ** i_layer),
                               noise_emb_channels=noise_emb_channels,
                               input_resolution=(patches_resolution[0] // (2 ** i_layer),
                                                 patches_resolution[1] // (2 ** i_layer)),
                               depth=depths[i_layer],
                               num_heads=num_heads[i_layer],
                               window_size=window_size,
                               mlp_ratio=self.mlp_ratio,
                               qkv_bias=qkv_bias, qk_scale=qk_scale,
                               drop=drop_rate, attn_drop=attn_drop_rate,
                               drop_path=dpr[sum(depths[:i_layer]):sum(depths[:i_layer + 1])],
                               norm_layer=norm_layer,
                               downsample=PatchMerging if (i_layer < self.num_layers - 1) else None,
                               upsample=None,
                               use_checkpoint=use_checkpoint)
            self.down_layers.append(layer)

        self.up_layers = nn.ModuleList()

        for i_layer in range(self.num_layers):
            rest_layer = self.num_layers - i_layer - 1
            if i_layer == 0:
                # no skip connection
                input_resolution = (patches_resolution[0] // (2 ** rest_layer),
                                    patches_resolution[1] // (2 ** rest_layer))
            else:
                # with skip connection
                input_resolution = (patches_resolution[0] // (2 ** (rest_layer+1)),
                                    patches_resolution[1] // (2 ** (rest_layer+1)))
            layer = BasicLayer(dim=int(embed_dim * 2 ** rest_layer),
                               noise_emb_channels=noise_emb_channels,
                               input_resolution=input_resolution,
                               depth=depths[rest_layer],
                               num_heads=num_heads[rest_layer],
                               window_size=window_size,
                               mlp_ratio=self.mlp_ratio,
                               qkv_bias=qkv_bias, qk_scale=qk_scale,
                               drop=drop_rate, attn_drop=attn_drop_rate,
                               drop_path=dpr[sum(depths[:rest_layer]):sum(depths[:rest_layer + 1])],
                               norm_layer=norm_layer,
                               downsample=None,
                               upsample=PatchBreakup if (i_layer > 0) else None,
                               use_checkpoint=use_checkpoint)
            self.up_layers.append(layer)

        # [B, D, pH, pW] -> [B, D, H, W]
        self.read_out = nn.Sequential(
            nn.ConvTranspose2d(embed_dim, embed_dim, kernel_size=patch_size, stride=patch_size),
            nn.Conv2d(embed_dim, embed_dim, kernel_size=1, stride=1, padding=0),
            nn.Conv2d(embed_dim, embed_dim, kernel_size=1, stride=1, padding=0)
        )

        # Mapping. Following EDM network.
        self.map_noise = PositionalEmbedding(num_channels=embed_dim)
        self.map_layer0 = nn.Linear(in_features=embed_dim, out_features=noise_emb_channels)
        self.map_layer1 = nn.Linear(in_features=noise_emb_channels, out_features=noise_emb_channels)

        self.norm = norm_layer(self.embed_dim)

        self.readout_adj_mlp = Mlp(in_features=embed_dim, hidden_features=embed_dim, out_features=out_chans_adj)  # emb to adj value
        self.readout_node_mlp = Mlp(in_features=embed_dim, hidden_features=embed_dim, out_features=out_chans_node)  # emb to feat value
        self.apply(self._init_weights)

    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            trunc_normal_(m.weight, std=.02)
            if isinstance(m, nn.Linear) and m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, nn.LayerNorm):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)

    @torch.jit.ignore
    def no_weight_decay(self):
        return {'absolute_pos_embed'}

    @torch.jit.ignore
    def no_weight_decay_keywords(self):
        return {'relative_position_bias_table'}

    def forward_features(self, x, emb, node_flags, noise_labels):
        B, _, H, W = x.shape
        x = self.patch_embed(x, emb, node_flags)  # B L C
        x = self.pos_drop(x)

        # Encoder.
        skips = []
        for layer in self.down_layers:
            x = layer(x, emb)
            skips.append(x)

        # Decoder.
        for layer in self.up_layers:
            if layer.upsample is not None:
                x = torch.cat([x, skips.pop()], dim=-1)
            else:
                skips.pop()
            x = layer(x, emb)

        x = self.norm(x)  # B L C

        x = x.view(B, *self.patches_resolution, -1).permute(0, 3, 1, 2).contiguous()
        x = self.read_out(x)  # B C H W

        return x

    def forward(self, adj, node, node_flags, noise_labels, self_cond_x=None, self_cond_feat=None):
        # Mapping. Noise conditioning.
        emb = self.map_noise(noise_labels)
        emb = silu(self.map_layer0(emb))
        emb = self.map_layer1(emb)
        emb = silu(emb)
        def _shape_trimming_x(in_tensor):
            if len(in_tensor.shape) == 3:
                in_tensor = in_tensor.unsqueeze(1)  # [B, 1, N, N] <- [B, N, N]
            return in_tensor

        def _shape_trimming_feat(in_tensor):
            if len(in_tensor.shape) == 2:
                in_tensor = in_tensor.unsqueeze(1)  # [B, 1, N] <- [B, N]
            elif len(in_tensor.shape) == 3:
                in_tensor = in_tensor.permute(0, 2, 1)  # [B, C, N] <- [B, N, C]
            return in_tensor

        adj = _shape_trimming_x(adj)  # [B, 1, N, N]

        node = node.to(torch.float32)  # [B, N]
        node = _shape_trimming_feat(node)  # [B, C, N]

        # self_conditioning
        if self.self_condition:
            adj_self_cond = torch.zeros_like(adj) if self_cond_x is None else _shape_trimming_x(self_cond_x)
            adj = torch.cat([adj_self_cond, adj], dim=1)  # [B, 2 * D, N, N]
            node_self_cond = torch.zeros_like(node) if self_cond_feat is None else _shape_trimming_feat(self_cond_feat)
            node = torch.cat([node_self_cond, node], dim=1)  # [B, 2 * C, N]

        # concatenate node edge values directly
        node_mat = node.unsqueeze(-1).expand(-1, -1, -1, node.size(-1))  # [B, C, N, N]
        node_mat_t = node_mat.transpose(-1, -2)  # [B, C, N, N]
        node_mat_cat = torch.cat([node_mat, node_mat_t], dim=1)  # [B, 2*C, N, N]
        node_mat_cat = mask_adjs(node_mat_cat, node_flags)  # [B, 2*C, N, N]

        adj_cat = torch.cat([adj, node_mat_cat], dim=1)  # [B, 2 + 2*2, N, N] = [B, 6, N, N] for self-conditioning mode
        shared_rep = self.forward_features(adj_cat, emb, node_flags, noise_labels)  # [B, C_out, N, N]

        # final layer to get edge/node update
        if self.out_chans_adj == 1:
            adj_out = self.readout_adj_mlp(shared_rep.permute(0, 2, 3, 1)).squeeze(-1)  # [B, N, N]
        else:
            adj_out = self.readout_adj_mlp(shared_rep.permute(0, 2, 3, 1)).permute(0, 3, 1, 2)  # [B, C, N, N]

        # padding aware pooling
        shared_rep_ = mask_adjs(shared_rep, node_flags, in_place=False)  # [B, C, N, N]
        node_feat_vec = shared_rep_.mean(dim=-1)  # [B, C, N]

        if self.out_chans_node == 1:
            node_out = self.readout_node_mlp(node_feat_vec.permute(0, 2, 1)).squeeze(-1)  # [B, N]
        else:
            node_out = self.readout_node_mlp(node_feat_vec.permute(0, 2, 1))  # [B, N, C]

        # Enforce matrix symmetry
        node_out = mask_nodes(node_out, node_flags)
        adj_out = mask_adjs(adj_out, node_flags)
        if self.symmetric_noise:
            adj_out = 0.5 * (adj_out + adj_out.transpose(-1, -2))
        else:
            pass
        return adj_out, node_out
