import os
import re
import json
import json5
from typing import Tuple, Union
from ray.util import pdb

from .base import BaseSystem

TOOL_VERSION = os.getenv('TOOL_VERSION', 'en_v3')
TOOL_CALL_TAG = os.getenv('TOOL_CALL_TAG', 'tool_call')
REQUIRED_KEY_DICT = {
    'en_v2': {"name", "arguments", "objective"},
    'en_v3': {"purpose", "name", "arguments"},
    'en_v3_code': {"purpose", "name", "arguments"},
}

NAME_DICT = {
    'en_v2': "name",
    'en_v3': "name",
    'en_v3_code': "name",
}

PURPOSE_DICT = {
    'en_v2': "objective",
    'en_v3': "purpose",
    'en_v3_code': "purpose",
}

ANSWER_TAG = None if os.getenv('ANSWER_TAG', 'answer').lower() == "none" else os.getenv('ANSWER_TAG', 'answer')

class System2(BaseSystem):
# class System2():
    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        
    @staticmethod
    def parse_output(text: Union[str, None]):
        if text is None: # exceed max_context_length
            return None, None
        if TOOL_CALL_TAG == "tools":
            tool_part = System2.extract_tools_string(text)
        else:
            tool_part = System2.extract_tools(text)
        answer_part = System2.extract_answers(text)
        return tool_part, answer_part

    @staticmethod
    def extract_answers(text: str) -> dict | None:
        if ANSWER_TAG is not None:
            pattern = r"<{TAG}>(.*?)</{TAG}>".format(TAG=ANSWER_TAG)
            match = re.search(pattern, text, re.DOTALL)
            if match:
                answer_output = match.group(1).strip()
                boxed_answer = System2.extract_box(answer_output)
                return {
                    "answer_tag_part": answer_output,  # the output in <answer> section
                    "boxed_answer": boxed_answer,
                }
            return None
        else:
            boxed_answer = System2.extract_box(text)
            if boxed_answer is not None:
                return {
                    "answer_tag_part": "ANSWER_TAG is None",
                    "boxed_answer": boxed_answer,
                }
            else:
                return None # must return None if boxed_answer is None

    @staticmethod
    def extract_tools(text: str) -> dict | None:
        pattern = r"<{tag}>(.*?)</{tag}>".format(tag=TOOL_CALL_TAG)
        code_pattern = r"<code>(.*?)</code>"
        # pattern_incomplete = r"<tools>(.*?)$"

        match = re.search(pattern, text, re.DOTALL)
        if match:
            tool_content = match.group(1).strip()
            code_match = re.search(code_pattern, tool_content, re.DOTALL)
            if code_match:
                code_content = code_match.group(1).strip()
                # 移除代码块部分，留下JSON
                tool_content = tool_content[:code_match.start()].strip()

            try:
                tool_params = json5.loads(tool_content)
            except Exception as e:
                print(f"Error parsing JSON: {e}")
                return None
            
            # error check
            valid = System2.validate_tool_params(tool_params, has_code=True if code_match else False)

            if valid:
                if code_match:
                    tool_params['arguments']['code'] = code_content

                return {
                    "tool_name": str(tool_params[NAME_DICT[TOOL_VERSION]]), # 强制str，防止报错 if tool_name not in self.function_map:
                    "purpose": str(tool_params[PURPOSE_DICT[TOOL_VERSION]]),
                    "tool_args": tool_params['arguments'],
                    "messages": [],
                } # tool_params
            
        # match_incomplete = re.search(pattern_incomplete, text, re.DOTALL)
        # if match_incomplete:
        #     return match_incomplete.group(1).strip()
        return None
    
    @staticmethod
    def validate_tool_params(tool_params, has_code=False) -> bool:
        # 检查必需的键
        # allowed_keys = {"name", "arguments", "objective"}
        # # 检查是否只包含允许的键
        # if not set(tool_params.keys()).issubset(allowed_keys):
        #     # print(f"Invalid keys in tool_params: {set(tool_params.keys()) - allowed_keys}")
        #     return False

        # tool_params must be dict
        if not isinstance(tool_params, dict):
            return False

        required_keys = REQUIRED_KEY_DICT[TOOL_VERSION]
        if set(tool_params.keys()) != required_keys:
            return False
        
        # 如果有代码块，验证 arguments 格式
        if has_code:
            args = tool_params.get("arguments", {})
            if not isinstance(args, dict) or list(args.keys()) != ["code"] or args["code"] != "":
                # print("For code tools, arguments must be exactly {'code': ''}")
                return False
                
        return True

    @staticmethod
    def extract_tools_string(text: str) -> dict | None:
        pattern = r"<tools>(.*?)</tools>"
        # pattern_incomplete = r"<tools>(.*?)$"
        
        match = re.search(pattern, text, re.DOTALL)
        if match:
            tool_text = match.group(1).strip()
            tool_pattern = r'FUNCTION:\s*(.*?)\s*PURPOSE:\s*(.*?)\s*ARGS:\s*(.*)$'
            tool_match = re.search(tool_pattern, tool_text, re.DOTALL)
    
            if tool_match:
                function_name = tool_match.group(1).strip()
                purpose = tool_match.group(2).strip()
                args = tool_match.group(3).strip()
                # try:
                #     args = json.loads(args)  # 尝试将args解析为JSON
                # except json.JSONDecodeError:
                #     pass  # 如果不是有效的JSON，保持原字符串
                
                return {
                    "tool_name": str(function_name),
                    "purpose": str(purpose),
                    "tool_args": args,
                    "messages": [],
                }
            
        # match_incomplete = re.search(pattern_incomplete, text, re.DOTALL)
        # if match_incomplete:
        #     return match_incomplete.group(1).strip()
        return None

    @staticmethod
    def extract_box(solution_text: str) -> str:
        def find_balanced_braces(text: str, start_index: int) -> Tuple[str, int]:
            """
            Finds the content within balanced braces starting from start_index.
            Args:
                text (str): The text to search within.
                start_index (int): The index to start searching from (right after the opening brace).
            Returns:
                tuple: A tuple containing the extracted content and the index of the closing brace.
                    Returns (None, -1) if no balanced braces are found.
            """
            stack = ['{']  
            content = [] 
            for i, char in enumerate(text[start_index:], start=start_index):
                if char == '{':
                    stack.append('{')
                    content.append(char)
                elif char == '}':
                    stack.pop()
                    if not stack:
                        # Found the matching closing brace for \boxed{
                        return ''.join(content), i
                    content.append(char)
                else:
                    content.append(char)
            return None, -1
        
        # 找第一个
        # match = re.search(r'\\boxed\{', solution_text)
        # if match:
        #     start_index = match.end()
        #     content, end_index = find_balanced_braces(solution_text, start_index)
        #     if content is not None:
        #         return content.strip()
        # return None

        # 找到所有 \boxed{ 的位置
        matches = list(re.finditer(r'\\boxed\{', solution_text))
        if matches:
            # 取最后一个匹配
            last_match = matches[-1]
            start_index = last_match.end()
            content, end_index = find_balanced_braces(solution_text, start_index)
            if content is not None:
                return content.strip()
        return None

if __name__=="__main__":
    texts = [
        # '<research>Wolfhart Pannenberg\'s key influences in his engagement with physics and his reinterpretation of history can be difficult to deduce without a comprehensive review of his works and engagements with other theologians. However, given that Pannenberg is known to have significant influences in systematic theology and his work with natural theology, he likely drew upon prominent figures in the theological traditions. Paul Tillich and Karl Barth are often considered key figures in the theological landscape of the 20th century, known for their contributions to systematic theology and their respective theological approaches. Pannenberg is known to have had points of engagement and cross-pollination with both of these theologians.\n\nTherefore, Pannenberg may have primarily drawn upon Paul Tillich and Karl Barth to support his arguments, especially in the context of a model of cosmic history and historical theology where the artificial distinctions between various types of historical accounts are questioned. \n\nThe answer is likely:\n\nB. Paul Tillich and Karl Barth\n\nOther figures listed such as Dietrich Bonhoeffer and Jurgen Moltmann, while significant to their contemporaries, are less likely to be Pannenberg’s primary influences given the established trajectory of his work and the nature of Pannenberg\'s interaction with other theologians regarding the nature of history and its place in theological discourse.\n</research>\n\n<tool_call>\n{"objective": "To confirm the key influences of Wolfhart Pannenberg regarding his model of cosmic history and reinterpretation of history", "name": "GoogleSearch", "arguments": {"queries": ["Wolfhart Pannenberg influences systematic theology", "Paul Tillich and Karl Barth influences on Wolfhart Pannenberg", "Pannenberg\'s engagement with Paul Tillich and Karl Barth"]}}\n</tool_call>',

        # '<research>Wolfhart Pannenbeiscourse.\n</research>\n\n<tool_call>\n{"objective": "To confirm the key influences of Wolfhart Pannenberg regarding his model of cosmic history and reinterpretation of history", "name": "python_interpreter", "arguments": {"code": ""}}\n<code>import json</code></tool_call>'

        # '<research>Wolfhart Pannenbeiscourse.\n</research>\n\n<tool_call>\n{"objective": "To confirm the key influences of Wolfhart Pannenberg regarding his model of cosmic history and reinterpretation of history", "name": "python_interpreter", "arguments": {"code": ""}}\n<code>\n\nimport json\n\n</code>\n</tool_call>'

        # '<research>Wolfhart Pannenberg\'s key influences in his engagement with physics and his reinterpretation of history can be difficult to deduce without a comprehensive review of his works and engagements with other theologians. However, given that Pannenberg is known to have significant influences in systematic theology and his work with natural theology, he likely drew upon prominent figures in the theological traditions. Paul Tillich and Karl Barth are often considered key figures in the theological landscape of the 20th century, known for their contributions to systematic theology and their respective theological approaches. Pannenberg is known to have had points of engagement and cross-pollination with both of these theologians.\n\nTherefore, Pannenberg may have primarily drawn upon Paul Tillich and Karl Barth to support his arguments, especially in the context of a model of cosmic history and historical theology where the artificial distinctions between various types of historical accounts are questioned. \n\nThe answer is likely:\n\nB. Paul Tillich and Karl Barth\n\nOther figures listed such as Dietrich Bonhoeffer and Jurgen Moltmann, while significant to their contemporaries, are less likely to be Pannenberg’s primary influences given the established trajectory of his work and the nature of Pannenberg\'s interaction with other theologians regarding the nature of history and its place in theological discourse.\n</research>\n\n<tool_call>\n{"objective": "To confirm the key influences of Wolfhart Pannenberg regarding his model of cosmic history and reinterpretation of history", "arguments": {"queries": ["Wolfhart Pannenberg influences systematic theology", "Paul Tillich and Karl Barth influences on Wolfhart Pannenberg", "Pannenberg\'s engagement with Paul Tillich and Karl Barth"]}}\n</tool_call>',


        # '<research>Wolfhart Pannenberg\'s key influences in his engagement with physics and his reinterpretation of history can be difficult to deduce without a comprehensive review of his works and engagements with other theologians. However, given that Pannenberg is known to have significant influences in systematic theology and his work with natural theology, he likely drew upon prominent figures in the theological traditions. Paul Tillich and Karl Barth are often considered key figures in the theological landscape of the 20th century, known for their contributions to systematic theology and their respective theological approaches. Pannenberg is known to have had points of engagement and cross-pollination with both of these theologians.\n\nTherefore, Pannenberg may have primarily drawn upon Paul Tillich and Karl Barth to support his arguments, especially in the context of a model of cosmic history and historical theology where the artificial distinctions between various types of historical accounts are questioned. \n\nThe answer is likely:\n\nB. Paul Tillich and Karl Barth\n\nOther figures listed such as Dietrich Bonhoeffer and Jurgen Moltmann, while significant to their contemporaries, are less likely to be Pannenberg’s primary influences given the established trajectory of his work and the nature of Pannenberg\'s interaction with other theologians regarding the nature of history and its place in theological discourse.\n</research>\n\n<tool_call>\n{ "objective": "To confirm the key influences of Wolfhart Pannenberg regarding his model of cosmic history and reinterpretation of history", "arguments": {"queries": ["Wolfhart Pannenberg influences systematic theology", "Paul Tillich and Karl Barth influences on Wolfhart Pannenberg", "Pannenberg\'s engagement with Paul Tillich and Karl Barth"]}}\n</tool_call>',
    ]
    for text in texts:
        json_params = System2.parse_output(text)
        print(json_params[0]) # focus on tool part