# Copyright 2024 Bytedance Ltd. and/or its affiliates
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
Note that we don't combine the main with ray_trainer as ray_trainer is used by other main.
"""
import argparse
import os
import copy
import numpy as np
import importlib
import torch

os.environ['UNIFORM_EVAL_NO_BAN_MODULES'] = '1'
from uniform_eval.tasks.base import Task
from verl import DataProto
from verl.trainer.ppo.ray_trainer import RayPPOTrainer
from ray.util import pdb

class RewardManager:
    """The reward manager.
    """

    def __init__(self, tokenizer, num_examine, compute_score) -> None:
        self.tokenizer = tokenizer
        self.num_examine = num_examine  # the number of batches of decoded responses to print to the console
        self.compute_score = compute_score
        Task.setup_args(argparse.Namespace(**{"preprocessing":os.environ.get("UNIFORM_EVAL_PREPROCESS", "default")}))

        # import all reward score files
        current_dir = os.path.dirname(os.path.abspath(__file__))
        reward_dir = os.path.join(current_dir, '..', 'utils', 'reward_score')
        for file in os.listdir(reward_dir):
            if file.endswith(".py") and not file.startswith("_"):
                file_name = file[:file.find(".py")]
                importlib.import_module('verl.utils.reward_score.' + file_name)

    def __call__(self, data: DataProto):
        """We will expand this function gradually based on the available datasets"""

        # If there is rm score, we directly return rm score. Otherwise, we compute via rm_score_fn
        if 'rm_scores' in data.batch.keys():
            return data.batch['rm_scores']

        reward_tensor = torch.zeros_like(data.batch['responses'], dtype=torch.float32)

        # already_print_data_sources = {}

        decoded_responses = self.tokenizer.batch_decode(data.batch['response_for_reward'], skip_special_tokens=True)
        data.non_tensor_batch['decoded_responses'] = np.array(decoded_responses, dtype=object)
        # import pdb; pdb.set_trace() # data.non_tensor_batch['uid']
        # Evaluating samples of the same task together to accelerate
        index_scores_map = {}
        grouped_jobs = {} 
        for i, data_item in enumerate(data):
            job = copy.deepcopy(data_item.non_tensor_batch.copy()['job'])

            job['data_source'] = data_item.non_tensor_batch['data_source']
            job['task'] = data_item.non_tensor_batch['task']
            job['gen'] = [data_item.non_tensor_batch['decoded_responses']]
            job['inbatch_id'] = i
            job['stop_reason'] = data_item.non_tensor_batch['stop_reason']
            job['output_format'] = data_item.non_tensor_batch['output_format']

            task_name = job.get('task', 'dummy').split('/')[0]
            if task_name not in grouped_jobs:
                grouped_jobs[task_name] = []
            grouped_jobs[task_name].append(job)


        log_idx = 0
        for task_name, jobs in grouped_jobs.items():
            task = Task.get_task(task_name)
            task.compute_scores(jobs, tokenizer=None)
            for job in jobs:
                index_scores_map[job['inbatch_id']] = (job, task.get_reward(job))

        # update job for better logging (showing extracted_answer, etc.)
        data.non_tensor_batch['job'] = np.array([index_scores_map[i][0] for i in range(len(data))], dtype=object)

        task_list = []
        for i, data_item in enumerate(data):
            prompt_ids = data_item.batch['prompts']

            prompt_length = prompt_ids.shape[-1]
            valid_response_length = data_item.batch['attention_mask'][prompt_length:].sum()

            reward_tensor[i, valid_response_length - 1] = index_scores_map[i][1]
            
            task = data_item.non_tensor_batch['task']
            task_list.append(task)

        return reward_tensor, task_list

    def log_messages(self, job):
        task = job['task']
        inbatch_id = job['inbatch_id']
        original_gen = job.get('original_gen', [None])[-1]
        gen = job['gen'][-1]
        answer = job.get('answer', None)
        keys = job.keys()
        print(f'Task: {task}, Inbatch ID: {inbatch_id}, Job Keys: {keys}')
        if 'messages' in job:
            for message in job['messages']:
                role = message['role']
                content = message['content']
                print(f'[{role}]\n{content!r}\n')
        elif 'prompt' in job:
            if 'eval_args' in job and 'system_str' in job['eval_args']:
                system_message = job["eval_args"]["system_str"]
                print(f'[system]\n{system_message!r}\n')
            if type(job['prompt']) is str:
                user_message = job["prompt"]
                print(f'[user]\n{user_message!r}\n')
            else:
                for user_message, assistant_message in zip(job['prompt'], job.get('response', [])):
                    print(f'[user]\n{user_message!r}\n')
                    print(f'[assistant]\n{assistant_message!r}\n')
                remain_idx = len(job.get('response', []))
                for user_message in job['prompt'][remain_idx:]:
                    print(f'[user]\n{user_message!r}\n')
        else:
            print(f'[warning] No message found in job\n')
        if original_gen:
            print(f'[original gen]\n{original_gen}\n')
        print(f'[gen]\n{gen}\n')
        if answer:
            print(f'[answer]\n{answer}\n')
        if 'reference' in job:
            reference = job['reference']
            print(f'[reference]\n{reference}\n')
        if 'json_schema' in job:
            json_schema = job['json_schema']
            print(f'[json_schema]\n{json_schema}\n')
        if 'tool_schemas' in job:
            tool_schemas = job['tool_schemas']
            if tool_schemas is not None:
                for idx, tool_schema in enumerate(tool_schemas):
                    print(f'[tool_schema {idx}]\n{tool_schema}\n')
        if 'tool_calls' in job:
            tool_calls = job['tool_calls']
            if tool_calls is not None:
                for idx, tool_call in enumerate(tool_calls):
                    print(f'[tool_call {idx}]\n{tool_call}\n')
        if 'judge_first' in job:
            judge_first = job['judge_first']
            print(f'[judge_first]\n{judge_first}\n')
        if 'judgment_response' in job:
            judgment_response = job['judgment_response']
            print(f'[judgment_response]\n{judgment_response}\n')

import ray
import hydra


@hydra.main(config_path='config', config_name='ppo_trainer', version_base=None)
def main(config):
    run_ppo(config)


def run_ppo(config, compute_score=None):
    if not ray.is_initialized():
        # this is for local ray cluster
        ray.init(runtime_env={'env_vars': {'TOKENIZERS_PARALLELISM': 'true', 'NCCL_DEBUG': 'WARN'}})

    ray.get(main_task.remote(config, compute_score))


@ray.remote(num_cpus=1)  # please make sure main_task is not scheduled on head
def main_task(config, compute_score=None):
    from verl.utils.fs import copy_local_path_from_hdfs

    # print initial config
    from pprint import pprint
    from omegaconf import OmegaConf
    pprint(OmegaConf.to_container(config, resolve=True))  # resolve=True will eval symbol values
    OmegaConf.resolve(config)

    # download the checkpoint from hdfs
    local_tokenizer_path = copy_local_path_from_hdfs(config.actor_rollout_ref.model.tokenizer)

    # instantiate tokenizer
    from verl.utils import hf_tokenizer
    tokenizer = hf_tokenizer(local_tokenizer_path)

    # define worker classes
    if config.actor_rollout_ref.actor.strategy == 'fsdp':
        assert config.actor_rollout_ref.actor.strategy == config.critic.strategy
        from verl.workers.fsdp_workers import ActorRolloutRefWorker, CriticWorker
        from verl.single_controller.ray import RayWorkerGroup
        ray_worker_group_cls = RayWorkerGroup

    elif config.actor_rollout_ref.actor.strategy == 'megatron':
        assert config.actor_rollout_ref.actor.strategy == config.critic.strategy
        from verl.workers.megatron_workers import ActorRolloutRefWorker, CriticWorker
        from verl.single_controller.ray.megatron import NVMegatronRayWorkerGroup
        ray_worker_group_cls = NVMegatronRayWorkerGroup

    else:
        raise NotImplementedError

    from verl.trainer.ppo.ray_trainer import ResourcePoolManager, Role

    role_worker_mapping = {
        Role.ActorRollout: ray.remote(ActorRolloutRefWorker),
        Role.Critic: ray.remote(CriticWorker),
        Role.RefPolicy: ray.remote(ActorRolloutRefWorker)
    }

    global_pool_id = 'global_pool'
    resource_pool_spec = {
        global_pool_id: [config.trainer.n_gpus_per_node] * config.trainer.nnodes,
    }
    mapping = {
        Role.ActorRollout: global_pool_id,
        Role.Critic: global_pool_id,
        Role.RefPolicy: global_pool_id,
    }

    # we should adopt a multi-source reward function here
    # - for rule-based rm, we directly call a reward score
    # - for model-based rm, we call a model
    # - for code related prompt, we send to a sandbox if there are test cases
    # - finally, we combine all the rewards together
    # - The reward type depends on the tag of the data
    if config.reward_model.enable:
        if config.reward_model.strategy == 'fsdp':
            from verl.workers.fsdp_workers import RewardModelWorker
        elif config.reward_model.strategy == 'megatron':
            from verl.workers.megatron_workers import RewardModelWorker
        else:
            raise NotImplementedError
        role_worker_mapping[Role.RewardModel] = ray.remote(RewardModelWorker)
        mapping[Role.RewardModel] = global_pool_id

    reward_manager_name = config.reward_model.get("reward_manager", "unieval")
    if reward_manager_name == 'unieval':
        reward_manager_cls = RewardManager
    elif reward_manager_name == 'naive':
        from verl.workers.reward_manager import NaiveRewardManager
        reward_manager_cls = NaiveRewardManager
    elif reward_manager_name == 'prime':
        from verl.workers.reward_manager import PrimeRewardManager
        reward_manager_cls = PrimeRewardManager
    else:
        raise NotImplementedError
    reward_fn = reward_manager_cls(tokenizer=tokenizer, num_examine=0, compute_score=compute_score)

    # Note that we always use function-based RM for validation
    val_reward_fn = reward_manager_cls(tokenizer=tokenizer, num_examine=1, compute_score=compute_score)

    resource_pool_manager = ResourcePoolManager(resource_pool_spec=resource_pool_spec, mapping=mapping)

    trainer = RayPPOTrainer(config=config,
                            tokenizer=tokenizer,
                            role_worker_mapping=role_worker_mapping,
                            resource_pool_manager=resource_pool_manager,
                            ray_worker_group_cls=ray_worker_group_cls,
                            reward_fn=reward_fn,
                            val_reward_fn=val_reward_fn)
    trainer.init_workers()
    trainer.fit()


if __name__ == '__main__':
    main()

