# Copyright 2024 Bytedance Ltd. and/or its affiliates
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
FSDP PPO Trainer with Ray-based single controller.
This trainer supports model-agonistic model initialization with huggingface
"""

import os
import uuid
import json
import re
import string
from contextlib import contextmanager
from dataclasses import dataclass, field
from enum import Enum
from pprint import pprint
from typing import Type, Dict, Tuple, List
from copy import deepcopy
from collections import Counter, defaultdict

import numpy as np
from codetiming import Timer
from omegaconf import OmegaConf, open_dict
from verl import DataProto
from verl.protocol import pad_dataproto_to_divisor, unpad_dataproto
from verl.single_controller.base import Worker
from verl.single_controller.ray import RayResourcePool, RayWorkerGroup, RayClassWithInitArgs
from verl.single_controller.ray.base import create_colocated_worker_cls
from verl.trainer.ppo import core_algos
from verl.utils.seqlen_balancing import get_seqlen_balanced_partitions, log_seqlen_unbalance
from verl.utils.checkpoint.checkpoint_manager import find_latest_ckpt_path
from verl.utils.dataset.rl_dataset import RLHFDataset, collate_fn

from verl.utils import distribution_metrics
from verl.utils.distribution_metrics import compute_repetition_related_metrics_batch, compute_diversity_related_metrics_batch, compute_group_diversity_score, check_fuzz_version
from ray.util import pdb

WorkerType = Type[Worker]

DEFAULT_WORKSPACE: str = os.getenv('QWEN_AGENT_DEFAULT_WORKSPACE', 'workspace')
VERBOSE = os.getenv("VERBOSE", "0").lower() in ("true", "1", "yes", "on")
# PYTHON_CACHE_UUID = os.getenv("PYTHON_CACHE_UUID")

def replace_punctuation_with_space(text):
    # # 定义匹配标点符号的正则表达式
    # punctuation_pattern = r'[^\w\s]'
    # # 使用re.sub()函数将标点符号替换为空格
    # result = re.sub(punctuation_pattern, ' ', text)
    # return result
    return text.replace(","," ").replace("."," ").replace("!"," ").replace("?"," ").replace("，"," ").replace("。"," ").replace("！"," ").replace("？"," ")
    
class Role(Enum):
    """
    To create more roles dynamically, you can subclass Role and add new members
    """
    Actor = 0
    Rollout = 1
    ActorRollout = 2
    Critic = 3
    RefPolicy = 4
    RewardModel = 5
    ActorRolloutRef = 6


@dataclass
class ResourcePoolManager:
    """
    Define a resource pool specification. Resource pool will be initialized first.
    Mapping
    """
    resource_pool_spec: dict[str, list[int]]
    mapping: dict[Role, str]
    resource_pool_dict: dict[str, RayResourcePool] = field(default_factory=dict)

    def create_resource_pool(self):
        for resource_pool_name, process_on_nodes in self.resource_pool_spec.items():
            # max_colocate_count means the number of WorkerGroups (i.e. processes) in each RayResourcePool
            # For FSDP backend, we recommend using max_colocate_count=1 that merge all WorkerGroups into one.
            # For Megatron backend, we recommend using max_colocate_count>1 that can utilize different WorkerGroup for differnt models
            resource_pool = RayResourcePool(process_on_nodes=process_on_nodes,
                                            use_gpu=True,
                                            max_colocate_count=1,
                                            name_prefix=resource_pool_name)
            self.resource_pool_dict[resource_pool_name] = resource_pool

    def get_resource_pool(self, role: Role) -> RayResourcePool:
        """Get the resource pool of the worker_cls"""
        return self.resource_pool_dict[self.mapping[role]]


import torch
from verl.utils.torch_functional import masked_mean


def apply_kl_penalty(data: DataProto, kl_ctrl: core_algos.AdaptiveKLController, kl_penalty='kl'):
    responses = data.batch['responses']
    response_length = responses.size(1)
    token_level_scores = data.batch['token_level_scores']
    batch_size = data.batch.batch_size[0]
    attention_mask = data.batch['attention_mask']
    response_mask = data.batch['response_mask']

    # compute kl between ref_policy and current policy
    if 'ref_log_prob' in data.batch.keys():
        kld = core_algos.kl_penalty(data.batch['old_log_probs'], data.batch['ref_log_prob'],
                                    kl_penalty=kl_penalty)  # (batch_size, response_length)
        kld = kld * response_mask
        beta = kl_ctrl.value
    else:
        beta = 0
        kld = torch.zeros_like(response_mask, dtype=torch.float32)

    token_level_rewards = token_level_scores - beta * kld

    current_kl = masked_mean(kld, mask=response_mask, axis=-1)  # average over sequence
    current_kl = torch.mean(current_kl, dim=0).item()

    # according to https://github.com/huggingface/trl/blob/951ca1841f29114b969b57b26c7d3e80a39f75a0/trl/trainer/ppo_trainer.py#L837
    kl_ctrl.update(current_kl=current_kl, n_steps=batch_size)
    data.batch['token_level_rewards'] = token_level_rewards

    metrics = {'critic/kl': current_kl, 'critic/kl_coeff': beta}

    return data, metrics


def compute_advantage(data: DataProto, adv_estimator, gamma=1.0, lam=1.0, num_repeat=1):
    # prepare response group
    # TODO: add other ways to estimate advantages
    if adv_estimator == 'gae':
        values = data.batch['values']
        responses = data.batch['responses']
        response_length = responses.size(-1)
        attention_mask = data.batch['attention_mask']
        response_mask = data.batch['response_mask']
        token_level_rewards = data.batch['token_level_rewards']
        advantages, returns = core_algos.compute_gae_advantage_return(token_level_rewards=token_level_rewards,
                                                                      values=values,
                                                                      eos_mask=response_mask,
                                                                      gamma=gamma,
                                                                      lam=lam)
        data.batch['advantages'] = advantages
        data.batch['returns'] = returns
    elif adv_estimator == 'grpo':
        token_level_rewards = data.batch['token_level_rewards']
        index = data.non_tensor_batch['uid']
        responses = data.batch['responses']
        response_length = responses.size(-1)
        attention_mask = data.batch['attention_mask']
        response_mask = data.batch['response_mask']
        advantages, returns = core_algos.compute_grpo_outcome_advantage(token_level_rewards=token_level_rewards,
                                                                        eos_mask=response_mask,
                                                                        index=index)
        data.batch['advantages'] = advantages
        data.batch['returns'] = returns
    elif adv_estimator == 'reinforce_plus_plus':
        token_level_rewards = data.batch['token_level_rewards']
        responses = data.batch['responses']
        response_length = responses.size(-1)
        attention_mask = data.batch['attention_mask']
        response_mask = data.batch['response_mask']
        advantages, returns = core_algos.compute_reinforce_plus_plus_outcome_advantage(
            token_level_rewards=token_level_rewards, eos_mask=response_mask, gamma=gamma)
        data.batch['advantages'] = advantages
        data.batch['returns'] = returns
    elif adv_estimator == 'remax':
        token_level_rewards = data.batch['token_level_rewards']
        index = data.non_tensor_batch['uid']
        responses = data.batch['responses']
        response_length = responses.size(-1)
        attention_mask = data.batch['attention_mask']
        response_mask = data.batch['response_mask']

        reward_baselines = data.batch['reward_baselines']

        advantages, returns = core_algos.compute_remax_outcome_advantage(token_level_rewards=token_level_rewards,
                                                                         reward_baselines=reward_baselines,
                                                                         eos_mask=response_mask)

        data.batch['advantages'] = advantages
        data.batch['returns'] = returns
    else:
        raise NotImplementedError
    return data


def reduce_metrics(metrics: dict):
    for key, val in metrics.items():
        metrics[key] = np.mean(val)
    return metrics


def _compute_response_info(batch):
    response_length = batch.batch['responses'].shape[-1]

    prompt_mask = batch.batch['attention_mask'][:, :-response_length]
    response_mask = batch.batch['attention_mask'][:, -response_length:]

    prompt_length = prompt_mask.sum(-1).float()
    response_length = response_mask.sum(-1).float()  # (batch_size,)

    return dict(
        response_mask=response_mask,
        prompt_length=prompt_length,
        response_length=response_length,
    )


def compute_data_metrics(batch, use_critic=True):
    sequence_score = batch.batch['token_level_scores'].sum(-1)
    sequence_reward = batch.batch['token_level_rewards'].sum(-1)

    advantages = batch.batch['advantages']
    returns = batch.batch['returns']

    max_response_length = batch.batch['responses'].shape[-1]

    prompt_mask = batch.batch['attention_mask'][:, :-max_response_length].bool()
    response_mask = batch.batch['response_mask'].bool()

    max_prompt_length = prompt_mask.size(-1)

    response_info = _compute_response_info(batch)
    prompt_length = response_info['prompt_length']
    response_length = response_info['response_length']

    valid_adv = torch.masked_select(advantages, response_mask)
    valid_returns = torch.masked_select(returns, response_mask)

    decoded_responses = batch.non_tensor_batch['decoded_responses']

    # the distribution of sample outcomes
    sequence_score_list = sequence_score.tolist()
    index = batch.non_tensor_batch['uid'].tolist()

    grouped_sequence_scores = {}
    grouped_decoded_responses = {}
    for i in range(len(sequence_score)):
        if index[i] not in grouped_sequence_scores:
            grouped_sequence_scores[index[i]] = []
            grouped_decoded_responses[index[i]] = []
        grouped_sequence_scores[index[i]].append(sequence_score_list[i])
        grouped_decoded_responses[index[i]].append(decoded_responses[i])
    
    grouped_sequence_scores_mean = [np.mean(y) for x,y in grouped_sequence_scores.items()]
    
     # the word counts of decoded responses
    decoded_responses_words = []
    for response in decoded_responses:
        decoded_responses_words += replace_punctuation_with_space(response.lower()).split()
    keywords_counter = Counter(decoded_responses_words)

    # sample diversity
    # check_fuzz_version()
    # assert distribution_metrics._version_checked == True
    # group_diversity_mean, group_diversity_min = [], []
    # for _, texts in grouped_decoded_responses.items():
    #     d_mean, d_min = compute_group_diversity_score(texts)
    #     group_diversity_mean.append(d_mean)
    #     group_diversity_min.append(d_min)
    # diversity_mean = np.mean(group_diversity_mean)
    # diversity_min = min(group_diversity_min)

    # format error ratio
    # format_wrong_cnts = [0 if "\\boxed{" in response else 1 for response in decoded_responses]
    
    if use_critic:
        values = batch.batch['values']
        valid_values = torch.masked_select(values, response_mask)
        return_diff_var = torch.var(valid_returns - valid_values)
        return_var = torch.var(valid_returns)

    metrics = {
        # score
        'critic/score/mean':
            torch.mean(sequence_score).detach().item(),
        'critic/score/max':
            torch.max(sequence_score).detach().item(),
        'critic/score/min':
            torch.min(sequence_score).detach().item(),
        # reward
        'critic/rewards/mean':
            torch.mean(sequence_reward).detach().item(),
        'critic/rewards/max':
            torch.max(sequence_reward).detach().item(),
        'critic/rewards/min':
            torch.min(sequence_reward).detach().item(),
        # adv
        'critic/advantages/mean':
            torch.mean(valid_adv).detach().item(),
        'critic/advantages/max':
            torch.max(valid_adv).detach().item(),
        'critic/advantages/min':
            torch.min(valid_adv).detach().item(),
        # returns
        'critic/returns/mean':
            torch.mean(valid_returns).detach().item(),
        'critic/returns/max':
            torch.max(valid_returns).detach().item(),
        'critic/returns/min':
            torch.min(valid_returns).detach().item(),
        **({
            # values
            'critic/values/mean': torch.mean(valid_values).detach().item(),
            'critic/values/max': torch.max(valid_values).detach().item(),
            'critic/values/min': torch.min(valid_values).detach().item(),
            # vf explained var
            'critic/vf_explained_var': (1.0 - return_diff_var / (return_var + 1e-5)).detach().item(),
        } if use_critic else {}),

        # response length
        'response_length/mean':
            torch.mean(response_length).detach().item(),
        'response_length/max':
            torch.max(response_length).detach().item(),
        'response_length/min':
            torch.min(response_length).detach().item(),
        'response_length/clip_ratio':
            torch.mean(torch.eq(response_length, max_response_length).float()).detach().item(),
        
        # prompt length
        'prompt_length/mean':
            torch.mean(prompt_length).detach().item(),
        'prompt_length/max':
            torch.max(prompt_length).detach().item(),
        'prompt_length/min':
            torch.min(prompt_length).detach().item(),
        'prompt_length/clip_ratio':
            torch.mean(torch.eq(prompt_length, max_prompt_length).float()).detach().item(),
        
        # grouped sampling acc
        'grouped_sampling_acc/max':
            max(grouped_sequence_scores_mean),
        'grouped_sampling_acc/min':
            min(grouped_sequence_scores_mean),
        'grouped_sampling_acc/75th_percentile':
            np.percentile(grouped_sequence_scores_mean,75),
        'grouped_sampling_acc/50th_percentile':
            np.percentile(grouped_sequence_scores_mean,50),
        'grouped_sampling_acc/25th_percentile':
            np.percentile(grouped_sequence_scores_mean,25),

        # emergent keywords
        'emergent_keywords/wait_count_per_sample': 
            keywords_counter['wait']/len(batch),
        'emergent_keywords/alternatively_count_per_sample':
            keywords_counter['alternatively']/len(batch),
        'emergent_keywords/but_count_per_sample': 
            keywords_counter['but']/len(batch),
        'emergent_keywords/yes_count_per_sample':
            keywords_counter['yes']/len(batch),
        'emergent_keywords/no_count_per_sample':
            keywords_counter['no']/len(batch),

        # # diversity
        # "group_diversity/mean": diversity_mean, 
        # "group_diversity/min": diversity_min,
    }

    # other diversity related
    # metrics.update(compute_diversity_related_metrics_batch(batch))

    # repetition       
    # metrics.update(compute_repetition_related_metrics_batch(batch))
    return metrics


def compute_timing_metrics(batch, timing_raw):
    response_info = _compute_response_info(batch)
    num_prompt_tokens = torch.sum(response_info['prompt_length']).item()
    num_response_tokens = torch.sum(response_info['response_length']).item()
    num_overall_tokens = num_prompt_tokens + num_response_tokens

    num_tokens_of_section = {
        'gen': num_response_tokens,
        **{
            name: num_overall_tokens for name in ['ref', 'values', 'adv', 'update_critic', 'update_actor']
        },
    }

    return {
        **{
            f'timing_s/{name}': value for name, value in timing_raw.items()
        },
        **{
            f'timing_per_token_ms/{name}': timing_raw[name] * 1000 / num_tokens_of_section[name] for name in set(num_tokens_of_section.keys(
            )) & set(timing_raw.keys())
        },
    }


@contextmanager
def _timer(name: str, timing_raw: Dict[str, float]):
    import sys
    print(f"timer {name} start", flush=True, file=sys.stderr)
    with Timer(name=name, logger=None) as timer:
        yield
    timing_raw[name] = timer.last
    print(f"timer {name} end", flush=True, file=sys.stderr)


class RayPPOTrainer(object):
    """
    Note that this trainer runs on the driver process on a single CPU/GPU node.
    """

    # TODO: support each role have individual ray_worker_group_cls,
    # i.e., support different backend of different role
    def __init__(self,
                 config,
                 tokenizer,
                 role_worker_mapping: dict[Role, WorkerType],
                 resource_pool_manager: ResourcePoolManager,
                 ray_worker_group_cls: RayWorkerGroup = RayWorkerGroup,
                 reward_fn=None,
                 val_reward_fn=None):

        # assert torch.cuda.is_available(), 'cuda must be available on driver'

        self.tokenizer = tokenizer
        self.config = config
        self.reward_fn = reward_fn
        self.val_reward_fn = val_reward_fn

        self.hybrid_engine = config.actor_rollout_ref.hybrid_engine
        assert self.hybrid_engine, 'Currently, only support hybrid engine'

        if self.hybrid_engine:
            assert Role.ActorRollout in role_worker_mapping, f'{role_worker_mapping.keys()=}'

        self.role_worker_mapping = role_worker_mapping
        self.resource_pool_manager = resource_pool_manager
        self.use_reference_policy = Role.RefPolicy in role_worker_mapping
        self.use_rm = Role.RewardModel in role_worker_mapping
        self.ray_worker_group_cls = ray_worker_group_cls

        # define KL control
        if self.use_reference_policy:
            if config.algorithm.kl_ctrl.type == 'fixed':
                self.kl_ctrl = core_algos.FixedKLController(kl_coef=config.algorithm.kl_ctrl.kl_coef)
            elif config.algorithm.kl_ctrl.type == 'adaptive':
                assert config.algorithm.kl_ctrl.horizon > 0, f'horizon must be larger than 0. Got {config.critic.kl_ctrl.horizon}'
                self.kl_ctrl = core_algos.AdaptiveKLController(init_kl_coef=config.algorithm.kl_ctrl.kl_coef,
                                                               target_kl=config.algorithm.kl_ctrl.target_kl,
                                                               horizon=config.algorithm.kl_ctrl.horizon)
            else:
                raise NotImplementedError
        else:
            self.kl_ctrl = core_algos.FixedKLController(kl_coef=0.)

        if self.config.algorithm.adv_estimator == 'gae':
            self.use_critic = True
        elif self.config.algorithm.adv_estimator == 'grpo':
            self.use_critic = False
        elif self.config.algorithm.adv_estimator == 'reinforce_plus_plus':
            self.use_critic = False
        elif self.config.algorithm.adv_estimator == 'remax':
            self.use_critic = False
        else:
            raise NotImplementedError
        
        # addition params for multi-agent rl
        self.multi_agent_train = self.config.actor_rollout_ref.rollout.multiagent and self.config.actor_rollout_ref.rollout.multiturn_agent and self.config.actor_rollout_ref.rollout.multi_agent_pattern.system1_sampling_params.system1_mode == 'training'

        self._validate_config()
        self._create_dataloader()

    def _validate_config(self):
        config = self.config
        # number of GPUs total
        n_gpus = config.trainer.n_gpus_per_node * config.trainer.nnodes

        # 1. Check total batch size for data correctness
        real_train_batch_size = config.data.train_batch_size * config.actor_rollout_ref.rollout.n
        assert real_train_batch_size % n_gpus == 0, \
            f"real_train_batch_size ({real_train_batch_size}) must be divisible by total n_gpus ({n_gpus})."

        # A helper function to check "micro_batch_size" vs "micro_batch_size_per_gpu"
        # We throw an error if the user sets both. The new convention is "..._micro_batch_size_per_gpu".
        def check_mutually_exclusive(mbs, mbs_per_gpu, name: str):
            if mbs is None and mbs_per_gpu is None:
                raise ValueError(f"[{name}] Please set at least one of '{name}.micro_batch_size' or "
                                 f"'{name}.micro_batch_size_per_gpu'.")

            if mbs is not None and mbs_per_gpu is not None:
                raise ValueError(f"[{name}] You have set both '{name}.micro_batch_size' AND "
                                 f"'{name}.micro_batch_size_per_gpu'. Please remove '{name}.micro_batch_size' "
                                 f"because only '*_micro_batch_size_per_gpu' is supported (the former is deprecated).")

        if not config.actor_rollout_ref.actor.use_dynamic_bsz:
            # actor: ppo_micro_batch_size vs. ppo_micro_batch_size_per_gpu
            check_mutually_exclusive(config.actor_rollout_ref.actor.ppo_micro_batch_size,
                                     config.actor_rollout_ref.actor.ppo_micro_batch_size_per_gpu,
                                     "actor_rollout_ref.actor")

            # reference: log_prob_micro_batch_size vs. log_prob_micro_batch_size_per_gpu
            check_mutually_exclusive(config.actor_rollout_ref.ref.log_prob_micro_batch_size,
                                     config.actor_rollout_ref.ref.log_prob_micro_batch_size_per_gpu,
                                     "actor_rollout_ref.ref")

            #  The rollout section also has log_prob_micro_batch_size vs. log_prob_micro_batch_size_per_gpu
            check_mutually_exclusive(config.actor_rollout_ref.rollout.log_prob_micro_batch_size,
                                     config.actor_rollout_ref.rollout.log_prob_micro_batch_size_per_gpu,
                                     "actor_rollout_ref.rollout")

        if self.use_critic and not config.critic.use_dynamic_bsz:
            # Check for critic micro-batch size conflicts
            check_mutually_exclusive(config.critic.ppo_micro_batch_size, config.critic.ppo_micro_batch_size_per_gpu,
                                     "critic")

        # Check for reward model micro-batch size conflicts
        if config.reward_model.enable and not config.reward_model.use_dynamic_bsz:
            check_mutually_exclusive(config.reward_model.micro_batch_size, config.reward_model.micro_batch_size_per_gpu,
                                     "reward_model")

        # Actor
        # if NOT dynamic_bsz, we must ensure:
        #    ppo_mini_batch_size is divisible by ppo_micro_batch_size
        #    ppo_micro_batch_size * sequence_parallel_size >= n_gpus
        if not config.actor_rollout_ref.actor.use_dynamic_bsz:
            sp_size = config.actor_rollout_ref.actor.get('ulysses_sequence_parallel_size', 1)
            if config.actor_rollout_ref.actor.ppo_micro_batch_size is not None:
                assert config.actor_rollout_ref.actor.ppo_mini_batch_size % config.actor_rollout_ref.actor.ppo_micro_batch_size == 0
                assert config.actor_rollout_ref.actor.ppo_micro_batch_size * sp_size >= n_gpus

        # critic
        if self.use_critic and not config.critic.use_dynamic_bsz:
            sp_size = config.critic.get('ulysses_sequence_parallel_size', 1)
            if config.critic.ppo_micro_batch_size is not None:
                assert config.critic.ppo_mini_batch_size % config.critic.ppo_micro_batch_size == 0
                assert config.critic.ppo_micro_batch_size * sp_size >= n_gpus

        # Check if use_remove_padding is enabled when using sequence parallelism for fsdp
        if config.actor_rollout_ref.actor.strategy == 'fsdp':
            if config.actor_rollout_ref.actor.get('ulysses_sequence_parallel_size', 1) > 1 or \
                    config.actor_rollout_ref.ref.get('ulysses_sequence_parallel_size', 1) > 1:
                assert config.actor_rollout_ref.model.use_remove_padding, \
                    "When using sequence parallelism for actor/ref policy, you must enable `use_remove_padding`."

        if self.use_critic and config.critic.strategy == 'fsdp':
            if config.critic.get('ulysses_sequence_parallel_size', 1) > 1:
                assert config.critic.model.use_remove_padding, \
                    "When using sequence parallelism for critic, you must enable `use_remove_padding`."

        print("[validate_config] All configuration checks passed successfully!")

    def _create_dataloader(self):
        from torch.utils.data import DataLoader, RandomSampler, SequentialSampler
        # TODO: we have to make sure the batch size is divisible by the dp size
        from verl.utils.dataset.rl_dataset import RLHFDataset, collate_fn
        self.train_dataset = RLHFDataset(jsonl_files=self.config.data.train_files,
                                         tokenizer=self.tokenizer,
                                         prompt_key=self.config.data.prompt_key,
                                         max_prompt_length=self.config.data.max_prompt_length,
                                         filter_prompts=True,
                                         return_raw_chat=self.config.data.get('return_raw_chat', False),
                                         truncation='error')
        # use sampler for better ckpt resume
        if self.config.trainer.train_shuffle:
            train_dataloader_generator = torch.Generator()
            train_dataloader_generator.manual_seed(self.config.data.get('seed', 1))
            sampler = RandomSampler(data_source=self.train_dataset, generator=train_dataloader_generator)
        else:
            sampler = SequentialSampler(data_source=self.train_dataset)

        self.train_dataloader = DataLoader(dataset=self.train_dataset,
                                           batch_size=self.config.data.train_batch_size,
                                           drop_last=True,
                                           collate_fn=collate_fn,
                                           sampler=sampler)

        self.val_dataset = RLHFDataset(jsonl_files=self.config.data.val_files,
                                       tokenizer=self.tokenizer,
                                       prompt_key=self.config.data.prompt_key,
                                       max_prompt_length=self.config.data.max_prompt_length,
                                       filter_prompts=True,
                                       return_raw_chat=self.config.data.get('return_raw_chat', False),
                                       truncation='error')
        self.val_dataloader = DataLoader(dataset=self.val_dataset,
                                         batch_size=len(self.val_dataset),
                                         shuffle=self.config.trainer.val_shuffle,
                                         drop_last=False,
                                         collate_fn=collate_fn)

        assert len(self.train_dataloader) >= 1
        assert len(self.val_dataloader) >= 1

        print(f'Size of train dataloader: {len(self.train_dataloader)}')
        print(f'Size of val dataloader: {len(self.val_dataloader)}')

        # inject total_training_steps to actor/critic optim_config. This is hacky.
        total_training_steps = len(self.train_dataloader) * self.config.trainer.total_epochs

        if self.config.trainer.total_training_steps is not None:
            total_training_steps = self.config.trainer.total_training_steps

        self.total_training_steps = total_training_steps
        print(f'Total training steps: {self.total_training_steps}')

        OmegaConf.set_struct(self.config, True)
        with open_dict(self.config):
            self.config.actor_rollout_ref.actor.optim.total_training_steps = total_training_steps
            self.config.critic.optim.total_training_steps = total_training_steps

    def _maybe_log_val_generations_to_wandb(self, inputs, outputs, scores):
        """Log a table of validation samples to wandb"""

        generations_to_log = self.config.trainer.val_generations_to_log_to_wandb

        if generations_to_log == 0:
            return

        if generations_to_log > 0 and 'wandb' not in self.config.trainer.logger:
            print(
                'WARNING: `val_generations_to_log_to_wandb` is set to a positive value, but no wandb logger is found. ')
            return

        import wandb
        import numpy as np

        # Create tuples of (input, output, score) and sort by input text
        samples = list(zip(inputs, outputs, scores))
        samples.sort(key=lambda x: x[0])  # Sort by input text

        # Use fixed random seed for deterministic shuffling
        rng = np.random.RandomState(42)
        rng.shuffle(samples)

        # Take first N samples after shuffling
        samples = samples[:generations_to_log]

        # Create column names for all samples
        columns = ["step"] + sum([[f"input_{i+1}", f"output_{i+1}", f"score_{i+1}"] for i in range(len(samples))], [])

        if not hasattr(self, 'validation_table'):
            # Initialize the table on first call
            self.validation_table = wandb.Table(columns=columns)

        # Create a new table with same columns and existing data
        # Workaround for https://github.com/wandb/wandb/issues/2981#issuecomment-1997445737
        new_table = wandb.Table(columns=columns, data=self.validation_table.data)

        # Add new row with all data
        row_data = []
        row_data.append(self.global_steps)
        for sample in samples:
            row_data.extend(sample)

        new_table.add_data(*row_data)

        # Update reference and log
        wandb.log({"val/generations": new_table}, step=self.global_steps)
        self.validation_table = new_table

    def filter_data_by_keywords(self, data: DataProto, filter_keys: List[str], key: str = 'task') -> Tuple[DataProto, dict]:
        """根据关键词列表过滤数据
        
        Args:
            data: 原始数据
            filter_keys: 需要过滤的关键词列表，例如 ['system1', 'dummy']
            key: 用于过滤的字段名，默认为'task'
            
        Returns:
            filtered_data: 过滤后的数据
            stats: 过滤统计信息
        """
        # 创建布尔掩码：如果数据包含任何一个filter_keys中的关键词，就标记为False
        mask = np.ones(len(data), dtype=bool)
        for keyword in filter_keys:
            mask &= ~np.array([keyword in task for task in data.non_tensor_batch[key]])
        
        # 使用布尔掩码筛选数据
        if data.batch is not None:
            filtered_batch = data.batch[torch.tensor(mask)]
        else:
            filtered_batch = None
            
        # 筛选non_tensor_batch
        filtered_non_tensor = {}
        for key, val in data.non_tensor_batch.items():
            filtered_val = val[mask]
            # 如果筛选后不是numpy数组，转换为numpy数组
            if not isinstance(filtered_val, np.ndarray):
                filtered_val = np.array([filtered_val], dtype=object)
            filtered_non_tensor[key] = filtered_val
        
        # 统计信息
        stats = {
            'total_samples': len(data),
            'filtered_samples': len(data) - mask.sum(),
            'remaining_samples': mask.sum(),
            'filtered_keywords': filter_keys
        }
        
        filtered_data = DataProto(
            batch=filtered_batch,
            non_tensor_batch=filtered_non_tensor,
            meta_info=data.meta_info
        )
        
        return filtered_data, stats

    def _validate(self):
        reward_tensor_lst = []
        data_source_lst = []

        # Lists to collect samples for the table
        sample_inputs = []
        sample_outputs = []
        sample_scores = []

        self.val_query_sample_list_pool = []
        for test_data in self.val_dataloader:
            test_batch = DataProto.from_single_dict(test_data)

            # we only do validation on rule-based rm
            if self.config.reward_model.enable and test_batch[0].non_tensor_batch['reward_model']['style'] == 'model':
                return {}

            # we assume that all examples in a batch share the same eval_args
            # TODO(suyang.fy): group test batch by eval_args.
            eval_args = test_batch[0].non_tensor_batch['job'].get('eval_args', {})

            # Store original inputs
            input_ids = test_batch.batch['input_ids']
            input_texts = [self.tokenizer.decode(ids, skip_special_tokens=True) for ids in input_ids]
            sample_inputs.extend(input_texts)

            test_gen_batch = test_batch.pop(
                batch_keys=['input_ids', 'attention_mask', 'position_ids'],
                non_tensor_batch_keys=['rollout_item', 'task', 'job'], # dict_keys(['data_source', 'answer', 'ability', 'task', 'reward_model', 'extra_info', 'messages', 'job', 'rollout_item', 'prompt_with_chat_template', 'index'])
            )
            # test_gen_batch.non_tensor_batch['task']
            test_gen_batch.meta_info = {
                'eos_token_id': self.tokenizer.eos_token_id,
                'pad_token_id': self.tokenizer.pad_token_id,
                'recompute_log_prob': False,
                'do_sample': True,
                'validate': True,
                'eval_args': eval_args,
            }
            # test_output_gen_batch.non_tensor_batch['task']
            # pad to be divisible by dp_size
            test_gen_batch_padded, pad_size = pad_dataproto_to_divisor(test_gen_batch, self.actor_rollout_wg.world_size, change_task=self.multi_agent_train)
            if len(test_gen_batch_padded) == pad_size or len(test_gen_batch_padded) == 0:
                continue
            test_output_gen_batch_padded = self.actor_rollout_wg.generate_sequences(test_gen_batch_padded)

            # for multi-agent rl
            if self.multi_agent_train:
                test_output_gen_batch, status = self.filter_data_by_keywords(test_output_gen_batch_padded, ['system1', 'dummy'], 'task')
            else:
                # unpad
                test_output_gen_batch = unpad_dataproto(test_output_gen_batch_padded, pad_size=pad_size) # DataProtoItem

            assert set(test_gen_batch.non_tensor_batch['task']) == set(test_output_gen_batch.non_tensor_batch['task']), f"validation ERROR: {set(test_gen_batch.non_tensor_batch['task'])}\n!=\n{set(test_output_gen_batch.non_tensor_batch['task'])}"
            print('validation generation end')
            # pdb.set_trace()
            # Store generated outputs
            output_ids = test_output_gen_batch.batch['responses']
            output_texts = [self.tokenizer.decode(ids, skip_special_tokens=True) for ids in output_ids]
            sample_outputs.extend(output_texts)

            if self.config.actor_rollout_ref.rollout.use_rl_agent:
                test_batch = DataProto(
                        batch=test_output_gen_batch.batch,
                        non_tensor_batch=test_output_gen_batch.non_tensor_batch,
                        meta_info=test_output_gen_batch.meta_info,
                    ) # test_batch is verl.protocol.DataProto
            else:
                test_batch = test_batch.union(test_output_gen_batch)

            # evaluate using reward_function
            reward_tensor, _ = self.val_reward_fn(test_batch)

            # Store scores
            scores = reward_tensor.sum(-1).cpu().tolist()
            sample_scores.extend(scores)

            reward_tensor_lst.append(reward_tensor)
            data_source_lst.append(test_batch.non_tensor_batch.get('data_source', ['unknown'] * reward_tensor.shape[0]))
            
            # save logs
            test_batch.batch['token_level_scores'] = reward_tensor
            tmp_val_query_sample_list_pool = self._reformat_sample_list(test_batch)
            self.val_query_sample_list_pool += tmp_val_query_sample_list_pool
            # test_batch.non_tensor_batch['decoded_responses'] = np.array([str(sample['response']) for sample in self.val_query_sample_list_pool], dtype=object)

        if len(sample_scores) == 0:
            return {} # {'val/test_score/empty': 0.0}
        
        self._maybe_log_val_generations_to_wandb(inputs=sample_inputs, outputs=sample_outputs, scores=sample_scores)

        reward_tensor = torch.cat(reward_tensor_lst, dim=0).sum(-1).cpu()  # (batch_size,)
        data_sources = np.concatenate(data_source_lst, axis=0)

        # evaluate test_score based on data source
        data_source_reward = {}
        for i in range(reward_tensor.shape[0]):
            data_source = data_sources[i]
            if data_source not in data_source_reward:
                data_source_reward[data_source] = []
            data_source_reward[data_source].append(reward_tensor[i].item())

        metric_dict = {}
        for data_source, rewards in data_source_reward.items():
            metric_dict[f'val/test_score/{data_source}'] = np.mean(rewards)
        
        self._save_query_samples(split="val")

        return metric_dict

    def init_workers(self):
        """Init resource pool and worker group"""
        self.resource_pool_manager.create_resource_pool()

        self.resource_pool_to_cls = {pool: {} for pool in self.resource_pool_manager.resource_pool_dict.values()}

        # create actor and rollout
        if self.hybrid_engine:
            resource_pool = self.resource_pool_manager.get_resource_pool(Role.ActorRollout)
            actor_rollout_cls = RayClassWithInitArgs(cls=self.role_worker_mapping[Role.ActorRollout],
                                                     config=self.config.actor_rollout_ref,
                                                     role='actor_rollout')
            self.resource_pool_to_cls[resource_pool]['actor_rollout'] = actor_rollout_cls
        else:
            raise NotImplementedError

        # create critic
        if self.use_critic:
            resource_pool = self.resource_pool_manager.get_resource_pool(Role.Critic)
            critic_cls = RayClassWithInitArgs(cls=self.role_worker_mapping[Role.Critic], config=self.config.critic)
            self.resource_pool_to_cls[resource_pool]['critic'] = critic_cls

        # create reference policy if needed
        if self.use_reference_policy:
            resource_pool = self.resource_pool_manager.get_resource_pool(Role.RefPolicy)
            ref_policy_cls = RayClassWithInitArgs(self.role_worker_mapping[Role.RefPolicy],
                                                  config=self.config.actor_rollout_ref,
                                                  role='ref')
            self.resource_pool_to_cls[resource_pool]['ref'] = ref_policy_cls

        # create a reward model if reward_fn is None
        if self.use_rm:
            # we create a RM here
            resource_pool = self.resource_pool_manager.get_resource_pool(Role.RewardModel)
            rm_cls = RayClassWithInitArgs(self.role_worker_mapping[Role.RewardModel], config=self.config.reward_model)
            self.resource_pool_to_cls[resource_pool]['rm'] = rm_cls

        # initialize WorkerGroup
        # NOTE: if you want to use a different resource pool for each role, which can support different parallel size,
        # you should not use `create_colocated_worker_cls`. Instead, directly pass different resource pool to different worker groups.
        # See https://github.com/volcengine/verl/blob/master/examples/ray/tutorial.ipynb for more information.
        all_wg = {}
        self.wg_dicts = []
        for resource_pool, class_dict in self.resource_pool_to_cls.items():
            worker_dict_cls = create_colocated_worker_cls(class_dict=class_dict)
            wg_dict = self.ray_worker_group_cls(resource_pool=resource_pool, ray_cls_with_init=worker_dict_cls)
            spawn_wg = wg_dict.spawn(prefix_set=class_dict.keys())
            all_wg.update(spawn_wg)
            # keep the referece of WorkerDict to support ray >= 2.31. Ref: https://github.com/ray-project/ray/pull/45699
            self.wg_dicts.append(wg_dict)

        if self.use_critic:
            self.critic_wg = all_wg['critic']
            self.critic_wg.init_model()

        if self.use_reference_policy:
            self.ref_policy_wg = all_wg['ref']
            self.ref_policy_wg.init_model()

        if self.use_rm:
            self.rm_wg = all_wg['rm']
            self.rm_wg.init_model()

        # we should create rollout at the end so that vllm can have a better estimation of kv cache memory
        self.actor_rollout_wg = all_wg['actor_rollout']
        self.actor_rollout_wg.init_model()

    def _reformat_sample_list(self, batch:DataProto):
        messages = batch.non_tensor_batch[self.config.data.prompt_key].tolist() # for multi_agent, message is the get_rl_data in context
        responses = batch.non_tensor_batch['decoded_responses']  # for multi_agent, responses is the truncated data in `vllm_rollout_multi_agent.get_response()`
        rewards = batch.batch['token_level_scores'].sum(1)
        jobs = batch.non_tensor_batch.get('job', [None] * len(messages))  # original data + some infos in messages
        data_sources = batch.non_tensor_batch.get('data_source', [None] * len(messages))
        tasks = batch.non_tensor_batch.get('task', [None] * len(messages))

        # in case the data instances do not have the ground truth
        reward_model_configs = batch.non_tensor_batch.get('reward_model',None) # batch.non_tensor_batch['data_source']
        if reward_model_configs is not None:
            answers = [reward_model_config.get("ground_truth","UNK") for reward_model_config in reward_model_configs]
            sample_list = [{'messages': messages[i], "response": responses[i], "ground_truth": answers[i], "reward": rewards[i].item(), "job": jobs[i]} for i in range(len(messages))]
        else:
            sample_list = [{'data_source': data_sources[i], 'task': tasks[i], 'messages': messages[i], "response": responses[i], "reward": rewards[i].item(), "job": jobs[i]} for i in range(len(messages))]

        # for i in range(len(messages)):
        #     sample_list[i]['prompt_with_chat_template'] = batch.non_tensor_batch['prompt_with_chat_template'][i]

        return sample_list

    def _save_query_samples(self, split = "train"):
        local_path = os.path.join(self.config.trainer.default_local_dir, '{}_query_samples'.format(split))
        hdfs_path = None if self.config.trainer.default_hdfs_dir is None else os.path.join(
            self.config.trainer.default_hdfs_dir, '{}_query_samples'.format(split))
        
        print(f'Saving query samples to {local_path}')
        os.makedirs(local_path, exist_ok=True)
        with open(os.path.join(local_path,f'global_step_{self.global_steps}.jsonl'), 'w', encoding='utf-8') as f:
            if split == 'train':
                query_sample_list_pool = self.train_query_sample_list_pool # [:128]
            elif split == "val":
                query_sample_list_pool = self.val_query_sample_list_pool
            else:
                raise NotImplementedError
            for line in query_sample_list_pool:
                f.write(json.dumps(line, ensure_ascii=False) + '\n')

        if hdfs_path is not None:
            print(f'Uploading actor checkpoint to {hdfs_path}')
            hdfs_io.makedirs(hdfs_path, exist_ok=True)
            hdfs_io.copy(src=local_path, dst=hdfs_path)

    def _save_checkpoint(self, epoch, batches_consumed):
        # path: given_path + `/global_step_{global_steps}` + `/actor`
        local_global_step_folder = os.path.join(self.config.trainer.default_local_dir,
                                                f'global_step_{self.global_steps}')
        actor_local_path = os.path.join(local_global_step_folder, 'actor')

        actor_remote_path = None if self.config.trainer.default_hdfs_dir is None else os.path.join(
            self.config.trainer.default_hdfs_dir, f'global_step_{self.global_steps}', 'actor')
        self.actor_rollout_wg.save_checkpoint(actor_local_path,
                                              actor_remote_path,
                                              self.global_steps,
                                              remove_previous_ckpt=self.config.trainer.remove_previous_ckpt_in_save)

        if self.use_critic:
            critic_local_path = os.path.join(local_global_step_folder, 'critic')
            critic_remote_path = None if self.config.trainer.default_hdfs_dir is None else os.path.join(
                self.config.trainer.default_hdfs_dir, f'global_step_{self.global_steps}', 'critic')
            self.critic_wg.save_checkpoint(critic_local_path,
                                           critic_remote_path,
                                           self.global_steps,
                                           remove_previous_ckpt=self.config.trainer.remove_previous_ckpt_in_save)

        # save dataloader
        dataloader_local_path = os.path.join(local_global_step_folder, 'data.pt')
        import dill
        # The `DataLoader` object does not track the number of batches consumed in current epoch,
        # so we need to manually update it and use it as a flag to skip the batches that have been consumed
        # when resuming from a checkpoint
        self.train_dataloader.batches_consumed = batches_consumed
        self.train_dataloader.epochs_consumed = epoch
        torch.save(self.train_dataloader, dataloader_local_path, pickle_module=dill)

        # latest checkpointed iteration tracker (for atomic usage)
        local_latest_checkpointed_iteration = os.path.join(self.config.trainer.default_local_dir,
                                                           'latest_checkpointed_iteration.txt')
        with open(local_latest_checkpointed_iteration, 'w') as f:
            f.write(str(self.global_steps))

    def _load_checkpoint(self):
        if self.config.trainer.resume_mode == 'disable':
            return 0

        # load from hdfs
        if self.config.trainer.default_hdfs_dir is not None:
            NotImplementedError('load from hdfs is not implemented yet')
        else:
            checkpoint_folder = self.config.trainer.default_local_dir  # TODO: check path
            if not os.path.isabs(checkpoint_folder):
                working_dir = os.getcwd()
                checkpoint_folder = os.path.join(working_dir, checkpoint_folder)
            global_step_folder = find_latest_ckpt_path(checkpoint_folder)  # None if no latest

        # find global_step_folder
        if self.config.trainer.resume_mode == 'auto':
            if global_step_folder is None:
                print('Training from scratch')
                return 0
        else:
            if not (self.config.trainer.resume_from_path and global_step_folder is not None):
                assert isinstance(self.config.trainer.resume_mode, str), "resume ckpt must be str type"
                assert 'global_step_' in self.config.trainer.resume_mode, "resume ckpt must specify the global_steps"
                global_step_folder = self.config.trainer.resume_mode
                if not os.path.isabs(global_step_folder):
                    working_dir = os.getcwd()
                    global_step_folder = os.path.join(working_dir, global_step_folder)
        print(f'Load from checkpoint folder: {global_step_folder}')
        # set global step
        self.global_steps = int(global_step_folder.split('global_step_')[-1])

        print(f'Setting global step to {self.global_steps}')
        print(f'Resuming from {global_step_folder}')

        actor_path = os.path.join(global_step_folder, 'actor')
        critic_path = os.path.join(global_step_folder, 'critic')
        # load actor
        self.actor_rollout_wg.load_checkpoint(actor_path,
                                              del_local_after_load=self.config.trainer.del_local_ckpt_after_load)
        # load critic
        if self.use_critic:
            self.critic_wg.load_checkpoint(critic_path,
                                           del_local_after_load=self.config.trainer.del_local_ckpt_after_load)

        # load dataloader,
        # TODO: from remote not implemented yet
        dataloader_local_path = os.path.join(global_step_folder, 'data.pt')
        self.train_dataloader = torch.load(dataloader_local_path, weights_only=False)
        if isinstance(self.train_dataloader.dataset, RLHFDataset):
            self.train_dataloader.dataset.resume_dataset_state()

    def _balance_batch(self, batch: DataProto, metrics, logging_prefix='global_seqlen'):
        """Reorder the data on single controller such that each dp rank gets similar total tokens"""
        attention_mask = batch.batch['attention_mask']
        batch_size = attention_mask.shape[0]
        global_seqlen_lst = batch.batch['attention_mask'].view(batch_size, -1).sum(-1).tolist()  # (train_batch_size,)
        world_size = self.actor_rollout_wg.world_size
        global_partition_lst = get_seqlen_balanced_partitions(global_seqlen_lst,
                                                              k_partitions=world_size,
                                                              equal_size=True)
        # reorder based on index. The data will be automatically equally partitioned by dispatch function
        global_idx = torch.tensor([j for partition in global_partition_lst for j in partition])
        batch.reorder(global_idx)
        global_balance_stats = log_seqlen_unbalance(seqlen_list=global_seqlen_lst,
                                                    partitions=global_partition_lst,
                                                    prefix=logging_prefix)
        metrics.update(global_balance_stats)

    def balance_multi_agent_samples(self, batch: DataProto) -> Tuple[DataProto, float]:
        balanced_batch, real_beta = self._perform_balance(batch, self.config.actor_rollout_ref.rollout.multi_agent_pattern.beta)

        return self.ensure_dataproto(balanced_batch), real_beta / self.config.actor_rollout_ref.rollout.multi_agent_pattern.beta

    def ensure_dataproto(self, data):
        """确保数据是DataProto类型,并且non_tensor_batch中的值都是numpy数组
        
        Args:
            data: DataProto或DataProtoItem对象
        
        Returns:
            DataProto: 格式化后的DataProto对象
        """
        if isinstance(data, DataProto):
            return data
            
        # 如果是DataProtoItem,先格式化non_tensor_batch
        formatted_non_tensor_batch = {}
        for key, val in data.non_tensor_batch.items():
            if not isinstance(val, np.ndarray):
                val = np.array([val], dtype=object)
            formatted_non_tensor_batch[key] = val
        
        # 转换为DataProto
        return DataProto.from_dict(
            tensors=data.batch,
            non_tensors=formatted_non_tensor_batch,
            meta_info=data.meta_info
        )

    def _perform_balance(self, batch: DataProto, beta: int) -> Tuple[DataProto, float]:
        """按照 system1/system2 比例平衡数据
        
        Args:
            batch: 包含 system1 和 system2 数据的 batch
            beta: 控制总数据量为 beta * n
        
        Returns:
            balanced_batch: 平衡后的数据
        """
        # assert isinstance(batch, DataProto), f'batch must be a DataProto but got {type(batch)}'
        # 1. divide data into two part (system1 & 2)
        uids = batch.non_tensor_batch['uid']
        system1_mask = np.array(['system1' in uid for uid in uids])
        system2_mask = np.array(['system2' in uid for uid in uids])
        assert np.all(system1_mask | system2_mask), 'batch must contain system1 and system2'
        system1_indices = torch.tensor(np.where(system1_mask)[0]) # torch.tensor(np.where(np.array([False, False, False]))[0])
        system1_data = batch[system1_indices] # DataProtoItem
        system2_indices = torch.tensor(np.where(system2_mask)[0])
        system2_data = batch[system2_indices]
        
        real_beta = (system1_data.batch.batch_size[0] + system2_data.batch.batch_size[0]) / system2_data.batch.batch_size[0]
        
        print(f"[rl data ratio]: system1: {system1_data.batch.batch_size[0]}, system2: {system2_data.batch.batch_size[0]}, beta: {real_beta} | {beta}")

        # 2. if system1 is empty, repeating system2
        if system1_data.batch.batch_size[0] == 0:
            balanced_batch = batch.repeat(repeat_times=beta)
            return balanced_batch, real_beta
        else:
            system1_data = self.ensure_dataproto(system1_data) # self.ensure_dataproto(system1_data).non_tensor_batch['task']
            system2_data = self.ensure_dataproto(system2_data)

            n = len(system2_data) # system2_data.batch.batch_size[0] # = batch size * rollout.n
            target_system1_samples = (beta - 1) * n  # target_num of system1
            current_system1_samples = len(system1_data) # system1_data.batch.batch_size[0]

            if current_system1_samples == target_system1_samples:
                return batch, real_beta
            
            # 3. 根据数量关系决定采样策略
            threshold = (beta - 1) * n / 2
            
            if current_system1_samples < target_system1_samples and current_system1_samples < threshold:
                # system1 样本太少，同时对两个系统进行上采样
                both_indices = self._sample_indices(len(batch), beta * n)
                balanced_batch = batch[both_indices] # DataProtoItem

                return balanced_batch, real_beta
            else:
                # 包含了对system1进行下采样或者重采样的所有情况  # system1_data.non_tensor_batch['task']
                system1_indices = self._sample_indices(system1_data.batch.batch_size[0], target_system1_samples)
                system1_data = system1_data[system1_indices]

            # 3. 合并处理后的数据
            balanced_batch = DataProto.concat([self.ensure_dataproto(system1_data), self.ensure_dataproto(system2_data)])
            
            return balanced_batch, real_beta

    def _sample_indices(self, current_size: int, target_size: int) -> torch.Tensor:
        """生成采样后的索引"""
        if target_size <= current_size:
            # 如果需要下采样或保持原样，随机选择 target_size 个索引
            return torch.randperm(current_size)[:target_size]
        
        # 计算需要重复的次数
        full_repeats = target_size // current_size
        remainder = target_size % current_size
        
        # 生成索引
        indices = []
        # 完整重复的部分
        for _ in range(full_repeats):
            indices.append(torch.randperm(current_size))
        # 处理余数
        if remainder > 0:
            indices.append(torch.randperm(current_size)[:remainder])
        
        return torch.cat(indices)

    def fit(self):
        """
        The training loop of PPO.
        The driver process only need to call the compute functions of the worker group through RPC to construct the PPO dataflow.
        The light-weight advantage computation is done on the driver process.
        """
        from verl.utils.tracking import Tracking
        from omegaconf import OmegaConf

        logger = Tracking(project_name=self.config.trainer.project_name,
                          experiment_name=self.config.trainer.experiment_name,
                          default_backend=self.config.trainer.logger,
                          config=OmegaConf.to_container(self.config, resolve=True))

        self.global_steps = 0

        # load checkpoint before doing anything
        self._load_checkpoint()
        
        # record current best performance
        monitor_key = self.config.actor_rollout_ref.rollout.monitor_key
        if monitor_key is not None:
            best_performance = -1

        # perform validation before training
        # currently, we only support validation using the reward_function.
        if self.val_reward_fn is not None and self.config.trainer.get('val_before_train', True):
            val_metrics = self._validate() # {'val/test_score/hle': 0.0}
            pprint(f'Initial validation metrics: {val_metrics}')
            logger.log(data=val_metrics, step=self.global_steps)
            if self.config.trainer.get('val_only', False):
                with open(os.path.join(self.config.trainer.default_local_dir, 'val_metrics.json'), 'w') as f:
                    json.dump(val_metrics, f, indent=2)
                return
            if monitor_key is not None:
                best_performance = val_metrics[monitor_key]

        # we start from step 1
        self.global_steps += 1

        batches_consumed_of_last_ckpt_in_current_epoch = getattr(self.train_dataloader, 'batches_consumed', 0)
        epochs_consumed = getattr(self.train_dataloader, 'epochs_consumed', 0)

        for epoch in range(epochs_consumed, self.config.trainer.total_epochs):
            batches_consumed = 0

            # We manually set the seed for the sampler to ensure the same order of data in each epoch
            if getattr(self.train_dataloader.sampler, "generator", None) is not None:
                self.train_dataloader.sampler.generator.manual_seed(epoch)

            for batch_dict in self.train_dataloader:
                # if we resume from a checkpoint, we need to skip the batches that have been consumed
                if batches_consumed < batches_consumed_of_last_ckpt_in_current_epoch:
                    batches_consumed += 1
                    continue
                
                # Reset the flag for the next epoch to avoid skipping batches again
                batches_consumed_of_last_ckpt_in_current_epoch = 0

                print('=' * 20, f'Epoch {epoch}, consumed {batches_consumed} batches, Step {self.global_steps}', '=' * 20)
                metrics = {}
                timing_raw = {}
                
                batch: DataProto = DataProto.from_single_dict(batch_dict)

                with _timer('step', timing_raw):
                    batch.non_tensor_batch['uid'] = np.array([str(uuid.uuid4()) for _ in range(len(batch.batch))],
                                                            dtype=object)
                    # repeat to align with repeated responses in rollout
                    batch = batch.repeat(repeat_times=self.config.actor_rollout_ref.rollout.n, interleave=False)
                    # import pdb; pdb.set_trace() # batch.non_tensor_batch['task']
                    # pop those keys for generation
                    gen_batch = batch.pop(
                        batch_keys=['input_ids', 'attention_mask', 'position_ids'],
                        non_tensor_batch_keys=['rollout_item', 'task', 'uid', 'job'], #  dict_keys(['data_source', 'answer', 'ability', 'task', 'reward_model', 'extra_info', 'messages', 'job', 'rollout_item', 'prompt_with_chat_template', 'index', 'uid'])
                    ) # batch.non_tensor_batch['job']

                    # generate a batch
                    print('start generation')
                    with _timer('gen', timing_raw):
                        # gen_batch: (batch_size * n, prompt_length)
                        # gen_batch_output: (batch_size * n, prompt_length + response_length)
                        gen_batch_output = self.actor_rollout_wg.generate_sequences(gen_batch)

                        # if "PythonInterpreter" in self.config.actor_rollout_ref.rollout.function_list:
                        #     delete_cached_python_codes()
                    
                    if self.config.algorithm.adv_estimator == 'remax':
                        with _timer('gen_max', timing_raw):
                            gen_baseline_batch = deepcopy(gen_batch)
                            gen_baseline_batch.meta_info['do_sample'] = False
                            gen_baseline_output = self.actor_rollout_wg.generate_sequences(gen_baseline_batch)

                            batch = batch.union(gen_baseline_output)
                            reward_baseline_tensor, _ = self.reward_fn(batch)
                            reward_baseline_tensor = reward_baseline_tensor.sum(dim=-1)

                            batch.pop(batch_keys=list(gen_baseline_output.batch.keys()))

                            batch.batch['reward_baselines'] = reward_baseline_tensor

                            del gen_baseline_batch, gen_baseline_output
                    
                    if self.config.actor_rollout_ref.rollout.use_rl_agent:
                        if self.multi_agent_train:
                            gen_batch_output, _ = self.filter_data_by_keywords(gen_batch_output, ['dummy'], 'task')
                        
                        # 先统计工具的使用情况
                        tool_usage_all = {}
                        ques_num = 0
                        for tool_t, tool_usage in zip(gen_batch_output.non_tensor_batch['task'], gen_batch_output.non_tensor_batch['tool_usage'], strict=True):
                            if 'system1' not in tool_t and "dummy" not in tool_t:
                                ques_num += 1
                                for tool, count in tool_usage.items():
                                    tool_usage_all[tool] = tool_usage_all.get(tool, 0) + count

                        metrics[f"data/tool_usage/tool_all"] = np.sum(list(tool_usage_all.values()))
                        metrics[f"data/tool_usage/ques_num"] = ques_num
                        metrics[f"data/tool_usage/tool_per_ques"] = metrics[f"data/tool_usage/tool_all"] / ques_num if ques_num > 0 else 0
                        for tool, count in tool_usage_all.items():
                            metrics[f"data/tool_usage/{tool}"] = count
                            metrics[f"data/tool_usage/{tool}_ratio"] = count / metrics[f"data/tool_usage/tool_all"] if metrics[f"data/tool_usage/tool_all"] > 0 else 0
                        
                        # 考虑平衡system1&2的数量
                        if self.multi_agent_train:
                            if self.config.actor_rollout_ref.rollout.multi_agent_pattern.adv:
                                gen_batch_output, _ = pad_dataproto_to_divisor(gen_batch_output, self.actor_rollout_wg.world_size, change_task=self.multi_agent_train)
                                batch = gen_batch_output
                            else:
                                with _timer('bal_samps', timing_raw):
                                    batch, beta_ratio = self.balance_multi_agent_samples(gen_batch_output)
                                metrics[f"data/beta_ratio"] = beta_ratio
                        else:
                            batch = gen_batch_output

                    else:
                        batch = batch.union(gen_batch_output)

                    if not self.config.actor_rollout_ref.rollout.multi_agent_pattern.adv:
                        # balance the number of valid tokens on each dp rank.
                        # Note that this breaks the order of data inside the batch.
                        # Please take care when you implement group based adv computation such as GRPO and rloo
                        self._balance_batch(batch, metrics=metrics)

                        # compute global_valid tokens
                        batch.meta_info['global_token_num'] = torch.sum(batch.batch['attention_mask'], dim=-1).tolist()

                    # recompute old_log_probs
                    print('start compute log prob')
                    with _timer('old_log_prob', timing_raw):
                        old_log_prob = self.actor_rollout_wg.compute_log_prob(batch)
                        batch = batch.union(old_log_prob)

                    if self.use_reference_policy:
                        # compute reference log_prob
                        print('start compute ref log prob')
                        with _timer('ref', timing_raw):
                            ref_log_prob = self.ref_policy_wg.compute_ref_log_prob(batch)
                            batch = batch.union(ref_log_prob)

                    # compute values
                    if self.use_critic:
                        print('start compute values')
                        with _timer('values', timing_raw):
                            values = self.critic_wg.compute_values(batch)
                            batch = batch.union(values)
                    
                    with _timer('adv', timing_raw):
                        print('start compute adv')
                        # compute scores. Support both model and function-based.
                        # We first compute the scores using reward model. Then, we call reward_fn to combine
                        # the results from reward model and rule-based results.
                        if self.use_rm:
                            # TODO: 还不支持rm下的system1先计算adv，在平衡数据；需要将dummpy对应的reward删除（会影响登记的结果，比如solver all
                            # we first compute reward model score
                            reward_tensor = self.rm_wg.compute_rm_score(batch)
                            batch = batch.union(reward_tensor)

                        if self.config.actor_rollout_ref.rollout.multi_agent_pattern.adv and self.multi_agent_train: # remote dummy data
                            batch, _ = self.filter_data_by_keywords(batch, ['dummy'], 'task')

                        # we combine with rule-based rm
                        reward_tensor, task_list = self.reward_fn(batch)
                        batch.batch['token_level_scores'] = reward_tensor

                        #log to metrics
                        task_d = defaultdict(list)
                        for i in range(len(task_list)):
                            if "dummy" in task_list[i]:
                                continue
                            task_d[task_list[i].split('/')[-1]].append(reward_tensor[i].sum().item())

                        for key in task_d:
                            metrics[f"critic/{key}_score/mean"] = np.mean(task_d[key])
                            metrics[f"critic/{key}_score/max"] = np.max(task_d[key])
                            metrics[f"critic/{key}_score/min"] = np.min(task_d[key])

                        # Group rewards by uid
                        uids = batch.non_tensor_batch['uid']
                        unique_uids = np.unique(uids)
                        valid_mask = torch.ones(len(uids), dtype=torch.bool)
                        solve_none = 0
                        solve_all = 0
                        for uid in unique_uids:
                            uid_mask = uids == uid
                            uid_rewards = reward_tensor[uid_mask].sum(-1)  # Sum rewards for each sequence
                            
                            # Check if all rewards are 0 or all are 1 for this uid
                            if (uid_rewards == 0).all():
                                valid_mask[uid_mask] = False
                                solve_none += 1
                            elif (uid_rewards == 1).all():
                                valid_mask[uid_mask] = False
                                solve_all += 1
                        
                        # Log to metrics
                        metrics['batch/solve_none'] = solve_none
                        metrics['batch/solve_all'] = solve_all


                        # compute rewards. apply_kl_penalty if available
                        if not self.config.actor_rollout_ref.actor.get('use_kl_loss', False):
                            batch, kl_metrics = apply_kl_penalty(batch,
                                                                 kl_ctrl=self.kl_ctrl,
                                                                 kl_penalty=self.config.algorithm.kl_penalty)
                            metrics.update(kl_metrics)
                        else:
                            batch.batch['token_level_rewards'] = batch.batch['token_level_scores']

                        # compute advantages, executed on the driver process
                        batch = compute_advantage(batch,
                                                  adv_estimator=self.config.algorithm.adv_estimator,
                                                  gamma=self.config.algorithm.gamma,
                                                  lam=self.config.algorithm.lam,
                                                  num_repeat=self.config.actor_rollout_ref.rollout.n)

                    if self.config.actor_rollout_ref.rollout.multi_agent_pattern.adv:
                        with _timer('bal_samps', timing_raw):
                            batch, beta_ratio = self.balance_multi_agent_samples(batch)
                        metrics[f"data/beta_ratio"] = beta_ratio

                        # balance the number of valid tokens on each dp rank.
                        # Note that this breaks the order of data inside the batch.
                        # Please take care when you implement group based adv computation such as GRPO and rloo
                        self._balance_batch(batch, metrics=metrics)

                        # compute global_valid tokens
                        batch.meta_info['global_token_num'] = torch.sum(batch.batch['attention_mask'], dim=-1).tolist()

                    # save querys sampled each step 
                    self.train_query_sample_list_pool = self._reformat_sample_list(batch) 
                    # batch.non_tensor_batch['decoded_responses'] = np.array([str(sample['response']) for sample in self.train_query_sample_list_pool], dtype=object)
                    self._save_query_samples(split="train") 
                    
                    # update critic
                    if self.use_critic:
                        with _timer('update_critic', timing_raw):
                            # pdb.set_trace()
                            critic_output = self.critic_wg.update_critic(batch)
                        critic_output_metrics = reduce_metrics(critic_output.meta_info['metrics'])
                        metrics.update(critic_output_metrics)

                    # implement critic warmup
                    if self.config.trainer.critic_warmup <= self.global_steps:
                        # update actor
                        with _timer('update_actor', timing_raw):
                            # pdb.set_trace()
                            actor_output = self.actor_rollout_wg.update_actor(batch)
                        actor_output_metrics = reduce_metrics(actor_output.meta_info['metrics'])
                        metrics.update(actor_output_metrics)

                    # validate
                    if monitor_key is not None:
                        cur_performance = -2
                        
                    if self.val_reward_fn is not None and self.config.trainer.test_freq > 0 and \
                        self.global_steps % self.config.trainer.test_freq == 0:
                        with _timer('testing', timing_raw):
                            val_metrics: dict = self._validate() # {'val/test_score/hle': 0.0}
                        metrics.update(val_metrics)

                        if monitor_key is not None:
                            cur_performance = val_metrics[monitor_key]

                    if self.config.trainer.save_freq > 0 and \
                            self.global_steps % self.config.trainer.save_freq == 0:
                        if monitor_key is not None:
                            if cur_performance > best_performance:
                                best_performance = cur_performance
                                with _timer('save_checkpoint', timing_raw):
                                    self._save_checkpoint(epoch=epoch, batches_consumed=batches_consumed+1)
                        else:
                            with _timer('save_checkpoint', timing_raw):
                                self._save_checkpoint(epoch=epoch, batches_consumed=batches_consumed+1)

                # collect metrics
                metrics.update(compute_data_metrics(batch=batch, use_critic=self.use_critic))
                metrics.update(compute_timing_metrics(batch=batch, timing_raw=timing_raw))

                # TODO: make a canonical logger that supports various backend
                logger.log(data=metrics, step=self.global_steps)

                self.global_steps += 1
                batches_consumed += 1

                if self.global_steps >= self.total_training_steps:

                    # perform validation after training
                    if self.val_reward_fn is not None:
                        val_metrics = self._validate()
                        pprint(f'Final validation metrics: {val_metrics}')
                        logger.log(data=val_metrics, step=self.global_steps)
                    if self.config.trainer.save_freq > 0 and \
                            (self.global_steps - 1) % self.config.trainer.save_freq != 0:
                        with _timer('save_checkpoint', timing_raw):
                            self._save_checkpoint(epoch=epoch, batches_consumed=batches_consumed+1)
                    return
